/*
 * main.c
 *
 * Created: 2012-03-08 01:29:22
 *  Author: marek
 */ 

#include <avr/pgmspace.h>
#include <avr/eeprom.h>

#include "processor.h"
#include "setup.h"
#include "main.h"
#include "protocolWago.h"
#include "timers.h"
#include "uart.h"
#include "flash.h"
#include "led.h"
#include "crc.h"
#include "adc.h"
#include "dipswitch.h"


// FUSES ={
// 	{
// 		0x90,					// FUSEBYTE0:
// 		0xFF,					// FUSEBYTE1:
// 		0xFF,					// FUSEBYTE5:
// 	}
// };


const _deviceInfo PROGMEM DeviceInfo = 
{
	"BINFO",
	__DEV_STRUCT_VER__,
	MAKE_VERSION(__BOOT_VER_MAIN__, __BOOT_VER_REV__),
};

_bootData bootData;


/*!
	@brief Metoda sprawdza integralnosc pamieci FLASH w zakresie sekcji aplikacji
	@details Ostatnie dwa bajty sekcji aplikacji poprawnego programu powinny zawierac sume CRC Modbus obliczonych z pozostalej pamieci (bez bootloadera).
		Sprawdzenie integralnosci polega na weryfikacji poprawnosci ej sumy kontrolnej.
		Suma kontrolna jest obliczana przez program dzialajacy w sekcji aplikacji na zadanie bootloadera.
		Bootloader wysyla zadanie do programu zwyklago po zakonczeniu jego wgrywania i po odebraniu polecenia przejscia do sekcji zwyklego programu.
		Polecenie przekazywane jest w dwoch ostatnich bajtach pamieci SRAM.	
		
		Dodatkowo metoda wyszukuje znacznika z wersja programu.
	@return
		0	: pamiec FLASH (sekcja aplikacji) jest spojna: wykryto pastylke informacyjna oraz zgodnosc sumy CRC tej pamieci
		-1	: wykryto pastylke informacyjna, brak zgosnosci sumy CRC
		-2	: brak pastylki informacyjnej
*/
uint8_t checkAppSection()
{
	//sprawdzenie czy zawartosc pamieci flash (app section) zawiera program i rownoczesne wyciagniecie wersji programu (jesli jest):
	/*!
		Zmienna tworzaca maszyne stanu, gdzie odpowiednie stany to:
			0 - nic nie znaleziono 
			1 - znaleziono pierwszy znak znacznika
			...-...
			5 - znaleziono piaty znak znacznika
			6 - odczytano wersje struktury danych
			7 - odczytano mlodszy bajt wersji programu
			8 - odczytano starszy bajt wersji programu
			255 - koniec (znaleziono pastylke informacyjna)
	*/
	bootData.softVersion = 0xFFFF;
	uint8_t flashWalkerState = 0;
	uint16_t flashCrc = 0xFFFF;
	uint16_t softVersionTmp = 0xFFFF;
	uint16_t uartSpeedEeAddTmp = 0xFFFF;
	uint16_t modbusAddEeAddTmp = 0xFFFF;
	uint8_t flashData;
	uint16_t tmp = 0;
	for (uint32_t i = 0; i < BOOT_SECTION_START;i++)
	{
		flashData = pgm_read_byte_far(i);
		crc16ModbusNext(flashData, &flashCrc);
		if(flashWalkerState != 255)
		{
			if(flashWalkerState == 0 && flashData == 'A')
				flashWalkerState ++;
			else if(flashWalkerState == 1 && flashData == 'I')
				flashWalkerState ++;
			else if(flashWalkerState == 2 && flashData == 'N')
				flashWalkerState ++;
			else if(flashWalkerState == 3 && flashData == 'F')
				flashWalkerState ++;
			else if(flashWalkerState == 4 && flashData == 'O')
				flashWalkerState ++;
			else if(flashWalkerState == 5)
			{
				//odczyt wersji struktury pastylki informacyjnej
				if(flashData == 1)
					flashWalkerState ++;
				else
				{
					flashWalkerState = 255;
				}					
			}			
			else if(flashWalkerState == 6)
			{
				//odczyt wersji programu (LSB)
				tmp = flashData;
				flashWalkerState ++;
			}				
			else if(flashWalkerState == 7)
			{
				//odczyt wersji programu (MSB)
				tmp |= (((uint16_t)flashData) << 8);
				softVersionTmp = tmp;
				flashWalkerState ++;
			}			
			else if(flashWalkerState == 8)
			{
				//odczyt adresu eeprom dla predkosci uart (LSB)
				tmp = flashData;
				flashWalkerState ++;
			}
			else if(flashWalkerState == 9)
			{
				//odczyt adresu eeprom dla predkosci uart (MSB)
				tmp |= (((uint16_t)flashData) << 8);
				uartSpeedEeAddTmp = tmp;
				flashWalkerState ++;
			}			
			else if(flashWalkerState == 10)
			{
				//odczyt adresu eeprom dla adresu modbus (LSB)
				tmp = flashData;
				flashWalkerState ++;
			}
			else if(flashWalkerState == 11)
			{
				//odczyt adresu eeprom dla adresu modbus (MSB)
				tmp |= (((uint16_t)flashData) << 8);
				modbusAddEeAddTmp = tmp;
				flashWalkerState = 255;
			}
			else
			{
				//powrot do stanu poczatkowego
				flashWalkerState = 0;
			}
		}			
	}	
	if(flashWalkerState==255)
	{
		//wykryto pastylke informacyjna:
		if(flashCrc == 0)
		{
			//zgodnosc sumy CRC:
			bootData.softVersion = softVersionTmp;
			bootData.eepromSaveUartSpeed = eeprom_read_word((uint16_t*)uartSpeedEeAddTmp);
			bootData.eepromSaveModbusAdd = eeprom_read_word((uint16_t*)modbusAddEeAddTmp);
			return 0;
		}
		return -1;
		
	}
	return -2;
}

/*!
	@brief Sprawdzenie spojnosci danych (lacznie z wyciagnieciem wersji programu) i odczytanie ewentualnego polecenia od zwyklej aplikacji 
*/
void scanAppCmd(void)
{
	_command appCmd = *((uint16_t*)(RAMEND-1));
	*((uint16_t*)(RAMEND-1)) = CMD_EMPTY;
	bootData.eepromSaveUartSpeed = DEFAULT_BAUD_RATE;
	bootData.eepromSaveModbusAdd = MODBUS_ADDRESS_SERVICE;
	bootData.appDetect = checkAppSection();	
	if(bootData.appDetect == 0)
	{
		//wykryto spojnosc danych w sekcji aplikacji
		if((appCmd & CMD_MASK) == CMD_STAY_IN_BOOT)
		{
			bootData.dipswitchValue = (uint8_t)(appCmd & DATA_MASK);
			setWorkState(MODE_ACT);
		}
		else
 		{
			 *((uint16_t*)(RAMEND-1)) = CMD_APP_NORMAL_RUN | bootData.dipswitchValue;
			jumpToApplication();
		}
	}
	else
	{
		setWorkState(MODE_ACT_ERR);
	}
	
	bootData.appLabel = pgm_read_byte(&DeviceInfo.label[0]);
}

//*********************************************
int main(void)
{
	_workState lastState;
	_workState actualState;
	uint8_t actualizationBrakeFlag;
	uint8_t actualizationFinishedFlag;
	
	initHardwareEarly();
	adcEnable();
	dipSwitchInit();
	initData();
#ifndef _DEBUG
	scanAppCmd();		//Sprawdzenie spojnosci danych (lacznie z wyciagnieciem wersji programu) i odczytanie ewentualnego polecenia od zwyklej aplikacji	
#endif	//_DEBUG
	initTimers();	
	initUart();
	
	initHardwareLate();	
	lastState = getWorkState();	
	
	while(1)
	{
		hardwareService();
		uartService();
		timersService();
		ledService();
				
		actualState = getWorkState();		
		actualizationBrakeFlag = (lastState == MODE_ACT && actualState == MODE_ACT_ERR);
		actualizationFinishedFlag = actualState == MODE_ACT_END && !uart0.isTransmit();
		if(actualizationBrakeFlag || actualizationFinishedFlag)
		{
			//sytuacja ma miejsce kiedy rozpoczeto aktualizacje i zostala ona samoczynnie przerwana
			//nalezy sprawdzic czy flash jest spojny, jesli tak to nalezy do niego przeskoczyc:
			bootData.appDetect = checkAppSection();
				
			if(bootData.appDetect == 0)
			{
				//wykryto spojnosc danych w sekcji aplikacji, przeskok bez analizy komunikatow
				*((uint16_t*)(RAMEND-1)) = CMD_APP_NORMAL_RUN | bootData.dipswitchValue;
				jumpToApplication();
			}
			else
			{
				setWorkState(MODE_ACT_ERR);
			}
		}
		lastState = actualState;
    }
}



void setWorkState(_workState newWorkState)
{
	if(newWorkState == MODE_ACT)
	{			
		setTimer1Sec(TMR_1SEC_ACTUALIZATION_ERROR, MAIN_ACTUALIZATION_NEXT_MESSAGE_TIME);
	}
	else if(newWorkState == MODE_ACT_END)
	{
		setTimer1Sec(TMR_1SEC_ACTUALIZATION_ERROR, MAIN_ACTUALIZATION_NEXT_MESSAGE_TIME);
	}
	else
	{
		setTimer1Sec(TMR_1SEC_ACTUALIZATION_ERROR, 0);
	}
	bootData.deviceWorkState = newWorkState;
}

_workState getWorkState(void)
{
	return bootData.deviceWorkState;
}	