#ifndef MAIN_H_
#define MAIN_H_

#include <stdint.h>
#include "led.h"
#include "uart.h"

#define MAKE_VERSION(MSB, LSB)		(MSB)<<8 | (LSB)

#define __DEV_STRUCT_VER__	1
#define __BOOT_VER_MAIN__	2
#define __BOOT_VER_REV__	4


/*! Czas oczekiwania na otrzymanie kolejnej ramki podczas aktualizacji
	(jesli po tym czasie nie otrzyma sie kolejnej ramki to aktualizacja jest przerywana)
*/
#define MAIN_ACTUALIZATION_NEXT_MESSAGE_TIME	60

/*!
	@enum _workState
	@brief Zbir wartoci oznaczajcych tryb pracy bootloadera.
	@details Zdefiniowane tu wartoci mona zapisywa poprzez Modbus do rejestru o adresie 13 (funkcja 6)
*/
typedef enum
{
	MODE_NONE			= 0xFFFF,		/*!< Stan nieustalony */
	MODE_ACT_ERR		= 0xF00F,		/*!< Normalny stan bootloadera/blad aktualizacji (szybkie mruganie diody) */
	MODE_ACT			= 0x00FF,		/*!< Aktualizacja oprogramowania (powolne mruganie diody) */
	MODE_ACT_END		= 0xFF00,		/*!< Wyjcie z trybu bootloadera */
}_workState;

/*!
	@enum _command
	@brief Zbir komend jakie moga byc przekazywane pomiedzy programem zwykym a bootloaderem
*/
typedef enum
{
	CMD_EMPTY				= 0xFFFF,		/*!< Komenda pusta (brak komendy) */
	CMD_MASK				= 0xFF00,		/*!< Maska bitowa komendy */
	DATA_MASK				= 0x00FF,		/*!< Maska bitowa danych */
	//polecenia od normalnego programu do bootloadera:
	CMD_STAY_IN_BOOT		= 0x5100,		/*!< Niech bootloader po sprawdzeniu spjnoci pamici flash pozostanie w bootloaderze */
	//polecenia od bootloadera do normalnego programu:
	CMD_APP_NORMAL_RUN		= 0x5400,
}_command;

/*!
	@struct _deviceInfo
	@brief Struktura zawierajace metryke programu.
*/
typedef struct
{
	uint8_t label[5];
	uint8_t structVersion;
	uint16_t softVersion;
	uint16_t uartSpeedEeAdd;		/*!< adres w pamieci eeprom, gdzie zapisana jest predkosc z jaka odbywa sie komunikacja */
	uint16_t modbusAddEeAdd;		/*!< adres w pamieci eeprom, gdzie zapisany jest programowo ustawiany adres modbus urzadzenia */
}_deviceInfo;

/*!
	@struct _bootData
	@brief Struktura zawierajace gwne dane wykorzystywane przez bootloader.
*/
typedef struct
{
	_workState deviceWorkState;			/*!< Aktualny tryb pracy urzdzenia */
	uint16_t actualizationAddress;		/*!< Adres slowa, na ktory trzeba zaadresowac kolejna porcje danych podczas aktualizacji*/
	
	uint8_t appDetect;					/*!< 0 - nie wykryto programu glownego; 1 - program glowny wykryty (zgodnosc CRC); 2 - program glowny wykryty (wykryto tylko pastylke informacyjna) */
	uint8_t appLabel;					/*!< zmienna wymuszajaca umieszczenie znacznika aplikacji w pamieci */
	uint16_t softVersion;				/*!< wersja programu dzialajacego w ApplicationSection */	
	_baudRate uartSpeed;				/*!< predkosc z jaka odbywa sie komunikacja */
	uint16_t modbusAdd;					/*!< aktualny programowo ustawiany adres modbus urzadzenia */
	uint16_t eepromSaveUartSpeed;		/*!< predkosc z jaka odbywa sie komunikacja odczytana z eepromu */
	uint16_t eepromSaveModbusAdd;		/*!< programowo ustawiany adres modbus urzadzenia odczytany z eepromu */
	uint8_t dipswitchValue;				/*!< adres wybrany na dipswitchu */
	
	uint16_t actualizationCrc;			/*!< CRC obliczone z aktualizowanych danych */
	uint8_t actualizationCrcCorrect;	/*!< flaga wskazujaca czy CRC nowego porogramu jest prawidlowe */
}_bootData;
extern _bootData bootData;

/*!
	@brief Metoda sluzy do ustawiania nowego trybu pracy
	@param newWorkState Nowy tryb pracy
*/
void setWorkState(_workState newWorkState);

/*!
	@brief Pobranie aktualnego trybu pracy
	@return Aktualny tryb pracy
*/
_workState getWorkState(void);


#endif //MAIN_H_
