/*!
 *  @file processor.c
 */ 
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/wdt.h>

#include "processor.h"
#include "led.h"


void delayedUart0Send(_baudRate baudRate)
{
	uint8_t tmp = 0;
	TCNT0 = 0;
	switch(baudRate)
	{
		case BAUDRATE_1200:
		tmp = 255;					// < 3.5znaku!
		break;
		case BAUDRATE_2400:
		tmp = 159;
		break;
		case BAUDRATE_4800:
		tmp = 80;
		break;
		case BAUDRATE_9600:
		tmp = 40;
		break;
		case BAUDRATE_19200:
		tmp = 20;
		break;
		case BAUDRATE_38400:
		tmp = 10;
		break;
		case BAUDRATE_57600:
		tmp = 7;
		break;
		case BAUDRATE_115200:
		tmp = 4;
		break;
	}
	OCR0A = tmp;
	TCCR0B = (1<<CS02)|(1<<CS00);		//1:1024
}

void preInit(void)
{
	cli();
    MCUSR = 0x00;	//wykasowanie flag zrodla resetu
    wdt_disable();
}

/*!
	@brief Przeniesienie wektora przerwan do sekcji bootloadera
*/
void moveInterruptToBoot(void)
{
	unsigned char tmp;
	unsigned char changeEnableCmd;
	unsigned char moveIntToBootCmd;
	cli();
	tmp = MCUCR;						/* GET MCUCR*/
	changeEnableCmd = tmp | (1<<IVCE);
	moveIntToBootCmd = tmp | (1<<IVSEL);
	MCUCR = changeEnableCmd;			/* Enable change of Interrupt Vectors  */
	MCUCR = moveIntToBootCmd;			/* Move interrupts to Boot Flash section */	
}

/*!
	@brief Przeniesienie wektora przerwan do sekcji aplikacji
*/
void moveInterruptToApp(void)
{
	unsigned char tmp;
	unsigned char changeEnableCmd;
	unsigned char moveIntToAppCmd;
	cli();
	tmp = MCUCR;						/* GET MCUCR*/
	changeEnableCmd = tmp | (1<<IVCE);
	moveIntToAppCmd = tmp & ~(1<<IVSEL);
	MCUCR = changeEnableCmd;			/* Enable change of Interrupt Vectors  */
	MCUCR = moveIntToAppCmd;			/* Move interrupts to Boot Flash section */		
}

void initHardwareEarly(void)
{
	moveInterruptToBoot();
	
	//porty:
	//-----------------------------------------
	//uart0:
	UART0_TXD_DIR_REG &= ~(1<<UART0_TXD_LINE);		//input
	UART0_TXD_PORT_REG	|= ~(1<<UART0_TXD_LINE);	//pulled high
	UART0_RXD_DIR_REG &= ~(1<<UART0_RXD_LINE);		//input
	UART0_RXD_PORT_REG	|= ~(1<<UART0_RXD_LINE);	//pulled high
	UART0_RE_DIR_REG |= _BV(UART0_RE_LINE);
	UART0_DE_DIR_REG |= _BV(UART0_DE_LINE);
	//LED:
	LED_DIR_REG |= _BV(LED_LINE);
	LED_OFF();
	//dipswitch:
	DIP_L_DIR_REG &= ~(1<<DIP_L_LINE);				//input
	DIP_H_DIR_REG &= ~(1<<DIP_H_LINE);				//input
	
	//timery:
	//-----------------------------------------
	//Timer0 - opoznienie rozpoczecia wysylania odpowiedzi po interfejsie uart0:
	TIMSK0 = (1<<OCIE0A);
	TCCR0A = (1<<WGM01);
	TCCR0B = 0;				//timer stop
	//-----------------------------------------
	//Timer2 - odmierzanie standardowego interwalu (TIMEOUT_UNIT[ms]):
	OCR2A = TIMER_VALUE;
	TIMSK2 = (1<<OCIE2A);
	TCCR2A = (1<<WGM21);	
	TCCR2B = (1<<CS22)|(1<<CS21)|(1<<CS20);
}

void initHardwareLate(void)
{
	sei();
#ifndef _DEBUG
	wdt_enable(WDTO_4S);
#endif	//_DEBUG	
}

void hardwareTurnOff(void)
{
	cli();
	wdt_disable();
	TCCR2B = 0;
}

inline void hardwareService(void){
	wdt_reset();
}

void jumpToApplication(void)
{
	hardwareTurnOff();
	uartTurnOff();
	moveInterruptToApp();
	((void (*)(void))0)();
}