#include <avr/io.h>
#include <stdint.h>
#include <string.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>

#include "protocolWago.h"
#include "crc.h"
#include "uart.h"
#include "led.h"
#include "main.h"
#include "processor.h"
#include "flash.h"


_uartData * uartWAGO;


uint16_t parseFrameWago(void)
{
	uint16_t bytesToSend = 0;
	
	uint8_t receiverAddress = uartWAGO->bufferRx[0];
	uint8_t functionCode = uartWAGO->bufferRx[1];
	uint16_t beginAddress = uartWAGO->bufferRx[2]<<8 | uartWAGO->bufferRx[3];
	uint16_t regCount = uartWAGO->bufferRx[4]<<8 | uartWAGO->bufferRx[5];	
	uint8_t resultCode = MODBUS_EXCEPTION_NONE;
	
	if(receiverAddress == 0 || receiverAddress == bootData.modbusAdd)	
	{
	
	
		uartWAGO->bufferTx[0] = uartWAGO->bufferRx[0];
		uartWAGO->bufferTx[1] = uartWAGO->bufferRx[1];
	
		if(functionCode == 0x03)
		{
			//funkcja 3 pozwala wylacznie na odczytanie stanu urzadzenia z rejestru o adresie 0x0D
			setWorkState(MODE_ACT_ERR);				//Wywolanie tej funkcji powoduje przerwanie aktualizacji
			if(regCount!=1)
			{
				resultCode = MODBUS_EXCEPTION_QUANTITY;
			}
			else if(beginAddress != MODBUS_ACT_SOFT_STATE_ADDRESS )
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
			else
			{
				if(receiverAddress != 0)
				{				
					uint16_t workState = (uint16_t)getWorkState();
					uartWAGO->bufferTx[2] = (uint8_t)(2 * regCount);	//ilosc bajtow wynikowych
					uartWAGO->bufferTx[3] = (workState >> 8) & 0xFF;
					uartWAGO->bufferTx[4] = workState & 0xFF;
					bytesToSend = 5;
					uint16_t crc2 = CRC_MODBUS_INIT_VALUE;
					crc16ModbusCompute(&crc2, uartWAGO->bufferTx, bytesToSend);
 					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
 					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);	
				}
			}
		}
		if(functionCode == 0x04)
		{
			if(beginAddress >= MODBUS_SYSTEM_BOOT_VER && beginAddress <= MODBUS_SYSTEM_APP_VER)
			{
				if(beginAddress + regCount - 1 > MODBUS_SYSTEM_APP_VER )
				{
					resultCode = MODBUS_EXCEPTION_QUANTITY;
				}
				else
				{
					bytesToSend = regCount*2;
					uartWAGO->bufferTx[2] = (uint8_t)bytesToSend;			//ilosc bajtow wynikowych
					uint8_t * bufferPnt = uartWAGO->bufferTx + 3;
					uint16_t inputRegData;
					for(uint8_t i=0; i<regCount; i++)
					{
						if(beginAddress+i == MODBUS_SYSTEM_BOOT_VER)
							inputRegData = MAKE_VERSION(__BOOT_VER_MAIN__, __BOOT_VER_REV__);
						else
							inputRegData = bootData.softVersion;
						*bufferPnt = (inputRegData >> 8);
						*(bufferPnt+1) = (inputRegData & 0xFF);
						bufferPnt+=2;
					}
					bytesToSend = (uint16_t)bufferPnt - (uint16_t)uartWAGO->bufferTx;
					uint16_t crc2 = CRC_MODBUS_INIT_VALUE;
					crc16ModbusCompute(&crc2, uartWAGO->bufferTx, bytesToSend);
					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
				}
			}
			else
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
		}
		else if(functionCode == 0x06)
		{
			uint16_t value = regCount;
		
			if(beginAddress == MODBUS_ACT_SOFT_STATE_ADDRESS)
			{
				if(value == MODE_ACT)
				{
					flashActualizationBegin();
					setWorkState(MODE_ACT);					
					bootData.actualizationAddress = 0;
					bootData.actualizationCrc = CRC_MODBUS_INIT_VALUE;
					bootData.actualizationCrcCorrect = 0;
				}
				else if(value == MODE_ACT_END)
				{						
					if(getWorkState() == MODE_ACT && bootData.actualizationCrcCorrect)
					{
						setWorkState(MODE_ACT_END);
						flashSaveAllProgCrc();
						rwwEnable();
						//po zakonczeniu odsylania odpowiedzi nastepuje przejsc do zwyklego programu
					}
					else
					{
						resultCode = MODBUS_EXCEPTION_WRITE;
					}
				}
				else
				{
					resultCode = MODBUS_EXCEPTION_QUANTITY;
				}
			}
			else if(beginAddress == MODBUS_VERIFICATION_DATA_ADDRESS)
			{
				EFlashFinishResult finishResult;				
				setWorkState(MODE_ACT);
				finishResult = flashActualizationFinished();
				if(finishResult == EFlashFinishResult_SaveToFlash)
				{
					rwwEnable();
					flashComputePageCrc(&bootData.actualizationCrc, bootData.actualizationAddress * 2 - SPM_PAGESIZE/2, SPM_PAGESIZE/2);
				}
				
				if(bootData.actualizationCrc == value)
				{
					bootData.actualizationCrcCorrect = 1;
				}
				else
				{
					resultCode = MODBUS_EXCEPTION_WRITE;
				}
			}
			else 
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
			
			if(resultCode == MODBUS_EXCEPTION_NONE && receiverAddress != 0)
			{
				uartWAGO->bufferTx[2] = (uint8_t)(2 * regCount);	//ilosc bajtow wynikowych				
				bytesToSend = 6;
				memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, bytesToSend);
				uint16_t crc2 = CRC_MODBUS_INIT_VALUE;
				crc16ModbusCompute(&crc2,uartWAGO->bufferTx, bytesToSend);
				uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
				uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
			}
			
		}
		else if(functionCode == 0x10)
		{				
			_workState actualState = getWorkState();
			if(actualState != MODE_ACT)
			{
				resultCode = MODBUS_EXCEPTION_WRITE;
			}
			else if(beginAddress > bootData.actualizationAddress || beginAddress % (SPM_PAGESIZE/4) != 0)
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
			else if(regCount != 64)
			{
				resultCode = MODBUS_EXCEPTION_QUANTITY;
			}
			else
			{
				EFlashWriteResult saveProgramResult;
				saveProgramResult = EFlashWriteResult_Error;
				if(beginAddress == bootData.actualizationAddress)
					saveProgramResult = flashSaveFromModbus (beginAddress*2, uartWAGO->bufferRx + 7);
					
				if(saveProgramResult != EFlashWriteResult_Error)
				{
					setWorkState(MODE_ACT);
					bootData.actualizationAddress += regCount;
					if(saveProgramResult == EFlashWriteResult_SaveToFlash)
					{
						rwwEnable();
						flashComputePageCrc(&bootData.actualizationCrc, bootData.actualizationAddress * 2 - SPM_PAGESIZE, SPM_PAGESIZE);
					}

					if(receiverAddress != 0)
					{
						bytesToSend = 6;
						memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, bytesToSend);				
						uint16_t crc2 = CRC_MODBUS_INIT_VALUE;
						crc16ModbusCompute(&crc2,uartWAGO->bufferTx, bytesToSend);
						uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
						uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
					}
				}				
				else
				{
					resultCode = MODBUS_EXCEPTION_WRITE;
				}
			
			}
		}

		//jesli zostal zgloszony blad przy przetwarzaniu polecenia, zwrocic kod bledu:
		if(resultCode != MODBUS_EXCEPTION_NONE && receiverAddress != 0)
		{	
			bytesToSend = 0;
			uartWAGO->bufferTx[bytesToSend++] = uartWAGO->bufferRx[0];
			uartWAGO->bufferTx[bytesToSend++] = uartWAGO->bufferRx[1] | 0x80;
			uartWAGO->bufferTx[bytesToSend++] = resultCode;
			uint16_t crc2 = CRC_MODBUS_INIT_VALUE;
			crc16ModbusCompute(&crc2, uartWAGO->bufferTx, bytesToSend);
 			uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
 			uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
		}
	}	
	return bytesToSend;
}