#include <avr/io.h>
#include <stdint.h>
#include <avr/eeprom.h>

#include "uart.h"
#include "timers.h"
#include "processor.h"
#include "protocolWago.h"
#include "crc.h"
#include "processor.h"
#include "main.h"
#include "dipswitch.h"

_uartData uart0;
extern _uartData * uartWAGO;
extern _bootData bootData;

/*!
	@brief 
		Funkcja przeksztaca predko transmisji na warto konfiguracyjna odpowiadajca
		tej predkoci.
	@param baudRate
		Prdko komunikacji (podzielona przez 10) dla jakiej naley uzyska warto
		konfiguracyjn
	@return
		Warto konfiguracyjna jak naley ustawi dla portu UART eby uzyska dan
		prdko komunikacji lub 0xFFFF w przypadku prdkoci niedozwolonej
*/
static uint16_t baudRateToUbrr(_baudRate baudRate)
{
	uint16_t ubrr = BAUDRATE_NONE_UBRR;
	
	if(baudRate == BAUDRATE_1200)
		ubrr = BAUDRATE_1200_UBRR;
	else if(baudRate == BAUDRATE_2400)
		ubrr = BAUDRATE_2400_UBRR;
	else if(baudRate == BAUDRATE_4800)
		ubrr = BAUDRATE_4800_UBRR;
	else if(baudRate == BAUDRATE_9600)
		ubrr = BAUDRATE_9600_UBRR;
	else if(baudRate == BAUDRATE_19200)
		ubrr = BAUDRATE_19200_UBRR;
	else if(baudRate == BAUDRATE_38400)
		ubrr = BAUDRATE_38400_UBRR;
	else if(baudRate == BAUDRATE_57600)
		ubrr = BAUDRATE_57600_UBRR;
	else if(baudRate == BAUDRATE_115200)
		ubrr = BAUDRATE_115200_UBRR;
	
	return ubrr;
}

void uartService(void)
{
	int16_t transmitLength;
	if(uart0.readyToParse)
	{
		uart0.stopReceiving();
		transmitLength = uart0.parseFrame();			
		if(transmitLength)
		{
			uart0.startTransmit(transmitLength);
		}					
		else
		{
			uart0.startReceiving();
		}
		uart0.readyToParse = 0;
	}
				
}


void uartClear(_uartData * data)
{
	data->counterRx = 0;
	data->counterTx = 0;
	data->currentId = 0;
}

void startReceivingUart0(void);
void stopReceivingUart0(void);
void startTransmitUart0(int16_t frameSize);
void setBaudRateUart0(_baudRate baudRate);
uint16_t parseFrameUart0(void);
uint8_t isTransmitUart0(void);
uint8_t isReceivingUart0(void);

void initUart(void)
{
	//uart0 (WAGO):
	//przypisanie odwolan do wlasciwych funkcji:
	uart0.startReceiving = startReceivingUart0;
	uart0.stopReceiving = stopReceivingUart0;
	uart0.startTransmit = startTransmitUart0;
	uart0.setBaudRate = setBaudRateUart0;
	uart0.parseFrame = parseFrameWago;
	uart0.isTransmit = isTransmitUart0;
	uart0.isReceiving = isReceivingUart0;
	//ustawienie formatu ramek: (8bit, 1stop)
	UCSR0C = (3<<UCSZ00);
	//ustawienie predkosci na portach oraz wlaczenie odbiornika i nadajnika:
	if(bootData.dipswitchValue == DIPSWITCH_SERVICE_MODE)
	{
		bootData.uartSpeed = DEFAULT_BAUD_RATE;
		bootData.modbusAdd = MODBUS_ADDRESS_SERVICE;
	}		
	else
	{
		bootData.uartSpeed = (_baudRate)bootData.eepromSaveUartSpeed;
		if(bootData.dipswitchValue == DIPSWITCH_SOFT_MODE)			
			bootData.modbusAdd = bootData.eepromSaveModbusAdd;
		else
			bootData.modbusAdd = bootData.dipswitchValue;		
	}
	uart0.setBaudRate(bootData.uartSpeed);
	//rozpoczecie pracy:
	uart0.startReceiving();
	uartWAGO = &uart0;
}

void uartTurnOff(void)
{
	UCSR0B = 0;
}

/**************************************************************************
uart0:
**************************************************************************/
void startReceivingUart0(void)
{
	UART0_RE_PORT_REG &= ~_BV(UART0_RE_LINE);
	UART0_DE_PORT_REG &= ~_BV(UART0_DE_LINE);
	UCSR0B &= ~((1<<TXCIE0) | (1<<UDRIE0));
	uartClear(&uart0);
	UCSR0B |= (1<<RXCIE0);
}
void stopReceivingUart0(void)
{
	UART0_RE_PORT_REG |= _BV(UART0_RE_LINE);
	UART0_DE_PORT_REG |= _BV(UART0_DE_LINE);
	UCSR0B &= ~(1<<RXCIE0);
}
void startTransmitUart0(int16_t frameSize)
{
	UART0_RE_PORT_REG |= _BV(UART0_RE_LINE);
	UART0_DE_PORT_REG |= _BV(UART0_DE_LINE);
	UCSR0B &= ~(1<<RXCIE0);
	uartClear(&uart0);
	uart0.counterTx = frameSize;
	//UCSR0B |= (1<<TXCIE0) | (1<<UDRIE0);
	delayedUart0Send(uart0.baudRate);
}
void setBaudRateUart0(_baudRate baudRate)
{
	uint16_t newConfig;
	newConfig = baudRateToUbrr(baudRate);
	if(newConfig == BAUDRATE_NONE_UBRR)
	{
		baudRate = DEFAULT_BAUD_RATE;
		newConfig = baudRateToUbrr(DEFAULT_BAUD_RATE);
	}		
		
	uart0.baudRate = baudRate;
	UCSR0B &= ~((1<<RXEN0) | (1<<TXEN0));
	uartClear(&uart0);
	UBRR0 = newConfig;
	UCSR0B = (1<<RXEN0) | (1<<TXEN0);
}

uint8_t isTransmitUart0(void)
{
	if(UCSR0B & (1<<TXCIE0))
		return 1;
	return 0;
}	
	
uint8_t isReceivingUart0(void)
{
	if(UCSR0B & (1<<RXCIE0))
		return 1;
	return 0;
}