/*!
 *  @file uart.h
 *	@brief Obsługa komunikacji szeregowej UART
 */ 

#ifndef UART_H_
#define UART_H_

#include <stdint.h>
#include <avr/io.h>

#include "timers.h"
#include "crc.h"

//extern _uartData uart0;
extern uint8_t timersShort[TMR_SHORT_ALL];
/*! 
	@def UART_BUFFER_SIZE
    @brief Wielkość buforów na dane dla UART-ów.
*/
#define UART_BUFFER_SIZE						300

/*!
	@enum _baudRate
	@brief Prędkość z jaką odbywa się komunikacja
	
	@b UWAGA Reprezentacja bitowa poszcególnych elementów odpowiada wartości konfiguracyjnej danej prędkości
	w wyświetlaczu LCD.
*/
typedef enum {
	BAUDRATE_1200			= 120,				/*!< Prędkość komunikacji 1200bps */
	BAUDRATE_2400			= 240,				/*!< Prędkość komunikacji 2400bps */
	BAUDRATE_4800			= 480,				/*!< Prędkość komunikacji 4800bps */
	BAUDRATE_9600			= 960,				/*!< Prędkość komunikacji 9600bps */
	BAUDRATE_19200			= 1920,				/*!< Prędkość komunikacji 19200bps */
	BAUDRATE_38400			= 3840,				/*!< Prędkość komunikacji 38400bps */
	BAUDRATE_57600			= 5760,				/*!< Prędkość komunikacji 57600bps */
	BAUDRATE_115200			= 11520,			/*!< Prędkość komunikacji 115200bps */
	DEFAULT_BAUD_RATE = BAUDRATE_9600
}_baudRate;

/*!
	@struct _uartData
	@brief Dane dla UART-u.
	
	Struktura przechowująca dane dla komunikacji UART.
*/
typedef struct
{
	_baudRate baudRate;							/*!< Prędkość komunikacji ustawiona na danym porcie */
	uint8_t bufferRx[UART_BUFFER_SIZE];			/*!< Bufor odbieranego komunikatu */
	uint8_t bufferTx[UART_BUFFER_SIZE];			/*!< Bufor wysylanego komunikatu */
	int16_t counterRx;							/*!< Ilość otrzymanych do tej pory bajtów */
	int16_t counterTx;							/*!< Ilość bajtów jakie należy wysłać */
	int16_t currentId;							/*!< indeks bajtu który ma być wysłany */
	uint16_t crc;								/*!< Suma crc odebranych danych */
	uint8_t readyToParse;						/*!< Flaga wskazujaca ze odebrano cala ramke ktora jest gotowa do obsluzenia */
	
	/*!
		@brief Włączenie odbióru danych
		@details Podczas gdy metoda zostanie włączona podaczas wysyłania komunikatu, nastąpi
		przerwanie wysyłania komunikatu i przełączenie w tryb odbierania.
	*/
	void (*startReceiving)(void);
	
	/*!
		@brief Wyłączenie odbióru danych
		@details Metoda wyłącza odbiornik w celu umożliwienia przygotowania danych do wysłania
		(zapisu do bufora).
	*/
	void (*stopReceiving)(void);
	
	/*!
		@brief Rozpoczęcie wysyłania danych
		@details Metoda wyłącza odbiornik (jeśli nie był wcześniej wyłączony), aktywuje nadajnik
		i rozpoczyna wysyłanie danych.
		@param frameSize Ilość bajtów wysyłanego komunikatu
	*/
	void (*startTransmit)(int16_t frameSize);
	
	/*!
		@brief Ustawienie prędkości transmisji
		@details Metoda zmienia prędkość komunikacji danego portu i zeruje stan odebranych/wysłanych danych.
		@param baudRate Prędkość transmisji jaką należy ustawić
	*/
	void (*setBaudRate)(_baudRate baudRate);
		
	/*!
		@brief Okresla czy trwa obecnie wysylanie danych
	*/
	uint8_t (*isTransmit)(void);
	
	/*!
		@brief Okresla czy trwa obecnie pobieranie danych
	*/
	uint8_t (*isReceiving)(void);
		
	/*!
		@brief Analiza ramki
		@details Metoda analizuje otrzymaną ramkę i w razie potrzeby przygotowuje odpowiedź.
		@return Ilość bajtów jaka jest przygotowana do odesłania. W przypadku gdy nie potrzebna jest odpowiedź,
			zwracana jest wartość 0.
	*/
	uint16_t (*parseFrame)(void);
	
}_uartData;

/*!
	@brief Wykrywanie oraz obsługa komunikatów na obu portach
	
	@b UWAGA Funkcja ta musi być wywoływana możliwie jak najczęściej (najlepiej w przebiegu pętli głównej progrmu).
*/
void uartService(void);

/*!
    @brief Inizjalizacja portu UART
*/
void initUart(void);

/*!
	@brief Wyłączenie obu portów uart
*/
void uartTurnOff(void);

/*!
    @brief Wyczyszczenie struktury.
	@param data Wskaźnik do struktury danych
*/
void uartClear(_uartData * data);

/// Struktura danych dla komunikacji z WAGO
extern _uartData uart0;
//uart0:
/*
//stara wersja, urzadzenie restartuje sie czesto
static inline void uart0RxVectImpl(void)
{
	uint8_t tmp = UDR0;
	if(uart0.counterRx < UART_BUFFER_SIZE)
	{
		uart0.bufferRx[uart0.counterRx++] = tmp;
		setTimerShort(TMR_SHORT_UART0, UART_WAGO_TIMEOUT);
	}	
}
*/

static inline void uart0RxVectImpl(void)
{
	uint8_t tmp = UDR0;
	
	//jesli nie zakonczono obrobki poprzedniego komunikatu, to nie analizuj tego bajtu:
	if(uart0.readyToParse)
	{
		return;
	}		
		
	if(!timersShort[TMR_SHORT_UART0] || uart0.counterRx >= UART_BUFFER_SIZE)
	{
		uart0.counterRx = 0;
		uart0.counterTx = 0;
		uart0.currentId = 0;
		uart0.crc = 0xFFFF;
	}
	uart0.bufferRx[uart0.counterRx++] = tmp;
	crc16ModbusNext(tmp, &uart0.crc);
	if(uart0.crc == 0)
	{
		uint8_t cmdWago = uart0.bufferRx[1];
		//przetestowanie pod katem ilości odebranych danych w porównaniu to wywolywanego polecenia (WAGO):
		if(cmdWago >= 1 && cmdWago <= 6)
		{
			if(uart0.counterRx == 8)
			{
				setTimerShort(TMR_SHORT_UART0, 0);
				uart0.readyToParse = 1;
				return;
			}						
		}
		else if(cmdWago == 0x0F ||
				cmdWago == 0x10)
		{
			if(uart0.counterRx == 9 + uart0.bufferRx[6])
			{
				setTimerShort(TMR_SHORT_UART0, 0);
				uart0.readyToParse = 1;
				return;
			}						
		}
	}
	setTimerShort(TMR_SHORT_UART0, UART_WAGO_TIMEOUT);
}
static inline void uart0UdreVectImpl(void)
{
	if(uart0.currentId < UART_BUFFER_SIZE && uart0.counterTx > uart0.currentId)
		UDR0 = uart0.bufferTx[uart0.currentId++];
	else
		UCSR0B &= ~(1<<UDRIE0);	
}
static inline void uart0TxVectImpl(void)
{
	uart0.startReceiving();
}
#endif /* UART_H_ */
