﻿/*
 * ds.c
 *
 * Created: 2012-02-29 00:36:52
 *  Author: marek
 */ 

#include "processor.h"

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>

#include "ds.h"
#include "main.h"
#include "lcdButton.h"
#include "led.h"
#include "eemem.h"

// global search state
unsigned char ROM_NO[DS_ROM_CODE_LENGTH];

//tablica aktualnie wykrytych czujników:
uint8_t dsRomCode[DS_SENSOR_COUNT][DS_ROM_CODE_LENGTH];		/**< ROM CODE czujników DS */


uint8_t const PROGMEM crcDsTable[] = {
		0, 94,188,226, 97, 63,221,131,194,156,126, 32,163,253, 31, 65,
      157,195, 33,127,252,162, 64, 30, 95,  1,227,189, 62, 96,130,220,
       35,125,159,193, 66, 28,254,160,225,191, 93,  3,128,222, 60, 98,
      190,224,  2, 92,223,129, 99, 61,124, 34,192,158, 29, 67,161,255,
       70, 24,250,164, 39,121,155,197,132,218, 56,102,229,187, 89,  7,
      219,133,103, 57,186,228,  6, 88, 25, 71,165,251,120, 38,196,154,
      101, 59,217,135,  4, 90,184,230,167,249, 27, 69,198,152,122, 36,
      248,166, 68, 26,153,199, 37,123, 58,100,134,216, 91,  5,231,185,
      140,210, 48,110,237,179, 81, 15, 78, 16,242,172, 47,113,147,205,
       17, 79,173,243,112, 46,204,146,211,141,111, 49,178,236, 14, 80,
      175,241, 19, 77,206,144,114, 44,109, 51,209,143, 12, 82,176,238,
       50,108,142,208, 83, 13,239,177,240,174, 76, 18,145,207, 45,115,
      202,148,118, 40,171,245, 23, 73,  8, 86,180,234,105, 55,213,139,
       87,  9,235,181, 54,104,138,212,149,203, 41,119,244,170, 72, 22,
      233,183, 85, 11,136,214, 52,106, 43,117,151,201, 74, 20,246,168,
      116, 42,200,150, 21, 75,169,247,182,232, 10, 84,215,137,107, 53};


uint8_t dsReadWriteBit(uint8_t data)
{
	uint8_t result = 0;
	dsForceHigh();
	wait_us(20);
	dsForceLow();
	wait_1us();
	if(data)
		dsForceHigh();
	wait_us(12);
	if(dsInput())
		result = 1;
	wait_us(50);
	dsForceHigh();
	return result;
}

uint8_t dsStartSeq(void)
{
	dsForceHigh();
	wait_1us();
	dsForceLow();
	wait_us(500);
	dsForceHigh();
	wait_us(61);
	uint8_t result = dsInput();
	wait_us(500);
	return result;
}

uint8_t dsReadWriteByte(uint8_t data)
{
	uint8_t result = 0;
	uint8_t bit = 0;
	for(uint8_t i=0; i<8; i++)
	{
		result = (result>>1);		
		bit = dsReadWriteBit(data & _BV(0));
		if(bit)
			result |= _BV(7);
		data = (data>>1);
	}
	return result;
}

/*!
	Obliczenie sumy CRC dla ds
*/
uint8_t crc8Ds(uint8_t * data, int length)
{
	uint8_t result = 0;
	for(int i=0; i<length; i++)
	{
		result =  pgm_read_byte(&crcDsTable[result ^ (*data)]);
		data++;
	}
	return result;
}	

//==========================//
// Funkcjonalnosc			//
//==========================//

void dsPrepareToWork(void)
{
	//rozgloszeniowe ustawienie rejestru konfiguracyjnego:
	if(!dsStartSeq())
	{
		//zapis 11bitowej dokladnosci pomiaru
		dsReadWriteByte(DS_CMD_SPROM);
		dsReadWriteByte(DS_FNC_WRSCR);
		dsReadWriteByte(0);	//SCRATCHPAD
		dsReadWriteByte(0);	//SCRATCHPAD
		dsReadWriteByte(0x5F);	//REJESTR  KONFIGURACYJNY
	}
	
	eepromReadDsRomCode(dsRomCode[0]);
	uint16_t dsStatusRegister = readModbus(MODBUS_READ_ONLY_BIT_DATA_2);
	uint16_t mask = 1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_ROM;
	uint8_t dsEmptyRom = DS_SENSOR_COUNT;
	//sprawdzenie zapisanych w eepromie numerów ROM CODE:
	for(uint8_t dsIndex = 0; dsIndex<DS_SENSOR_COUNT; dsIndex++)
	{
		if((dsRomCode[dsIndex][0] == FAMILY_CODE_DS18B20  || dsRomCode[dsIndex][0] == FAMILY_CODE_DS18S20) 		//warunek niezbędny dla przypadku gdy cała sprawdzana tablica jest wyzerowana (wowczas crc = 0)
			&& crc8Ds(dsRomCode[dsIndex], DS_ROM_CODE_LENGTH-1) == dsRomCode[dsIndex][DS_ROM_CODE_LENGTH-1])
		{
			//crc zgodne, dane prezentuja ROM CODE czujnika
			dsStatusRegister |= mask;
			dsEmptyRom--;			
		}
		else
		{
			//crc niezgodne, dane NIE prezentuja ROM CODE czujnika;
			dsStatusRegister &= ~mask;
		}
		mask <<= 1;
	}	
	
	
	uint8_t firstLoop = 1;
	while(dsEmptyRom > 0)
	{
		//jeśli nie wszystkie miejsca na czujniki są zajęte, to sprawdzamy podpięte czujniki:
		uint8_t searchResult;
		if(firstLoop)
			searchResult = OWFirst();
		else
			searchResult = OWNext();
		firstLoop = 0;
		if(searchResult)
		{
			//jeśli wykryto czujnik, sprawdzic czy już nie jest zapisany jego kod ROM:
			uint8_t romUsed = 0;
			for(int i = 0; i <DS_SENSOR_COUNT; i++)
			{
				if(dsRomCode[i][0] == FAMILY_CODE_DS18B20 || dsRomCode[i][0] == FAMILY_CODE_DS18S20)
				{
					uint8_t differentRomFlag = 0;
					for(int j = 0; j <DS_SENSOR_COUNT; j++)
					{
						if(dsRomCode[i][j] != ROM_NO[j])
						{
							differentRomFlag = 1;
							break;
						}							
					}
					if(differentRomFlag == 0)
					{
						romUsed = 1;
						break;
					}
				}	
			}
			
			if(!romUsed)
			{
				//jeśli wykryty kod ROM należy do nowego czujnika, należy zapisać jego kod:
				//wyszukiwanie wolnego miejsca:
				for(int i = 0; i <DS_SENSOR_COUNT; i++)
				{
					if(dsRomCode[i][0] != FAMILY_CODE_DS18B20 && dsRomCode[i][0] != FAMILY_CODE_DS18S20)
					{
						//puste miejsce => zapisujemy tu kod ROM nowego czujnika
						for(int j=0; j<8; j++)
							dsRomCode[i][j] = ROM_NO[j];
						dsStatusRegister |= (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_ROM+i));
						ledShowNewDs();
						break;
					}						
				}
			}
		}
		else
		{
			//nie wykryto kodu ROM czujnika
			break;
		}
		dsEmptyRom--;
	}
	writeSpecialModbus(MODBUS_READ_ONLY_BIT_DATA_2, dsStatusRegister);
	//wyzerowanie wartosci temperatur:
	for(int i = 0; i < DS_SENSOR_COUNT; i++)
	{
		writeSpecialModbus(MODBUS_READ_ONLY_TEMP_0 + i, 0);
	}	
	eepromWriteDsRomCode(dsRomCode[0]);
}


 
void dsServiceWork(void)
{
	static _dsWorkState dsWorkState = DS_STATE_IDLE;
	static uint8_t dsIndeks = 0;		//indeks czujnika z ktorego należy odczytac temperatu

	uint16_t dsStatusRegister;
	switch(dsWorkState)
	{
		case DS_STATE_IDLE:
			if(!getTimer100msec(TMR_100MSEC_CONVERT_TEMP_DELAY))
			{
				dsWorkState = DS_STATE_CONV;
				if(!dsStartSeq())
				{
					dsReadWriteByte(DS_CMD_SPROM);
					dsReadWriteByte(DS_FNC_CONV);
					setTimer100msec(TMR_100MSEC_CONVERT_TEMP, 5);
				}
				else
				{
					//na linii nie ma zadnego czujnika:
					dsWorkState = DS_STATE_IDLE;
					setTimer100msec(TMR_100MSEC_CONVERT_TEMP_DELAY, 10);
				}					
			}
			break;
		case DS_STATE_CONV:
			if(dsInput() || !getTimer100msec(TMR_100MSEC_CONVERT_TEMP))
			{
				//koniec pomiaru lub przekroczony zostal przeznaczony na to maksymalny czas:
				dsWorkState = DS_STATE_READ_TEMP;
				dsIndeks = 0;
			}
			break;
		case DS_STATE_READ_TEMP:
			dsStatusRegister = readModbus(MODBUS_READ_ONLY_BIT_DATA_2);
			if(dsStatusRegister & (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_ROM + dsIndeks)))
			{
				//jesli pod dana pozycja zapisany jest jakis kod ROM:
				uint16_t mask = (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_OK + dsIndeks));
				dsStatusRegister &= ~mask;
				if(!dsStartSeq())
				{
					uint8_t dsData[9];
					dsReadWriteByte(DS_CMD_MAROM);
					for(uint8_t i=0; i<8; i++)
					{
						dsReadWriteByte(dsRomCode[dsIndeks][i]);
					}
					dsReadWriteByte(DS_FNC_RDSCR);
					for(uint8_t i=0; i<9; i++)
					{
						dsData[i] = dsReadWriteByte(0xFF);
					}
					uint8_t crc = crc8Ds(dsData, 8);					
					if(crc == dsData[8])
					{
						//jesli crc jest zgodne:
						dsStatusRegister |= mask;
						int16_t temp = 0;
						int8_t powerOnResetFlag = 0;
						switch(dsRomCode[dsIndeks][0])
						{
							case FAMILY_CODE_DS18S20:
								//sprawdz czy zmierzona temperatura jest inna niz 85stC (wartosc inicjalizacyjna czujnika ds):
								if(dsData[0] == 0xAA && dsData[1] == 0x00)
								{
									powerOnResetFlag = 1;
								}
								else
								{
									temp = (int16_t)((dsData[0] >> 1) & 0xFF);
									if(dsData[1])
									{
										temp |= 0xFF80;
									}
								}
								break;
							case FAMILY_CODE_DS18B20:
								//sprawdz czy zmierzona temperatura jest inna niz 85stC (wartosc inicjalizacyjna czujnika ds):
								if(dsData[0] == 0x50 && dsData[1] == 0x05)
								{
									powerOnResetFlag = 1;
								}
								else
								{
									temp = (((int16_t)(dsData[1] & 0x07)) << 4) | ((int16_t)dsData[0] >> 4);
									if(dsData[1] & 0x80)
									{
										temp |= 0xFF80;
									}
								}
								break;
						}
						if(!powerOnResetFlag && readModbus(MODBUS_READ_ONLY_TEMP_0 + dsIndeks) != temp)
						{
							writeSpecialModbus(MODBUS_READ_ONLY_TEMP_0 + dsIndeks, (uint16_t)temp);
						}
					}
				}
				writeSpecialModbus(MODBUS_READ_ONLY_BIT_DATA_2, dsStatusRegister);
			}
			dsIndeks++;
			if(dsIndeks >= DS_SENSOR_COUNT)
			{
				dsWorkState = DS_STATE_IDLE;
				setTimer100msec(TMR_100MSEC_CONVERT_TEMP_DELAY, 10);
			}			
			break;
	}
	
}

void dsDeleteRomCodes(uint16_t deleteMask)
{
	uint16_t deleteMaskReader = (1<<BIT_MODBUS_VOLATILE_DS_0_CONTROL);
	uint16_t romStatusMaskWriter = (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_ROM) | (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_OK);
	uint16_t dsStatusRegister = readModbus(MODBUS_READ_ONLY_BIT_DATA_2);
	for(uint8_t i=0; i<DS_SENSOR_COUNT; i++)
	{
		if(deleteMaskReader & deleteMask)
		{
			dsStatusRegister &= ~romStatusMaskWriter;			
			for(uint8_t j=0; j<DS_ROM_CODE_LENGTH; j++)
			{
				dsRomCode[i][j] = 0xFF;
			}
			writeSpecialModbus(MODBUS_READ_ONLY_TEMP_0 + i, 0);
		}
		deleteMaskReader <<= 1;
		romStatusMaskWriter <<= 1;
	}
	writeSpecialModbus(MODBUS_READ_ONLY_BIT_DATA_2, dsStatusRegister);
	eepromWriteDsRomCode(dsRomCode[0]);
}

void dsSwapRomCodes(uint16_t swapMask)
{
	uint8_t romCodeSrcIndex = 0xFF;
	uint8_t romCodeDestIndex = 0xFF;	
	uint16_t swapMaskReader = (1<<BIT_MODBUS_VOLATILE_DS_0_CONTROL);
	uint16_t dsStatusRegister = readModbus(MODBUS_READ_ONLY_BIT_DATA_2);
	for(uint8_t i=0; i<DS_SENSOR_COUNT; i++)
	{
		if(swapMaskReader & swapMask)
		{
			if(romCodeSrcIndex == 0xFF)
			{
				romCodeSrcIndex = i;
			}				
			else
			{
				//wybrano drugi czujnik do zamiany:
				romCodeDestIndex = i;
				uint8_t tmpData;
				for(uint8_t j=0; j<DS_ROM_CODE_LENGTH; j++)
				{
					tmpData = dsRomCode[romCodeDestIndex][j];
					dsRomCode[romCodeDestIndex][j] = dsRomCode[romCodeSrcIndex][j];
					dsRomCode[romCodeSrcIndex][j] = tmpData;
				}
				
				uint16_t tmpStatusData = dsStatusRegister;
				uint16_t romStatusSrcMaskWriter = (((1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_ROM) | (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_OK)) << romCodeSrcIndex);
				uint16_t romStatusDestMaskWriter = (((1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_ROM) | (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_OK)) << romCodeDestIndex);
				dsStatusRegister &= ~(romStatusSrcMaskWriter | romStatusDestMaskWriter);				//wyczyszczenie obecnych wartości
				dsStatusRegister |= (tmpStatusData & romStatusSrcMaskWriter) << (romCodeDestIndex-romCodeSrcIndex);			//skopiowanie bitow statusowych pierwszego czujnika
				dsStatusRegister |= (tmpStatusData & romStatusDestMaskWriter) >> (romCodeDestIndex-romCodeSrcIndex);		//skopiowanie bitow statusowych drugiego czujnika
				break;
			}
		}
		swapMaskReader <<= 1;
	}	
	writeSpecialModbus(MODBUS_READ_ONLY_BIT_DATA_2, dsStatusRegister);
	eepromWriteDsRomCode(dsRomCode[0]);
}


//======================//
// SEARCHING			//
//======================//

// global search state
int LastDiscrepancy;
int LastFamilyDiscrepancy;
int LastDeviceFlag;
unsigned char crc8;

// method declarations
int  OWVerify();
void OWTargetSetup(unsigned char family_code);
void OWFamilySkipSetup();
int  OWSearch();

/*!
	@brief Reset the 1-Wire bus and return the presence of any device
	@details From Maxim website.
	@return true : device present; false : no device present
*/
#define OWReset() (dsStartSeq() == 0)

/*!
	@brief Send 8 bits of data to the 1-Wire bus
	@details From Maxim website.
*/
#define OWWriteByte(byte_value) dsReadWriteByte(byte_value)

/*!
	@brief Send 1 bit of data to teh 1-Wire bus
	@details From Maxim website.
*/
#define OWWriteBit(bit_value) dsReadWriteBit(bit_value)

/*!
	@brief Read 1 bit of data from the 1-Wire bus 
	@details From Maxim website.
	@return 1 : bit read is 1; 0 : bit read is 0
*/
#define OWReadBit() (unsigned char)dsReadWriteBit(0xFF)

/*!
	@brief Calculate the CRC8 of the byte value provided with the current global 'crc8' value.
	@return current crc8 value
*/
#define docrc8(crc8, value) pgm_read_byte(&crcDsTable[crc8 ^ value])


int OWFirst()
{
   // reset the search state
   LastDiscrepancy = 0;
   LastDeviceFlag = false;
   LastFamilyDiscrepancy = 0;

   return OWSearch();
}

int OWNext()
{
   // leave the search state alone
   return OWSearch();
}

/*!
	@brief Perform the 1-Wire Search Algorithm on the 1-Wire bus using the existing search state.
	@details From Maxim website.
	@return true : device found, ROM number in ROM_NO buffer; false : device not found, end of search
*/
int OWSearch()
{
   int id_bit_number;
   int last_zero, rom_byte_number, search_result;
   int id_bit, cmp_id_bit;
   unsigned char rom_byte_mask, search_direction;

   // initialize for search
   id_bit_number = 1;
   last_zero = 0;
   rom_byte_number = 0;
   rom_byte_mask = 1;
   search_result = 0;
   crc8 = 0;

   // if the last call was not the last one
   if (!LastDeviceFlag)
   {
      // 1-Wire reset
      if (!OWReset())
      {
         // reset the search
         LastDiscrepancy = 0;
         LastDeviceFlag = false;
         LastFamilyDiscrepancy = 0;
         return false;
      }

      // issue the search command 
      OWWriteByte(0xF0);  

      // loop to do the search
      do
      {
         // read a bit and its complement
         id_bit = OWReadBit();
         cmp_id_bit = OWReadBit();

         // check for no devices on 1-wire
         if ((id_bit == 1) && (cmp_id_bit == 1))
            break;
         else
         {
            // all devices coupled have 0 or 1
            if (id_bit != cmp_id_bit)
               search_direction = id_bit;  // bit write value for search
            else
            {
               // if this discrepancy if before the Last Discrepancy
               // on a previous next then pick the same as last time
               if (id_bit_number < LastDiscrepancy)
                  search_direction = ((ROM_NO[rom_byte_number] & rom_byte_mask) > 0);
               else
                  // if equal to last pick 1, if not then pick 0
                  search_direction = (id_bit_number == LastDiscrepancy);

               // if 0 was picked then record its position in LastZero
               if (search_direction == 0)
               {
                  last_zero = id_bit_number;

                  // check for Last discrepancy in family
                  if (last_zero < 9)
                     LastFamilyDiscrepancy = last_zero;
               }
            }

            // set or clear the bit in the ROM byte rom_byte_number
            // with mask rom_byte_mask
            if (search_direction == 1)
              ROM_NO[rom_byte_number] |= rom_byte_mask;
            else
              ROM_NO[rom_byte_number] &= ~rom_byte_mask;

            // serial number search direction write bit
            OWWriteBit(search_direction);

            // increment the byte counter id_bit_number
            // and shift the mask rom_byte_mask
            id_bit_number++;
            rom_byte_mask <<= 1;

            // if the mask is 0 then go to new SerialNum byte rom_byte_number and reset mask
            if (rom_byte_mask == 0)
            {
                crc8 = docrc8(crc8, ROM_NO[rom_byte_number]);  // accumulate the CRC
                rom_byte_number++;
                rom_byte_mask = 1;
            }
         }
      }
      while(rom_byte_number < 8);  // loop until through all ROM bytes 0-7

      // if the search was successful then
      if (!((id_bit_number < 65) || (crc8 != 0)))
      {
         // search successful so set LastDiscrepancy,LastDeviceFlag,search_result
         LastDiscrepancy = last_zero;

         // check for last device
         if (LastDiscrepancy == 0)
            LastDeviceFlag = true;
         
         search_result = true;
      }
   }

   // if no device found then reset counters so next 'search' will be like a first
   if (!search_result || !ROM_NO[0])
   {//wnetrze_if2:
      LastDiscrepancy = 0;
      LastDeviceFlag = false;
      LastFamilyDiscrepancy = 0;
      search_result = false;
   }

   return search_result;
}

/*!
	@brief Verify the device with the ROM number in ROM_NO buffer is present.
	@details From Maxim website.
	@return true : device verified present; false : device not present
*/
int OWVerify()
{
   unsigned char rom_backup[8];
   int i,rslt,ld_backup,ldf_backup,lfd_backup;

   // keep a backup copy of the current state
   for (i = 0; i < 8; i++)
      rom_backup[i] = ROM_NO[i];
   ld_backup = LastDiscrepancy;
   ldf_backup = LastDeviceFlag;
   lfd_backup = LastFamilyDiscrepancy;

   // set search to find the same device
   LastDiscrepancy = 64;
   LastDeviceFlag = false;

   if (OWSearch())
   {
      // check if same device found
      rslt = true;
      for (i = 0; i < 8; i++)
      {
         if (rom_backup[i] != ROM_NO[i])
         {
            rslt = false;
            break;
         }
      }
   }
   else
     rslt = false;

   // restore the search state 
   for (i = 0; i < 8; i++)
      ROM_NO[i] = rom_backup[i];
   LastDiscrepancy = ld_backup;
   LastDeviceFlag = ldf_backup;
   LastFamilyDiscrepancy = lfd_backup;

   // return the result of the verify
   return rslt;
}

/*!
	@brief Setup the search to find the device type 'family_code' on the next call to OWNext() if it is present.
	@details From Maxim website.
*/
void OWTargetSetup(unsigned char family_code)
{
   int i;

   // set the search state to find SearchFamily type devices
   ROM_NO[0] = family_code;
   for (i = 1; i < 8; i++)
      ROM_NO[i] = 0;
   LastDiscrepancy = 64;
   LastFamilyDiscrepancy = 0;
   LastDeviceFlag = false;
}

/*!
	@brief Setup the search to skip the current device type on the next call to OWNext().
	@details From Maxim website.
*/
void OWFamilySkipSetup()
{
   // set the Last discrepancy to last family discrepancy
   LastDiscrepancy = LastFamilyDiscrepancy;
   LastFamilyDiscrepancy = 0;

   // check for end of list
   if (LastDiscrepancy == 0)
      LastDeviceFlag = true;
}

