﻿/*!
 *  @file lcd.c
 */ 

#include <avr/io.h>
#include "string.h"
#include "lcd.h"
#include "crc.h"
#include "lcdButtonTimers.h"
#include "uart.h"
#include "protocolLcd.h"
#include "main.h"

_lcdData lcdData;

/*!
	@brief Kolejka zadań.
	@details Kolejka zadań musi być obsługiwana tylko poprzez funkce:
		getLcdWorkQueuePeek(void), getLcdWorkQueueFree(void), enqueueLcdWork(void), oraz dequeueLcdWork(void)
*/
_lcdStateContext lcdWorkQueue[LCD_WORK_QUEUE_MAX]; /*!<  */
_lcdStateContext * lcdActuallContext;
uint8_t lcdWorkQueueFirstIndex;			/*!< Jeśli kolejka zadań nie jest pusta, to wskazuje na pierwsze zadanie w kolejce */
uint8_t lcdWorkQueueLastIndex;			/*!< Jeśli kolejka zadań nie jest pusta, to wskazuje na ostatnie zadanie w kolejce */
uint8_t lcdWorkQueueNotEmptyFlag;		/*!< Określa czy kolejka jest pusta (0) lub czy jest w niej jakiekolwiek zadanie (!0) */

_alarmBuzzerState alarmBuzzerState = ALARM_BUZZER_IDLE;

extern _lcdButtonData lcdButtonData;

/*!
	@brief Prędkość z jaką odbywać się będzie komunikacja z wyświetlaczem 
*/
const _baudRate LCD_BAUDRATE = BAUDRATE_115200;

/*!
	@brief Konfiguracja wyświetlacza
	
	Opis konfiguracji dla wartości 0x0C:
	-# upload command 0x72 when release touch panel
	-# command 0x73 will be uploaded every 100 ms until touch screen is released
	-# no interface
	-# backlight will not be controlled by touch screen or keyboard
	-# buzzer does not ring when touch screen is clicked or the keyboard is pressed
	-# 90 deg display rotate
*/
const uint8_t LCD_CONFIG = 0x0C;

/**************************************************************************
Funkcje wewnetrzne:
**************************************************************************/
//////////////////////////////////////////////////////////////////////////
void lcdStateInit(void)
{
	_baudRate nextBaudRate;
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_INIT_START:
			setTimer1Sec(TMR_1SEC_INIT_LCD, 3);
			lcdActuallContext->substate = LCD_SUBSTATE_INIT_ASK;
			break;
		case LCD_SUBSTATE_INIT_ASK:
			if(!uartLCD->isBusy() && !getTimer1Sec(TMR_1SEC_INIT_LCD))
			{
 				lcdSendHandshake();
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_INIT_RESP_WAIT;
			}
			break;
		case LCD_SUBSTATE_INIT_RESP_WAIT:
			if(lcdData.version[0] != 0 || lcdData.version[1] != 0)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_INIT_FOUND;
			}
			else if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				nextBaudRate = uartGetNextBaudRate(uartLCD->baudRate);
				if(nextBaudRate == BAUDRATE_NONE)
				{
					lcdActuallContext->substate = LCD_SUBSTATE_INIT_ERR;
				}
				else
				{
					uartLCD->setBaudRate(nextBaudRate);
					lcdActuallContext->substate = LCD_SUBSTATE_INIT_ASK;
				}
			}
			break;
		case LCD_SUBSTATE_INIT_FOUND:
			if(lcdData.baudRate == LCD_BAUDRATE && lcdData.options == LCD_CONFIG)
			{
				dequeueLcdWork();
			}
			else
			{
				if(!uartLCD->isBusy())
				{
					lcdSendConfig(LCD_BAUDRATE, LCD_CONFIG);
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_INIT_SET_BAUD_RATE;
				}					
			}
			break;
		case LCD_SUBSTATE_INIT_SET_BAUD_RATE:
			if(!uartLCD->isBusy() && !getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				lcdData.version[0] = 0;
				lcdData.version[1] = 0;
 				uartLCD->setBaudRate(LCD_BAUDRATE);
				lcdSendHandshake();
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_INIT_RESP_WAIT;
			}
			break;
		case LCD_SUBSTATE_INIT_ERR:
			dequeueLcdWork();
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_INIT_START;
			break;
	}
}
//////////////////////////////////////////////////////////////////////////
/*!
	@brief Obsługa zadania polegającego na wyswietlaniu loga, grafik i tekstów laczenia sie z kontrolerem WAGO
*/


void lcdStateWagoNoConnect()
{
	static uint8_t nextSubstate;
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_WAGO_NO_CONNECT_START:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0, 79, 1);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_START;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_INIT_DATA;
			}
			break;
		//wyświetlenie LOGO:
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_LOGO:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0x000A27, 14, 1);
				lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_LOGO;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_LOGO;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_LOGO:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0x000A27 + 14, 14, 1);
				lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_LOGO;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_LOGO;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_LOGO:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneTextRead(0, 1);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_LOGO;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_TEXT_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_LOGO;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_LOGO:
			setTimer1Sec(TMR_1SEC_INIT_LCD, 2);
			lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECTING;
			break;
		//wyświetlenie wizualizacji trwania połączenia:
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECTING:
			if(!getTimer1Sec(TMR_1SEC_INIT_LCD))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendSceneDataRead(0x000A27, 14, 1);
					lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECTING;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
					nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECTING;
				}
			}				
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECTING:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0x000A27 + 14*2, 14, 1);
				lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECTING;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECTING;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECTING:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneTextRead(1, 1);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECTING;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_TEXT_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECTING;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECTING:
			setTimer1Sec(TMR_1SEC_INIT_LCD, 2);
			lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_CHECK_MODBUS;
			break;
		//decyzja o udanym/nieudanym polaczeniu:
		case LCD_SUBSTATE_WAGO_NO_CONNECT_CHECK_MODBUS:
			if(!getTimer1Sec(TMR_1SEC_INIT_LCD))
			{
				if(lcdButtonData.modbusActive)
					lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_OK;	
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_ERROR;
				}					
			}
			break;
		//wyświetlenie wizualizacji udanego nawiązania połączenia:
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_OK:
			if(!getTimer1Sec(TMR_1SEC_INIT_LCD))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendSceneDataRead(0x000A27, 14, 1);
					lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_OK;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
					nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_OK;
				}
			}			
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_OK:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0x000A27 + 14*3, 14, 1);
				lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_OK;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_OK;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_OK:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneTextRead(2, 1);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_OK;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_TEXT_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECT_OK;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECT_OK:
			setTimer1Sec(TMR_1SEC_INIT_LCD, 1);
			lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_EXIT;
			break;
		//wyświetlenie wizualizacji braku połączenia:	
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_ERROR:
			if(!getTimer1Sec(TMR_1SEC_INIT_LCD))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendSceneDataRead(0x000A27, 14, 1);
					lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_ERROR;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
					nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_ERROR;
				}
			}			
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_ERROR:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0x000A27 + 14*4, 14, 1);
				lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_ERROR;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_ERROR;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_ERROR:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneTextRead(3, 1);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_ERROR;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_TEXT_GRAPHIC;
				nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECT_ERROR;
			}
			break;
		case LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECT_ERROR:
			setTimer1Sec(TMR_1SEC_INIT_LCD, 15);
			lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECTING;
			break;
		//koniec			
		case LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_EXIT:
			if(!getTimer1Sec(TMR_1SEC_INIT_LCD))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendSceneDataRead(0x000A27, 14, 1);
					lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_EXIT;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC;
					nextSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_EXIT;
				}
			}			
			break;
			
		case LCD_SUBSTATE_WAGO_NO_CONNECT_PRINT:
			if(!uartLCD->isBusy())
			{
				lcdSendDisplayIcon(
				*(lcdActuallContext->data+0),
				*((uint16_t *)(lcdActuallContext->data+1)),
				*((uint16_t *)(lcdActuallContext->data+3)),
				*((uint16_t *)(lcdActuallContext->data+5)),
				*((uint16_t *)(lcdActuallContext->data+7)),
				*((uint16_t *)(lcdActuallContext->data+9)),
				*((uint16_t *)(lcdActuallContext->data+11)));
				lcdActuallContext->substate = nextSubstate;
			}
			break;
			
		case LCD_SUBSTATE_WAGO_NO_CONNECT_SEND_TEXT:
			if(!uartLCD->isBusy())
			{
				lcdSendDisplayText(
					((uint16_t)lcdActuallContext->data[1]) << 8 | lcdActuallContext->data[0],				//x
					((uint16_t)lcdActuallContext->data[3]) << 8 | lcdActuallContext->data[2],				//y
					lcdActuallContext->data[4],																//library
					lcdActuallContext->data[5],																//font size
					((uint16_t)lcdActuallContext->data[7]) << 8 | lcdActuallContext->data[6],				//color
					(uint8_t *)(((uint16_t)lcdActuallContext->data[8]) << 8 | lcdActuallContext->data[8]),	//dataPnt
					40,//lcdButtonData.scene.system.textsLengths[lcdButtonData.scene.system.textsCount + 0],
					1);
				lcdActuallContext->substate = nextSubstate;
			}
			break;
		
		case LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_TEXT_GRAPHIC:
		case LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_INIT_DATA:
		case LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_ERROR;
				}
			}
			break;
					
		case LCD_SUBSTATE_WAGO_NO_CONNECT_EXIT:
			if(!getTimer1Sec(TMR_1SEC_INIT_LCD))
			{
				dequeueLcdWork();
				lcdBacklightService(LCD_BACKLIGHT_ON_CMD);
			}
			break;
			
		case LCD_SUBSTATE_WAGO_NO_CONNECT_ERROR:
		default:
			dequeueLcdWork();
			break;
	}		
}	

//////////////////////////////////////////////////////////////////////////
/*!
	@brief Obsługa zadania polegającego na wyczyszczeniu ekranu LCD podanym kolorem
	
	Wykorzystywane dane (kolejno w buforze data):
	- [2B] kolor tła
*/
void lcdStateClear(void)
{
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_CLEAR_START:
			if(!uartLCD->isBusy())
			{
				lcdSendColorPalette(0xFFFF /*bialy*/, lcdActuallContext->data[1]<<8 | lcdActuallContext->data[0]);
 				lcdActuallContext->substate = LCD_SUBSTATE_CLEAR_EXECUTE;
			}
			break;;				 
		case LCD_SUBSTATE_CLEAR_EXECUTE:
			if(!uartLCD->isBusy())
			{
				lcdSendClearScreen();
				lcdActuallContext->substate = LCD_SUBSTATE_CLEAR_EXIT;
				//lcdActuallContext->substate = LCD_SUBSTATE_CLEAR_TEST_1;
			}
			break;
			
		case LCD_SUBSTATE_CLEAR_EXIT:
			if(!uartLCD->isBusy())
			{				
 				uartWAGO->stopReceiving();
				memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, 6);
 				uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, 6);
  				uartWAGO->bufferTx[6] = (uint8_t)crc2;
  				uartWAGO->bufferTx[7] = (uint8_t)(crc2>>8);
 				uartWAGO->startTransmit(8);
				dequeueLcdWork();
			}
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_CLEAR_START;
			break;
	}		
}
//////////////////////////////////////////////////////////////////////////
/*!
	@brief Obsługa zadania polegającego na wysłaniu do wyświetlacza LCD wartości kolejnych pikseli
	
	Wykorzystywane dane (kolejno w buforze data):
	- [2B] wskaźnik na dane
*/
void lcdStatePrint(void)
{
	uint8_t * dataPnt;
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_PRINT_START:
			if(!uartLCD->isBusy())
			{
 				dataPnt = (uint8_t *)(lcdActuallContext->data[0] | lcdActuallContext->data[1]<<8);
 				lcdSendDirectVideo(dataPnt);
				setTimerShort(TMR_SHORT_LCD_NO_RESP, 100/TIMEOUT_UNIT);
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_EXIT;
			}			 
			break;
			
		case LCD_SUBSTATE_PRINT_EXIT:
			//if(!uartLCD->isBusy() && !getTimerShort(TMR_SHORT_LCD_NO_RESP))
			if(!uartLCD->isBusy())
			{
 				uartWAGO->stopReceiving();
				memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, 6);
 				uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, 6);
 				uartWAGO->bufferTx[6] = (uint8_t)crc2;
 				uartWAGO->bufferTx[7] = (uint8_t)(crc2>>8);
 				uartWAGO->startTransmit(8);
				dequeueLcdWork();
			}
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_PRINT_START;
			break;
	}		
}
//////////////////////////////////////////////////////////////////////////
/*!
	@brief Zapisanie aktualnego obranu wyświetlacza do pamięci
	
	Wykorzystywane dane (kolejno w buforze data):
	- [1B] identyfikator ekranu
*/
void lcdStateSaveImage(void)
{
	uint8_t screenId;
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_IMAGE_SAVE_START:
			if(!uartLCD->isBusy())
			{
				screenId = lcdActuallContext->data[0];
				lcdSendImageSave(screenId);
				setTimerShort(TMR_SHORT_LCD_NO_RESP, 10/TIMEOUT_UNIT);	//10ms
				lcdActuallContext->substate = LCD_SUBSTATE_IMAGE_SAVE_EXIT;
 			}
			break;
		case LCD_SUBSTATE_IMAGE_SAVE_EXIT:
			if(!uartLCD->isBusy() && !getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
 				uartWAGO->stopReceiving();
				memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, 6);
 				uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, 6);
 				uartWAGO->bufferTx[6] = (uint8_t)crc2;
 				uartWAGO->bufferTx[7] = (uint8_t)(crc2>>8);
 				uartWAGO->startTransmit(8);
				dequeueLcdWork();
			}
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_IMAGE_SAVE_START;
			break;
	}		
}
//////////////////////////////////////////////////////////////////////////
/*!
	@brief Obsługa zadania polegającego na odczycie wartości crc wyświetlanego obrazu
	
	Wykorzystywane dane (kolejno w buforze data):
	- [1B] identyfikator ekranu
*/
void lcdStateImageCrc(void)
{
	static uint8_t trialCount = 0;
	uint8_t screenId;
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_IMAGE_CRC_START:
			trialCount = 5;
			lcdActuallContext->substate = LCD_SUBSTATE_IMAGE_CRC_ASK;
			break;
		case LCD_SUBSTATE_IMAGE_CRC_ASK:
			if(trialCount > 0)
			{
				if(!uartLCD->isBusy() && !getTimer1Sec(TMR_SHORT_LCD_NO_RESP))
				{
					screenId = lcdActuallContext->data[0];
					lcdSendImagePrintWithCrc(screenId);
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					trialCount--;
				}
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_IMAGE_CRC_EXIT;
			}
			break;
		case LCD_SUBSTATE_IMAGE_CRC_EXIT:
			dequeueLcdWork();
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_IMAGE_CRC_START;
			break;
	}		
}
//////////////////////////////////////////////////////////////////////////
void lcdStateProjectSave(void)

{
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_PROJECT_SAVE_START:
			lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_SAVE_WORKING;
			break;
		case LCD_SUBSTATE_PROJECT_SAVE_WORKING:
			if(!uartLCD->isBusy() && 
				!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				int bytesToSend = 0;
				uartWAGO->stopReceiving();
				memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, 6);
				uartWAGO->bufferTx[bytesToSend++] = uartWAGO->bufferRx[0];
				uartWAGO->bufferTx[bytesToSend++] = uartWAGO->bufferRx[1] | 0x80;
				uartWAGO->bufferTx[bytesToSend++] = MODBUS_EXCEPTION_WRITE;
				uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, bytesToSend);
				uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
				uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
				uartWAGO->startTransmit(8);
				dequeueLcdWork();
			}			
			break;
		case LCD_SUBSTATE_PROJECT_SAVE_EXIT:
			if(MODBUS_ADDRESS_DEST_TEST(uartWAGO->bufferRx[0]))
			{
				uartWAGO->stopReceiving();
				memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, 6);
 				uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, 6);
 				uartWAGO->bufferTx[6] = (uint8_t)crc2;
 				uartWAGO->bufferTx[7] = (uint8_t)(crc2>>8);
 				uartWAGO->startTransmit(8);
			}
			dequeueLcdWork();
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_SAVE_START;
			break;
	}		
}
//////////////////////////////////////////////////////////////////////////
void lcdStateSceneActive(uint8_t systemSceneFlag)
{
	static int objectsCounter;
	static uint32_t relAddress;
	static uint8_t withBackgroundFlag;
	//static uint8_t actualizationModbusAddress;
	
	uint8_t structurSize = 0;
	uint8_t passButtonsCount;
	_buttonData * button;
	_lcdStateContext * lcdStateContext;
	_lcdButtonScene * sceneWork;	
	if(systemSceneFlag)
		sceneWork = &lcdButtonData.scene.system;
	else
		sceneWork = &lcdButtonData.scene.user;	
	
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_SCENE_ACTIVE_START:
			/**lcdStateContext->data[1] = 1;
 			 * lcdActuallContext->data
				[0]: numer aktywowanej sceny
				[1]: 0: nie wyswietla na nowo tla (aktualizacja wybranego rejestru); !0: tlo jest ponownie wyswietlane
				[2]: adres modbus aktualizowanego rejestru; wyświetlane są tylko te grafiki które odnoszą się do niego (bez tła, tekstów i zwykłych obrazków)
 			 */
			withBackgroundFlag = lcdActuallContext->data[1];
			//TODO: actualizationModbusAddress - nie jest to dobrze rozwiazane
			//actualizationModbusAddress = lcdActuallContext->data[2];
			
			lcdButtonData.scene.sceneReady = 0;
			lcdButtonData.passwordBuild = 0;
			if(systemSceneFlag == 0)
			{
				lcdButtonData.scene.activeScene = lcdActuallContext->data[0];
				lcdButtonData.modbusSpace[MODBUS_VOLATILE_ACTIVE_SCENE] = lcdButtonData.scene.activeScene;
			}
			writeSpecialModbus(MODBUS_READ_ONLY_PASS_DIGITS, 0);
			writeSpecialModbus(MODBUS_READ_ONLY_PASSWORD, 0);
			
			lcdActuallContext->backupCounter = 0;
			lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_INIT_DATA;
			break;
		
		case LCD_SUBSTATE_SCENE_ACTIVE_READ_INIT_DATA:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0, 79, systemSceneFlag);				
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_SCENE_ACTIVE_READ_INIT_DATA;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_INIT_DATA;
			}			
			break;
			
		case LCD_SUBSTATE_SCENE_ACTIVE_PREPARE_CONTROL_DATA:
			objectsCounter = 0;
			relAddress = 0x0000004F;
			sceneWork->firstBar = (_barData *)(sceneWork->workData);
			sceneWork->firstState = (_stateData *)(sceneWork->workData);
			sceneWork->firstNumber = (_numberData *)(sceneWork->workData);
			sceneWork->firstButton = (_buttonData *)(sceneWork->workData);
			lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA;
			break;
			
		case LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA:
			if(!uartLCD->isBusy())
			{
				if(lcdActuallContext->backupCounter > 0)
				{
					//jesli poprzednim razem nie odebrano danych, powtarzamy krok (1):
					objectsCounter--;
				}
				
				if(objectsCounter < sceneWork->barsCount)
				{
					structurSize = 13;
					lcdActuallContext->data[0] = LCD_OBJECT_BAR;
				}
				else if(objectsCounter < sceneWork->barsCount + sceneWork->statesCount)
				{
					structurSize = 10;
					lcdActuallContext->data[0] = LCD_OBJECT_STATE;
				}
				else if(objectsCounter < sceneWork->barsCount + sceneWork->statesCount + sceneWork->numbersCount)
				{
					structurSize = 13;
					lcdActuallContext->data[0] = LCD_OBJECT_NUMBER;
				}
				else if(objectsCounter < sceneWork->barsCount + sceneWork->statesCount + sceneWork->numbersCount + sceneWork->buttonsCount)
				{
					structurSize = 19;
					lcdActuallContext->data[0] = LCD_OBJECT_BUTTON;
				}
				
				if(lcdActuallContext->backupCounter > 0)
				{
					//jesli poprzednim razem nie odebrano danych, powtarzamy krok (2):
					relAddress -= structurSize;
				}
				
				if(objectsCounter < sceneWork->barsCount + sceneWork->statesCount + sceneWork->numbersCount + sceneWork->buttonsCount)
				{
					lcdSendSceneDataRead(relAddress, structurSize, systemSceneFlag);
 					relAddress += structurSize;
 					objectsCounter++;
					
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_CONTROL_DATA;
				}
				else
				{
					//koniec
					//wskazniki first... wskazuja na pierwszy element po wlasnych danych, nalezy wiec to przesunac:
					sceneWork->firstButton = (_buttonData *)sceneWork->firstNumber;
					sceneWork->firstNumber = (_numberData *)sceneWork->firstState;
					sceneWork->firstState = (_stateData *)sceneWork->firstBar;
					sceneWork->firstBar = (_barData *)sceneWork->workData;
										
					//wyswietlenie tla:
					if(systemSceneFlag || withBackgroundFlag)
					{
						lcdSendDisplayIcon(
							*(lcdActuallContext->data + LCD_WORK_CONTEXT_BACKGROUND_INDEX + 0),
							*((uint16_t *)(lcdActuallContext->data + LCD_WORK_CONTEXT_BACKGROUND_INDEX + 1)),
							*((uint16_t *)(lcdActuallContext->data + LCD_WORK_CONTEXT_BACKGROUND_INDEX + 3)),
							*((uint16_t *)(lcdActuallContext->data + LCD_WORK_CONTEXT_BACKGROUND_INDEX + 5)),
							*((uint16_t *)(lcdActuallContext->data + LCD_WORK_CONTEXT_BACKGROUND_INDEX + 7)),
							0,
							0
							);
							
						if(systemSceneFlag == 0)
						{
							objectsCounter = 0;
							relAddress = 0x00000A27;
						}
						else
						{
							//jesli scena serwisowa, to pierwsze 5 grafik dotyczy wizualizacji polaczenia z kontrolerem wago
							objectsCounter = 5;
							relAddress = 0x00000A27 + 5*14;
						}
							
						lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_IMAGE;
					}						
					else
					{
						lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_EXIT;	
					}
				}
			}				
			break;
					
	
		case LCD_SUBSTATE_SCENE_ACTIVE_READ_IMAGE:
			if(!uartLCD->isBusy())
			{	
				if(lcdActuallContext->backupCounter > 0)
				{
					//jesli poprzednim razem nie odebrano danych, powtarzamy krok:
					objectsCounter--;
					relAddress -= 14;
				}
				
				//wyswietlanie grafik luznych:
				if(objectsCounter < sceneWork->labelsCount)
				{
					lcdSendSceneDataRead(relAddress, 14, systemSceneFlag);
					relAddress += 14;
					objectsCounter++;
					
					lcdActuallContext->data[0] = LCD_OBJECT_IMAGE;
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_SCENE_ACTIVE_READ_IMAGE;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_IMAGE_GRAPHIC;
				}
				else
				{
					if(systemSceneFlag == 0)
					{
						objectsCounter = 0;
					}
					else
					{
						//jesli scena serwisowa, to pierwsze 4 tekstów dotyczy wizualizacji polaczenia z kontrolerem wago
						objectsCounter = 4;
						relAddress += 4 * 14;
					}
					lcdActuallContext->backupCounter = 0;
					lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_TEXT;
				}
			}
			break;
			
		case LCD_SUBSTATE_SCENE_ACTIVE_READ_TEXT:
			if(!uartLCD->isBusy())
			{	
				if(objectsCounter < sceneWork->textsCount)
				{	
					lcdSendSceneTextRead(objectsCounter, systemSceneFlag);
					
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_SCENE_ACTIVE_READ_TEXT;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_TEXT;
				}	
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_EXIT;
				}
			}
			break;
					
		case LCD_SUBSTATE_SCENE_ACTIVE_SEND_GRAPHIC:
			if(!uartLCD->isBusy())
			{
				lcdSendDisplayIcon(
					*(lcdActuallContext->data+0),
					*((uint16_t *)(lcdActuallContext->data+1)),
					*((uint16_t *)(lcdActuallContext->data+3)),
					*((uint16_t *)(lcdActuallContext->data+5)),
					*((uint16_t *)(lcdActuallContext->data+7)),
					*((uint16_t *)(lcdActuallContext->data+9)),
					*((uint16_t *)(lcdActuallContext->data+11)));
				lcdActuallContext->substate = lcdActuallContext->backupSubstate;				
			}
			break;
		
		case LCD_SUBSTATE_SCENE_ACTIVE_SEND_TEXT:
			if(!uartLCD->isBusy())
			{	
				lcdSendDisplayText(
					((uint16_t)lcdActuallContext->data[1]) << 8 | lcdActuallContext->data[0],				//x
					((uint16_t)lcdActuallContext->data[3]) << 8 | lcdActuallContext->data[2],				//y
					lcdActuallContext->data[4],																//library
					lcdActuallContext->data[5],																//font size
					((uint16_t)lcdActuallContext->data[7]) << 8 | lcdActuallContext->data[6],				//color
					(uint8_t *)(((uint16_t)lcdActuallContext->data[8]) << 8 | lcdActuallContext->data[8]),	//dataPnt
					sceneWork->textsLengths[objectsCounter],
					systemSceneFlag);
				objectsCounter++;
				lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_TEXT;
			}			
			break;
		
		case LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_TEXT:
		case LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_IMAGE_GRAPHIC:
		case LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_CONTROL_DATA:
		case LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_INIT_DATA:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_ERROR;
				}				
			}
			break;
				
		case LCD_SUBSTATE_SCENE_ACTIVE_EXIT:
			//kontrola przyciskow passButton, jesli sa, nalezy je przemieszac
			//przyciski to (jesli sa) znajduja sie na poczatku listy przyciskow
			button = sceneWork->firstButton;
			for(passButtonsCount = 0; passButtonsCount < sceneWork->buttonsCount; passButtonsCount++)
			{
				if(button->type != BUT_CODE_PASS_NUM)
					break;
				button++;
			}
			if(passButtonsCount)
			{
				uint16_t xTmp, yTmp;
				uint16_t dx1, dy1;	//szerokosc i wysokosc wylosowanego przycisku
				int randIndex;
				while(passButtonsCount>1)
				{
					randIndex = getRandomInt(passButtonsCount);
					xTmp = (sceneWork->firstButton + randIndex)->xLGR;
					yTmp = (sceneWork->firstButton + randIndex)->yLGR;
					dx1 = (sceneWork->firstButton + randIndex)->xPDR - (sceneWork->firstButton + randIndex)->xLGR;
					dy1 = (sceneWork->firstButton + randIndex)->yPDR - (sceneWork->firstButton + randIndex)->yLGR;
					(sceneWork->firstButton + randIndex)->xLGR = (sceneWork->firstButton + passButtonsCount - 1)->xLGR;
					(sceneWork->firstButton + randIndex)->yLGR = (sceneWork->firstButton + passButtonsCount - 1)->yLGR;
					(sceneWork->firstButton + randIndex)->xPDR = (sceneWork->firstButton + randIndex)->xLGR + dx1;
					(sceneWork->firstButton + randIndex)->yPDR = (sceneWork->firstButton + randIndex)->yLGR + dy1;
					(sceneWork->firstButton + passButtonsCount - 1)->xLGR = xTmp;
					(sceneWork->firstButton + passButtonsCount - 1)->yLGR = yTmp;
					(sceneWork->firstButton + passButtonsCount - 1)->xPDR = (sceneWork->firstButton + passButtonsCount - 1)->xLGR + dx1;
					(sceneWork->firstButton + passButtonsCount - 1)->yPDR = (sceneWork->firstButton + passButtonsCount - 1)->yLGR + dy1;
					passButtonsCount--;
				}
			}
		
		
			//wyswietlenie obiektow BAR:
			if(sceneWork->barsCount)
			{
				lcdStateContext = getLcdWorkQueueFree();
				if(lcdStateContext != NULL)
				{
					lcdStateContext->data[0] = 0;								//indeks obiektu od ktorego rozpoczac wyswietlanie
					lcdStateContext->data[1] = sceneWork->barsCount;			//ilosc obiektow do wyswietlenia
					if(systemSceneFlag)
						lcdStateContext->state = LCD_STATE_SERVICE_PRINT_BARS;
					else
						lcdStateContext->state = LCD_STATE_PRINT_BARS;
					lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
					enqueueLcdWork();
				}				
			}
			//wyswietlenie obiektow STATE:
			if(sceneWork->statesCount)
			{
				lcdStateContext = getLcdWorkQueueFree();
				if(lcdStateContext != NULL)
				{
					lcdStateContext->data[0] = 0;								//indeks obiektu od ktorego rozpoczac wyswietlanie
					lcdStateContext->data[1] = sceneWork->statesCount;			//ilosc obiektow do wyswietlenia
					if(systemSceneFlag)
						lcdStateContext->state = LCD_STATE_SERVICE_PRINT_STATES;
					else
						lcdStateContext->state = LCD_STATE_PRINT_STATES;
					lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
					enqueueLcdWork();
				}					
			}
			//wyswietlenie obiektow NUMBER:
			if(sceneWork->numbersCount)
			{
				lcdStateContext = getLcdWorkQueueFree();
				if(lcdStateContext != NULL)
				{
					lcdStateContext->data[0] = 0;								//indeks obiektu od ktorego rozpoczac wyswietlanie
					lcdStateContext->data[1] = sceneWork->numbersCount;			//ilosc obiektow do wyswietlenia
					if(systemSceneFlag)
						lcdStateContext->state = LCD_STATE_SERVICE_PRINT_NUMBERS;
					else
						lcdStateContext->state = LCD_STATE_PRINT_NUMBERS;
					lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
					enqueueLcdWork();
				}				
			}
			//wyswietlenie obiektow BUTTON:
			if(sceneWork->buttonsCount)
			{
				lcdStateContext = getLcdWorkQueueFree();
				if(lcdStateContext != NULL)	
				{
					lcdStateContext->data[0] = 0;								//indeks obiektu od ktorego rozpoczac wyswietlanie
					lcdStateContext->data[1] = sceneWork->buttonsCount;			//ilosc obiektow do wyswietlenia
					if(systemSceneFlag)
						lcdStateContext->state = LCD_STATE_SERVICE_PRINT_BUTTONS;
					else
						lcdStateContext->state = LCD_STATE_PRINT_BUTTONS;
					lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
					enqueueLcdWork();
				}
			}
			
			if(systemSceneFlag)
			{
				//wyswietlenie sceny uzytkownika
 				lcdStateContext = getLcdWorkQueueFree();
 				if(lcdStateContext != NULL)
 				{
 					lcdStateContext->data[0] = lcdButtonData.scene.activeScene;
					lcdStateContext->data[1] = 1;
 					lcdStateContext->state = LCD_STATE_SCENE_ACTIVE;
 					lcdStateContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_START;
 					enqueueLcdWork();
 				}

			}
			else
			{				
				//ustawienie flagi konca wyswietlania nowej sceny
				lcdStateContext = getLcdWorkQueueFree();
				if(lcdStateContext != NULL)
				{
					lcdStateContext->state = LCD_STATE_SCENE_ACTIVE_DONE;
					enqueueLcdWork();
				}
			}
			//zakonczenie zadania active scene:
			dequeueLcdWork();
			break;
			
		case LCD_SUBSTATE_SCENE_ACTIVE_ERROR:
			//ustawienie flagi konca wyswietlania nowej sceny
			lcdStateContext = getLcdWorkQueueFree();
			if(lcdStateContext != NULL)
			{
				lcdStateContext->state = LCD_STATE_SCENE_ACTIVE_DONE;
				enqueueLcdWork();
			}
			dequeueLcdWork();
			break;
			

			
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_IMAGE_SAVE_START;
			break;
	}
}

//////////////////////////////////////////////////////////////////////////
void lcdStatePrintObjects(_lcdObjectType objectType, uint8_t systemSceneFlag)
{
	/**
	 * lcdActuallContext->data:
	 * [0]: indeks obiektu od ktorego rozpoczac wyswietlanie
	 * [1]: ilosc obiektow do wyswietlenia
	 */
	static uint8_t objectsStartIndex;
	static uint8_t objectsCounter;
	static uint8_t objectsToPrint;
	
	_lcdButtonScene * sceneWork;
	if(systemSceneFlag)
		sceneWork = &lcdButtonData.scene.system;
	else
		sceneWork = &lcdButtonData.scene.user;
	
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_PRINT_OBJECTS_START:
			objectsStartIndex = lcdActuallContext->data[0];
			objectsToPrint = lcdActuallContext->data[1];
			objectsCounter = objectsStartIndex;
			lcdActuallContext->backupCounter = 0;
			if(objectType == LCD_OBJECT_NUMBER)
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_PREPARE_NUMBER;
			else
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_READ_GRAPHIC;
			break;
			
			
		case LCD_SUBSTATE_PRINT_OBJECTS_PREPARE_NUMBER:
			if(objectsCounter < objectsStartIndex + objectsToPrint)
			{				
				/*!
					Tablica pomocnicza przy wyswietlaniu obiektu NUMBER; kazdy element tablicy odpowiada kolejnej cyfrze/znaku do wyświetlenia
					0-9	: grafika cyfry 0-9
					10	: grafika czyszczaca
					11	: grafika znaku minus
					12	: grafika znaku plus
					13	: brak jakiejkolwiek grafiki
				*/
				_numberData * numData = sceneWork->firstNumber + objectsCounter;
				uint8_t digitCountMax = (numData->status & 0x03) + 1;
				int16_t val = (int16_t)readModbus(numData->mbWord);
					
				//zakresy:
				uint8_t notChangeFlag = (numData->status & (1<<6)) || numData->mbWord < MODBUS_READ_ONLY_END;
				if(!notChangeFlag)
				{
					if(numData->value < numData->min)
						numData->value = numData->min;
					else if(numData->value > numData->max)
						numData->value = numData->max;
					
					if(numData->value != val)
					{
						writeModbus(numData->mbWord, numData->value);
						val = numData->value;
					}						
						
					if(digitCountMax == 4)
					{
						if(val < -9999)
							val = -9999;
						else if(val > 9999)
							val = 9999;
					}
					else if(digitCountMax == 3)
					{
						if(val < -999)
							val = -999;
						else if(val > 999)
							val = 999;
					}
					else if(digitCountMax == 2)
					{
						if(val < -99)
							val = -99;
						else if(val > 99)
							val = 99;
					}
					else
					{// if(digitCountMax == 1)
						if(val < -9)
							val = -9;
						else if(val > 9)
							val = 9;
					}
				}
	
				if((numData->status & 1<<5) == 0)
					digitCountMax += 1;
						
				for(int i=0; i<digitCountMax; i++)
					lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+i] = 10;	//pusta grafika (czyszczaca obraz)
				for(int i=digitCountMax; i<5; i++)
					lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+i] = 13;	//brak jakiejkolwiek grafiki
		
				if(val >= numData->min && val <= numData->max)	//specjalna kombinacja, dla ktorej na kazdym miejscu wyswietlany jest pusty znak
				{					
					int16_t valCopy = val;
					uint8_t index=0;
					uint8_t addDigits = 0;
					if(val<0)
						val *= -1;
					if(val>999)
					{
						lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+3] = val / 1000;
						val -= lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+3]*1000;
						index++;
						addDigits = 1;
					}
					if(addDigits || val>99)
					{
						lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+2] = val / 100;
						val -= lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+2]*100;
						index++;
						addDigits = 1;
					}
					if(addDigits || val>9)
					{
						lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+1] = val / 10;
						val -= lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+1]*10;
						index++;
						addDigits = 1;
					}
					lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+0] = val;
					index++;
					val = valCopy;
					
					if(numData->status & 1<<4)
					{
						for(; index < (numData->status & 0x03)+1; index++)
							lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+index] = 0;
					}
					
					if((numData->status & 1<<5) == 0)
					{
						if(val<0)
						{
							lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+index] = 11;
							index++;
						}
						else if(numData->status & 1<<3)
						{
							lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX+index] = 12;
							index++;
						}
					}
				}
										
				// ilosc grafik do wyswietlenia:
				lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_DIGIT_COUNT_INDEX] = digitCountMax;
				// numer grafiki wyswietlanej w kolejnym kroku
				lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_DIGIT_ACTUAL_INDEX] = 0;
				// wspolrzedne X
				*((uint16_t *)(lcdActuallContext->data + LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX)) = numData->x;
				*((uint16_t *)(lcdActuallContext->data + LCD_WORK_CONTEXT_NUMBER_DELTA_X_PREVIOUS_INDEX)) = 0;
				lcdActuallContext->backupCounter = 0;
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_READ_GRAPHIC;
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_EXIT;
			}
			break;

		case LCD_SUBSTATE_PRINT_OBJECTS_READ_GRAPHIC:
			//wyswietlanie grafik obiektow:
			if(objectsCounter < objectsStartIndex + objectsToPrint)
			{
				if(!uartLCD->isBusy())
				{
					if(objectType == LCD_OBJECT_BAR)
					{
						_barData * bar = sceneWork->firstBar + objectsCounter;						
						
						bar->value = readModbus(bar->mbWord);
						if(bar->mbWord >= MODBUS_READ_ONLY_END)
						{
							uint8_t refreshFlag = 0;
							if(bar->value < bar->min)
							{
								bar->value = bar->min;
								refreshFlag = 1;
							}
							else if(bar->value > bar->max)
							{
								bar->value = bar->max;
								refreshFlag = 1;
							}		
							if(refreshFlag)
							{
								writeModbus(bar->mbWord, bar->value);	
							}													
						}
						lcdSendBarGraphicRead(bar->graphId, bar->value-bar->min, systemSceneFlag);
					}
					else if(objectType == LCD_OBJECT_STATE)
					{						
						_stateData * state = sceneWork->firstState + objectsCounter;
						state->value = ((readModbus(state->mbWord) & (1<<(state->mbBit))) != 0);
						lcdSendStateGraphicRead(state->graphId, state->value, systemSceneFlag);
					}
					else if(objectType == LCD_OBJECT_NUMBER)
					{						
						_numberData * number = sceneWork->firstNumber + objectsCounter;
						uint8_t maxIndex = lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_DIGIT_COUNT_INDEX];
						uint8_t actualIndex = lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_DIGIT_ACTUAL_INDEX];					
						uint8_t digit;
						if(number->status & 1<<2)
							digit = lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX + maxIndex - actualIndex - 1];						
						else
							digit = lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX + actualIndex];

						number->value = readModbus(number->mbWord);						
						if(!(number->status & (1<<6)) && number->mbWord >= MODBUS_READ_ONLY_END)
						{
							uint8_t refreshFlag = 0;
							if(number->value < number->min)
							{
								number->value = number->min;
								refreshFlag = 1;
							}							
							else if(number->value > number->max)
							{
								number->value = number->max;
								refreshFlag = 1;
							}
							if(refreshFlag)
							{
								writeModbus(number->mbWord, number->value);
							}
						}
						lcdSendNumberGraphicRead(number->graphId, digit, systemSceneFlag);
					}
					else if(objectType == LCD_OBJECT_BUTTON)
					{
						_buttonData * button = sceneWork->firstButton + objectsCounter;
						
						uint8_t buttonGraphValue = 0;
						if(button->type == BUT_CODE_ROCKER)
						{
							uint16_t data = readModbus(button->data[0]);
							if((int16_t)data < *((int16_t *)(button->data + 3)))
							{
								writeModbus(button->data[0], *((int16_t *)(button->data + 3)));
							}								
							if((int16_t)data > *((int16_t *)(button->data + 5)))
							{
								writeModbus(button->data[0], *((int16_t *)(button->data + 5)));
							}
						}
						else if(button->type == BUT_CODE_INDEP ||
								button->type == BUT_CODE_INDEP_OFF ||
								button->type == BUT_CODE_BLIND)
						{
							button->value = readModbus(button->data[0]) & ((uint16_t)1<<button->data[1]);
						}
						else if(button->type == BUT_CODE_VALUE)
						{
							button->value = readModbus(button->data[0]);
							uint16_t tmpVal = button->data[1] | (((uint16_t)(button->data[2]))<<8);
							button->value = button->value == tmpVal;
						}
						if(button->value > 0)
							buttonGraphValue = 1;
						else
							buttonGraphValue = 0;	
						
						
						lcdSendSceneButtonGraphicRead(objectsCounter, buttonGraphValue, systemSceneFlag);
					}
					
					lcdActuallContext->data[0] = objectType;
					lcdActuallContext->data[1] = objectsCounter;					
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_PRINT_OBJECTS_READ_GRAPHIC;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_WATCHDOG;
				}					
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_EXIT;
			}
			break;

		case LCD_SUBSTATE_PRINT_OBJECTS_WATCHDOG:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_NEXT;
				}
			}
			break;

		case LCD_SUBSTATE_PRINT_OBJECTS_EXEC_GRAPHIC:
			if(!uartLCD->isBusy())
			{
				lcdSendDisplayIcon(
					*(lcdActuallContext->data+0),
					*((uint16_t *)(lcdActuallContext->data+1)),
					*((uint16_t *)(lcdActuallContext->data+3)),
					*((uint16_t *)(lcdActuallContext->data+5)),
					*((uint16_t *)(lcdActuallContext->data+7)),
					*((uint16_t *)(lcdActuallContext->data+9)),
					*((uint16_t *)(lcdActuallContext->data+11)));
				
				if(objectType == LCD_OBJECT_BAR || objectType == LCD_OBJECT_STATE || objectType == LCD_OBJECT_NUMBER)
					lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_NEXT;
				else if(objectType == LCD_OBJECT_BUTTON)
					lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_READ_TEXT;
			}
			break;

		case LCD_SUBSTATE_PRINT_OBJECTS_READ_TEXT:
			if(sceneWork->textsLengths[sceneWork->textsCount + objectsCounter] > 0)
			{
				if(!uartLCD->isBusy())
				{
					lcdSendSceneButtonTextRead(objectsCounter, systemSceneFlag);
					lcdActuallContext->backupSubstate = LCD_SUBSTATE_PRINT_OBJECTS_READ_TEXT;
					setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
					lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_WATCHDOG;	
				}
			}
			else
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_NEXT;
			break;
			

		case LCD_SUBSTATE_PRINT_OBJECTS_EXEC_TEXT:
			if(!uartLCD->isBusy())
			{				
				//tylko przyciski:
				lcdSendDisplayText(
					((uint16_t)lcdActuallContext->data[1]) << 8 | lcdActuallContext->data[0],				//x
					((uint16_t)lcdActuallContext->data[3]) << 8 | lcdActuallContext->data[2],				//y
					lcdActuallContext->data[4],																//library
					lcdActuallContext->data[5],																//font size
					((uint16_t)lcdActuallContext->data[7]) << 8 | lcdActuallContext->data[6],				//color
					(uint8_t *)(((uint16_t)lcdActuallContext->data[8]) << 8 | lcdActuallContext->data[8]),	//dataPnt
					sceneWork->textsLengths[sceneWork->textsCount + objectsCounter],
					systemSceneFlag);
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_NEXT;
			}
			break;
			
		case LCD_SUBSTATE_PRINT_OBJECTS_NEXT:		
			if(objectType == LCD_OBJECT_NUMBER)
			{
				if(lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_DIGIT_ACTUAL_INDEX] < lcdActuallContext->data[LCD_WORK_CONTEXT_NUMBER_DIGIT_COUNT_INDEX])
				{
					lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_READ_GRAPHIC;
				}
				else
				{
 					objectsCounter ++;
					lcdActuallContext->data[0] = objectsCounter ;
 					lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_PREPARE_NUMBER;
				}
			}
			else
			{
				objectsCounter++;
				lcdActuallContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_READ_GRAPHIC;
			}
			break;

		default:
		case LCD_SUBSTATE_PRINT_OBJECTS_EXIT:
			dequeueLcdWork();
			break;
	}
}
//////////////////////////////////////////////////////////////////////////
 void lcdStateClick(void)
 {
	static int objectsCounter;
	static uint8_t lightOnFlag;		//flaga okresla, ze dane wcisniecie sluzy wlaczeniu podswietlania, nie jest analizowane.
	uint16_t x, y;
	_buttonData * buttonData;
	_lcdStateContext * lcdStateContext;
	
 	switch(lcdActuallContext->substate)
 	{
 		case LCD_SUBSTATE_TOUCH_START:
 			/**
 			 * lcdActuallContext->data:
 			 * [0]	- 0: koniec klikniecia; !0: klikniecie
 			 * [1:2]- wsp X klikniecia
 			 * [3:4]- wsp Y klikniecia
 			 */
 			if(lcdActuallContext->data[0] == CLICK_NONE)
			{
				if(lightOnFlag)
				{
					lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_EXIT;
					lightOnFlag = 0;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_OFF;
				}
			}
 			else
			{
				if(!lightOnFlag)
				{
					lightOnFlag = lcdBacklightService(LCD_BACKLIGHT_ON_CMD);
				}					
				if(lightOnFlag)
				{
					lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_EXIT;
				}
				else
				{
					if(lcdActuallContext->data[0] == CLICK_SHORT)
						lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_ON;
					else //if(lcdActuallContext->data[0] == CLICK_LONG)
						lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_PRESS_ON;	
				}
			}				 
			
 			break;

 		case LCD_SUBSTATE_TOUCH_OFF:
			if(lcdButtonData.button.clickedState == CLICK_SHORT || lcdButtonData.button.clickedState == CLICK_LONG)
			{
				buttonData = lcdButtonData.scene.user.firstButton + lcdButtonData.button.clickedButtonIndex;
				
				if(buttonData->type == BUT_CODE_EMPTY ||
					buttonData->type == BUT_CODE_PASS_NUM ||
					buttonData->type == BUT_CODE_PASS_CLR ||
					buttonData->type == BUT_CODE_PASS_SET)
				{
					//zwolnienie przycisku:
					buttonData->value = 0;
					lcdStateContext = getLcdWorkQueueFree();
					if(lcdStateContext != NULL)
					{
						lcdStateContext->data[0] = lcdButtonData.button.clickedButtonIndex;				//indeks przycisku od ktorego rozpoczac wyswietlanie
						lcdStateContext->data[1] = 1;							//ilosc przyciskow do wyswietlenis
						lcdStateContext->state = LCD_STATE_PRINT_BUTTONS;
						lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
						enqueueLcdWork();
					}
				}					
				else if(buttonData->type == BUT_CODE_SCENE)
				{
					//zwolnienie przycisku:
					buttonData->value = 0;
					lcdStateContext = getLcdWorkQueueFree();
					if(lcdStateContext != NULL)
					{
						lcdStateContext->data[0] = lcdButtonData.button.clickedButtonIndex;				//indeks przycisku od ktorego rozpoczac wyswietlanie
						lcdStateContext->data[1] = 1;							//ilosc przyciskow do wyswietlenis
						lcdStateContext->state = LCD_STATE_PRINT_BUTTONS;
						lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
						enqueueLcdWork();
					}
					//zaladowanie nowej sceny:
					lcdStateContext = getLcdWorkQueueFree();
					if(lcdStateContext != NULL)
					{
						lcdStateContext->data[0] = buttonData->data[0];
						lcdStateContext->data[1] = 1;
						lcdStateContext->state = LCD_STATE_SCENE_ACTIVE;
						lcdStateContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_START;
						enqueueLcdWork();	
					}
					//jesli nowa scena jest inna niz dotychczas, to nalezy przerwac dzialajace timery przyciskow:
					if(buttonData->data[0] != lcdButtonData.scene.activeScene)
					{
						lcdButtonData.scene.sceneReady = 0;
						_buttonData * buttonCheckScene = lcdButtonData.scene.user.firstButton;
						for(int i=0; i<lcdButtonData.scene.user.buttonsCount; i++)
						{
							if(buttonCheckScene->type == BUT_CODE_INDEP_OFF ||
								buttonCheckScene->type == BUT_CODE_BLIND)
							{
								uint16_t data = readModbus(buttonCheckScene->data[0]) & ~((uint16_t)1<<buttonCheckScene->data[1]);
								writeModbus(buttonCheckScene->data[0],  data);
							}
							buttonCheckScene++;							
						}
					}
				}
				else if(buttonData->type == BUT_CODE_BLIND && lcdButtonData.button.clickedState == CLICK_LONG)
				{
					uint16_t data = readModbus(buttonData->data[0]) & (~(uint16_t)1<<buttonData->data[1]);
					writeModbus(buttonData->data[0], data);
				}
				else if(buttonData->type == BUT_CODE_ROCKER)
				{
					//zwolnienie przycisku:
					buttonData->value = 0;
					lcdStateContext = getLcdWorkQueueFree();
					if(lcdStateContext != NULL)
					{
						lcdStateContext->data[0] = lcdButtonData.button.clickedButtonIndex;				//indeks przycisku od ktorego rozpoczac wyswietlanie
						lcdStateContext->data[1] = 1;							//ilosc przyciskow do wyswietlenis
						lcdStateContext->state = LCD_STATE_PRINT_BUTTONS;
						lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
						enqueueLcdWork();
					}
				}
			}
			lcdButtonData.button.clickedState = CLICK_NONE;
 			lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_EXIT;
 			break;

 		case LCD_SUBSTATE_TOUCH_ON:
 			x = *((uint16_t *)(lcdActuallContext->data+1));
 			y = *((uint16_t *)(lcdActuallContext->data+3));
 			if(y > lcdButtonData.scene.logoHeight)
 			{
 				y -= lcdButtonData.scene.logoHeight;
				buttonData = lcdButtonData.scene.user.firstButton;
				for(int i=0; i < lcdButtonData.scene.user.buttonsCount; i++)
				{
					if(x >= buttonData->xLGR && x <= buttonData->xPDR && y >= buttonData->yLGR && y <= buttonData->yPDR)
					{
						lcdButtonData.button.clickedState = CLICK_SHORT;
						lcdSendTouchBuzzorOn(lcdButtonData.buzzerTime);
						lcdButtonData.button.clickedButtonIndex = i;
						objectsCounter = i;
						
						if(buttonData->type == BUT_CODE_SCENE ||
							buttonData->type == BUT_CODE_EMPTY)
						{
							buttonData->value = 1;		//wcisniete
						}
						else if(buttonData->type == BUT_CODE_PASS_CLR)
						{
							buttonData->value = 1;		//wcisniete
							writeSpecialModbus(MODBUS_READ_ONLY_PASS_DIGITS, 0);
							writeSpecialModbus(MODBUS_READ_ONLY_PASSWORD, 0);
							lcdButtonData.passwordBuild = 0;
						}
						else if(buttonData->type == BUT_CODE_PASS_SET)
						{
							buttonData->value = 1;		//wcisniete
							writeSpecialModbus(MODBUS_READ_ONLY_PASSWORD, lcdButtonData.passwordBuild);
							lcdButtonData.passwordBuild = 0;
							writeSpecialModbus(MODBUS_READ_ONLY_PASS_DIGITS, 0);
						}
						else if(buttonData->type == BUT_CODE_PASS_NUM)
						{
							buttonData->value = 1;		//wcisniete
							uint16_t passDigits = readModbus(MODBUS_READ_ONLY_PASS_DIGITS);
							if(passDigits < 4)
							{
								writeSpecialModbus(MODBUS_READ_ONLY_PASS_DIGITS, passDigits+1);
								lcdButtonData.passwordBuild = lcdButtonData.passwordBuild*10 + (uint16_t)buttonData->data[0];
							}								
						}
						else if(buttonData->type == BUT_CODE_INDEP)
						{
							uint16_t data = readModbus(buttonData->data[0]) ^ ((uint16_t)1<<buttonData->data[1]);
							writeModbus(buttonData->data[0], data);
						}
						else if(buttonData->type == BUT_CODE_INDEP_OFF)
						{
							uint16_t data = readModbus(buttonData->data[0]) ^ ((uint16_t)1<<buttonData->data[1]);
							writeModbus(buttonData->data[0], data);
							if(data & ((uint16_t)1<<buttonData->data[1]))
							{
								//100ms
								buttonData->timedown = (uint16_t)(buttonData->data[2]) | ((uint16_t)(buttonData->data[3]) << 8);
							}
						}
						else if(buttonData->type == BUT_CODE_VALUE)
						{
							uint16_t data = (uint16_t)(buttonData->data[1]) | (((uint16_t)(buttonData->data[2])) << 8);
							writeModbus(buttonData->data[0], data);
						}
						else if(buttonData->type == BUT_CODE_BLIND)
						{
							uint16_t data = readModbus(buttonData->data[0]) ^ ((uint16_t)1<<buttonData->data[1]);
							writeModbus(buttonData->data[0], data);
							if(data & ((uint16_t)1<<buttonData->data[1]))
							{
								//100ms
								buttonData->timedown = (uint16_t)(buttonData->data[4]) | ((uint16_t)(buttonData->data[5]) << 8);
								data = readModbus(buttonData->data[2]) & (~((uint16_t)1<<buttonData->data[3]));
								writeModbus(buttonData->data[2], data);
							}
						}
						else if(buttonData->type == BUT_CODE_ROCKER)
						{
							buttonData->value = 1;		//wcisniete
							int16_t data = (int16_t)readModbus(buttonData->data[0]) + (int8_t)buttonData->data[1];
							if(data < *((int16_t *)(buttonData->data + 3)))
								data = *((int16_t *)(buttonData->data + 3));
							if(data > *((int16_t *)(buttonData->data + 5)))
								data = *((int16_t *)(buttonData->data + 5));
							writeModbus(buttonData->data[0], data);
							//20ms
							buttonData->timedown = buttonData->data[7];
							buttonData->timedown2 = buttonData->data[8];
							
						}
						
						lcdStateContext = getLcdWorkQueueFree();
						if(lcdStateContext != NULL)
						{
							lcdStateContext->data[0] = objectsCounter;				//indeks przycisku od ktorego rozpoczac wyswietlanie
							lcdStateContext->data[1] = 1;							//ilosc przyciskow do wyswietlenis
							lcdStateContext->state = LCD_STATE_PRINT_BUTTONS;
							lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
							enqueueLcdWork();
						}
						break;
					}
					buttonData++;
				}
 			}
 			lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_EXIT;
 			break;

		case LCD_SUBSTATE_TOUCH_PRESS_OFF:
			if(lcdButtonData.button.clickedState == CLICK_LONG)
			{
				buttonData = lcdButtonData.scene.user.firstButton + lcdButtonData.button.clickedButtonIndex;
				if(buttonData->type == BUT_CODE_BLIND)
				{
					uint16_t data = readModbus(buttonData->data[0]) & (~(uint16_t)1<<buttonData->data[1]);
					writeModbus(buttonData->data[0], data);
					//zwolnienie przycisku:
					lcdStateContext = getLcdWorkQueueFree();
					if(lcdStateContext != NULL)
					{
						lcdStateContext->data[0] = lcdButtonData.button.clickedButtonIndex;				//indeks przycisku od ktorego rozpoczac wyswietlanie
						lcdStateContext->data[1] = 1;							//ilosc przyciskow do wyswietlenis
						lcdStateContext->state = LCD_STATE_PRINT_BUTTONS;
						lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
						enqueueLcdWork();
					}
				}
			}
			lcdButtonData.button.clickedState = CLICK_NONE;
 			lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_EXIT;
			break;
		

		case LCD_SUBSTATE_TOUCH_PRESS_ON:
			lcdButtonData.button.clickedState = CLICK_LONG;
			buttonData = lcdButtonData.scene.user.firstButton + lcdButtonData.button.clickedButtonIndex;
			if(buttonData->type == BUT_CODE_BLIND)
			{
				buttonData->timedown = 0;
			}
 			lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_EXIT; 
			break;

 		case LCD_SUBSTATE_TOUCH_EXIT:
 			dequeueLcdWork();
 			break;

 		default:
 			lcdActuallContext->substate = LCD_SUBSTATE_TOUCH_START;
 	}

 }
 //////////////////////////////////////////////////////////////////////////
 void lcdBrightness(void)
 {
	 switch(lcdActuallContext->substate)
	 {
		case LCD_SUBSTATE_BRIGHTNESS_JOB_START:
		default:
			 if(!uartLCD->isBusy())
			 {
				 lcdSendBrightness(lcdActuallContext->data[0]);
				 dequeueLcdWork();
			 }
			 break;
	 }	 
 }
//////////////////////////////////////////////////////////////////////////
void lcdInitWakeUp(void)
{
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_INIT_WAKE_UP_START:
			lcdButtonData.sceneSwitchFlag = 0;
			lcdButtonData.lightSceneSwitchFlag = 0;
			lcdButtonData.newValuesCount = 0;
			lcdActuallContext->backupCounter = 0;
			lcdActuallContext->substate = LCD_SUBSTATE_INIT_WAKE_UP_SEND_CMD;
			break;
			
		case  LCD_SUBSTATE_INIT_WAKE_UP_SEND_CMD:
			if(!uartLCD->isBusy())
			{
				lcdSendDataRead((uint32_t)COMMON_FUNCTIONS_PAGE * 0x020000 | 0, 54);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_INIT_WAKE_UP_SEND_CMD;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_INIT_WAKE_UP_WATCHDOG;
			}
			break;
			
		case LCD_SUBSTATE_INIT_WAKE_UP_WATCHDOG:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_INIT_WAKE_UP_END;
				}
			}
			break;
		case LCD_SUBSTATE_INIT_WAKE_UP_END:
			dequeueLcdWork();
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_INIT_WAKE_UP_START;
			break;
	}		
}
//////////////////////////////////////////////////////////////////////////
void lcdInitPid(void)
{
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_INIT_PID_START:
			lcdButtonData.pidCount = 0;
			lcdActuallContext->backupCounter = 0;
			lcdActuallContext->substate = LCD_SUBSTATE_INIT_PID_SEND_CMD;
			break;
		
		case LCD_SUBSTATE_INIT_PID_SEND_CMD:
			if(!uartLCD->isBusy())
			{
				lcdSendDataRead((uint32_t)COMMON_FUNCTIONS_PAGE * 0x020000 | 54, 64);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_INIT_PID_SEND_CMD;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_INIT_PID_WATCHDOG;
			}
			break;
		
		case LCD_SUBSTATE_INIT_PID_WATCHDOG:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_INIT_PID_END;
				}
			}
		break;
			case LCD_SUBSTATE_INIT_PID_END:
			dequeueLcdWork();
			break;
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_INIT_PID_START;
			break;
	}
}

void lcdClearNotuseProjekt(void)
{
	static uint8_t sceneToClear;
	int lenght;
	
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_START:
		sceneToClear = lcdButtonData.config.lastUserSceneSave + 1;
		lcdActuallContext->backupCounter = 0;
		lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_CMD;
		break;
		
		case LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_CMD:
		if(!uartLCD->isBusy())
		{
			setTimer1Sec(TMR_1SEC_HI_PRIO_DELAY, TMR_1SEC_HI_PRIO_DELAY_TIMEOUT);
			lenght = 0;
			uartLCD->bufferTx[lenght++] = LCD_FRAME_START;
			uartLCD->bufferTx[lenght++] = 0x90;
			uartLCD->bufferTx[lenght++] = 0x55;
			uartLCD->bufferTx[lenght++] = 0xAA;
			uartLCD->bufferTx[lenght++] = 0x5A;
			uartLCD->bufferTx[lenght++] = 0xA5;
			uartLCD->bufferTx[lenght++] = 0x00;
			uartLCD->bufferTx[lenght++] = (sceneToClear << 1);
			uartLCD->bufferTx[lenght++] = 0x00;
			uartLCD->bufferTx[lenght++] = 0x00;
			uartLCD->bufferTx[lenght++] = 0xFF;		//dane
			uartLCD->bufferTx[lenght++] = LCD_FRAME_END_1;
			uartLCD->bufferTx[lenght++] = LCD_FRAME_END_2;
			uartLCD->bufferTx[lenght++] = LCD_FRAME_END_3;
			uartLCD->bufferTx[lenght++] = LCD_FRAME_END_4;
			uartLCD->startTransmit(lenght);
			lcdActuallContext->backupSubstate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_CMD;
			setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
			lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_WATCHDOG;
		}
		break;
		
		case LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_WATCHDOG:
		if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
		{
			if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
			{
				lcdActuallContext->backupCounter++;
				lcdActuallContext->substate = lcdActuallContext->backupSubstate;
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_END;
			}
		}
		break;
		
		case LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_NEXT:
			sceneToClear++;
			if(sceneToClear < SCENES_COUNT)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_CMD;
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_END;
			}
			break;
		
		case LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_END:
			lcdButtonData.config.newUserConfigFlag = 0;
			dequeueLcdWork();
			break;
			
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_START;
			break;
	}
}

//////////////////////////////////////////////////////////////////////////
/*!
	@brief
		Inicjalizacja rejestrow modbus na podstawie danych zawartych w wyswietlaczu
	@details
		Funkcja przechodzi przez wszystkie sceny (lacznie z serwisowa)
		i na podstawie wlasciwosci zawartych w nich elementow weryfikuje poprawnosc
		danych w modbus (zmienia ich wartosc jesli wykraczaja poza zakres)
*/
void lcdValidModbusData(void)
{
	/*
		Pseudokod:
		1. Pobierz czesc A pierwszej sceny - ilosc poszczegolnych elementow dla danej sceny
		2. Dla kazdego obiektu BAR pobierz jego dane sterujace i ustal wartosc powiazanego z nim rejsetru
		3. Dla kazdego obiektu NUMBER pobierz jego dane sterujace i ustal wartosc powiazanego z nim rejsetru
		3. Dla kazdego obiektu BUTTON pobierz jego dane sterujace i dla przycisku typu
			ROCKER UP oraz ROCKER DOWN ustal wartosc powiazanego z nim rejsetru
	*/
	
	static int sceneIndexBefore = 0;
	
	static int actualBarIndex = 0;
	static int allBarSize = 0;
	
	static int allStateSize = 0;
	
	static int actualNumberIndex = 0;
	static int allNumberSize = 0;
	
	static int actualButtonIndex = 0;
	static int allButtonSize = 0;
	
	int16_t regValue, regMin, regMax;
	
	switch(lcdActuallContext->substate)
	{
		case LCD_SUBSTATE_VALID_MODBUS_START:
			sceneIndexBefore = lcdButtonData.scene.activeScene;
			lcdButtonData.scene.activeScene = 0;
			lcdActuallContext->backupCounter = 0;
			lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA;
			break;
		
		case LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead(0, 79, 0);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_INIT_DATA;
			}
			break;
		
		case LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_INIT_DATA:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_ERR;
				}
			}
			break;
			
		case LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA_OK:
			allBarSize = lcdActuallContext->data[0];
			allStateSize = lcdActuallContext->data[1];
			allNumberSize = lcdActuallContext->data[2];
			allButtonSize = lcdActuallContext->data[3];
			actualBarIndex = 0;
			actualNumberIndex = 0;
			actualButtonIndex = 0;
			
			if(allBarSize > actualBarIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA;	
			}
			else if(allNumberSize > actualNumberIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA;
			}
			else if(allButtonSize > actualButtonIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA;
			}else if(lcdButtonData.scene.activeScene < SCENES_COUNT - 1)
			{
				lcdButtonData.scene.activeScene++;
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA;				
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_END;
			}
			break;
		
		//obiekty BAR:
		case LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead((uint32_t)0x4F + actualBarIndex * 13, 13, 0);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BAR_CONTROL_DATA;
			}
			break;
			
		case LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BAR_CONTROL_DATA:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_ERR;
				}
			}
			break;
			
		case LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA_OK:
			regValue = readModbus(lcdActuallContext->data[11]);
			regMin =  *((int16_t *)(&lcdActuallContext->data[5]));
			regMax =  *((int16_t *)(&lcdActuallContext->data[7]));
			if(regValue < regMin)
				regValue = regMin;
			else if(regValue > regMax)
				regValue = regMax;
			writeModbus(lcdActuallContext->data[11], regValue);
			actualBarIndex++;
			if(allBarSize > actualBarIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA;
			}
			else if(allNumberSize > actualNumberIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA;
			}
			else if(allButtonSize > actualButtonIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA;
			}else if(lcdButtonData.scene.activeScene < SCENES_COUNT - 1)
			{
				lcdButtonData.scene.activeScene++;
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA;
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_END;
			}
			break;
			
		//obiekty NUMBER:
		case LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead((uint32_t)0x4F + allBarSize * 13 + allStateSize * 10 + actualNumberIndex * 13, 13, 0);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_NUMBER_CONTROL_DATA;
			}
			break;
		
		case LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_NUMBER_CONTROL_DATA:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_ERR;
				}
			}
			break;
		
		case LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA_OK:
			regValue = readModbus(lcdActuallContext->data[11]);
			regMin =  *((int16_t *)(&lcdActuallContext->data[5]));
			regMax =  *((int16_t *)(&lcdActuallContext->data[7]));
			if(regValue < regMin)
				regValue = regMin;
			else if(regValue > regMax)
				regValue = regMax;
			writeModbus(lcdActuallContext->data[11], regValue);
			actualNumberIndex++;
			if(allNumberSize > actualNumberIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA;
			}
			else if(allButtonSize > actualButtonIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA;
			}else if(lcdButtonData.scene.activeScene < SCENES_COUNT - 1)
			{
				lcdButtonData.scene.activeScene++;
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA;
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_END;
			}
			break;
			
		//obiekty BUTTON:
		case LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA:
			if(!uartLCD->isBusy())
			{
				lcdSendSceneDataRead((uint32_t)0x4F + allBarSize * 13 + allStateSize * 10 + allNumberSize * 13 + actualButtonIndex * 19, 19, 0);
				lcdActuallContext->backupSubstate = LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA;
				setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP);
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BUTTON_CONTROL_DATA;
			}
			break;
		
		case LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BUTTON_CONTROL_DATA:
			if(!getTimerShort(TMR_SHORT_LCD_NO_RESP))
			{
				if(lcdActuallContext->backupCounter < MAX_REQUEST_COUNT)
				{
					lcdActuallContext->backupCounter++;
					lcdActuallContext->substate = lcdActuallContext->backupSubstate;
				}
				else
				{
					lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_ERR;
				}
			}
			break;
		
		case LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA_OK:
			if(lcdActuallContext->data[8] == BUT_CODE_ROCKER)
			{
				regValue = readModbus(lcdActuallContext->data[9]);
				regMin =  *((int16_t *)(&lcdActuallContext->data[12]));
				regMax =  *((int16_t *)(&lcdActuallContext->data[14]));
				if(regValue < regMin)
					regValue = regMin;
				else if(regValue > regMax)
					regValue = regMax;
				writeModbus(lcdActuallContext->data[9], regValue);
			}
						
			actualButtonIndex++;
			if(allButtonSize > actualButtonIndex)
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA;
			}else if(lcdButtonData.scene.activeScene < SCENES_COUNT - 1)
			{
				lcdButtonData.scene.activeScene++;
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA;
			}
			else
			{
				lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_END;
			}
			break;
						
		case LCD_SUBSTATE_VALID_MODBUS_END:
			lcdButtonData.scene.activeScene = sceneIndexBefore;
			dequeueLcdWork();
			break;
		
		case LCD_SUBSTATE_VALID_MODBUS_ERR:
			lcdButtonData.scene.activeScene = sceneIndexBefore;
			dequeueLcdWork();
			break;
			
		
		default:
			lcdActuallContext->substate = LCD_SUBSTATE_VALID_MODBUS_START;
			break;
	}
	
}


//////////////////////////////////////////////////////////////////////////
/**************************************************************************
Funkcje zewnetrzne:
**************************************************************************/
//////////////////////////////////////////////////////////////////////////
void lcdTouchService(void)
{
	if(lcdButtonData.button.clickedState != CLICK_EMPTY && lcdButtonData.button.clickedState != CLICK_NONE)
	{
		if(!getTimerShort(TMR_SHORT_LCD_CLICK))
		{
			_lcdStateContext * lcdContext = getLcdWorkQueueFree();
			if(lcdContext != NULL)
			{
				lcdContext->data[0] = 0;	//0: koniec klikniecia
				lcdContext->state = LCD_STATE_TOUCH;
				lcdContext->substate = LCD_SUBSTATE_TOUCH_START;
				enqueueLcdWork();
			}
		}
	}
}
//////////////////////////////////////////////////////////////////////////
uint8_t lcdBacklightService(_lcdBacklightCmd cmd)
{
	static _lcdBacklightState brightState = LCD_BACKLIGHT_TO_LIGHT_STATE;

	uint8_t result = 0;
	uint16_t darkerTime;
	
	//LCD_BACKLIGHT_REFRESH (odswierzenie wyswietlanej jasnosci ekranu - bez zmiany czasu do automatycznego wygaszenia
	if(cmd == LCD_BACKLIGHT_REFRESH_CMD)
	{
		if(brightState == LCD_BACKLIGHT_LIGHT_STATE)
		{
			brightState = LCD_BACKLIGHT_TO_LIGHT_STATE;
			result = 1;
		}
		else if(brightState == LCD_BACKLIGHT_DARK_STATE)
		{
			brightState = LCD_BACKLIGHT_TO_DARK_STATE;
			result = 1;
		}
	}
	//LCD_BACKLIGHT_ON (wlaczenie oswietlenia, odliczanie na nowo)
	else if(cmd == LCD_BACKLIGHT_ON_CMD)
	{		
		darkerTime = readModbus(MODBUS_SYSTEM_DARKER_TIME);
		if(darkerTime)
		{
			setTimer100msec(TMR_100MSEC_DARKER, darkerTime);
			if(brightState != LCD_BACKLIGHT_LIGHT_STATE)
			{
				brightState = LCD_BACKLIGHT_TO_LIGHT_STATE;
				result = 1;
			}				
		}			
	}
	//LCD_BACKLIGHT_NOTHING (normalna obsluga przygasania)
	else
	{
		darkerTime = readModbus(MODBUS_SYSTEM_DARKER_TIME);
		if(darkerTime)
		{
			if(!getTimer100msec(TMR_100MSEC_DARKER))
			{
				if(brightState != LCD_BACKLIGHT_DARK_STATE)
				{
					brightState = LCD_BACKLIGHT_TO_DARK_STATE;
				}					
			}
			else
			{
				if(brightState != LCD_BACKLIGHT_LIGHT_STATE)
					brightState = LCD_BACKLIGHT_TO_LIGHT_STATE;			
			}
		}
	}	
	
		
	uint16_t lightLevel;
	_lcdStateContext * lcdContext;
	if(brightState == LCD_BACKLIGHT_TO_LIGHT_STATE)
	{				
		if(lcdButtonData.lightSceneSwitchFlag)
		{
			//aktywacja sceny swietlnej			
			writeModbus(MODBUS_SYSTEM_SHOW_LIGHT_SCENE, lcdButtonData.lightSceneSwitchMask);
		}
		for (int i=0; i<lcdButtonData.newValuesCount; i++)
		{
			writeModbus(lcdButtonData.wakeUpMbAddresses[i], lcdButtonData.wakeUpMbValues[i]);
		}
		lcdButtonData.modbusSpace[MODBUS_READ_ONLY_BIT_DATA_2] |= (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_URZADZENIE);
		
		lcdContext = getLcdWorkQueueFree();
		if(lcdContext != NULL)
		{
			lcdContext->state = LCD_STATE_BRIGHTNESS_JOB;
			lcdContext->substate = LCD_SUBSTATE_BRIGHTNESS_JOB_START;
			lightLevel = readModbus(MODBUS_SYSTEM_BACKLIGHT_ON);
			if(lightLevel == 0xFFFF)
			{
				uint16_t light = readModbus(MODBUS_READ_ONLY_LIGHT_VOLT) / 10;
				if(light > 100)
					light = 100;	//light w zakresie <0;100>
				lcdContext->data[0] = light;
			}				
			else
			{
				if(lightLevel > 100)
					lightLevel = 100;
				lcdContext->data[0] = (uint8_t)lightLevel;
			}
			enqueueSingleReplaceLcdWork();
			brightState = LCD_BACKLIGHT_LIGHT_STATE;
		}
	}
	else if(brightState == LCD_BACKLIGHT_TO_DARK_STATE)
	{
		_lcdStateContext * lcdContext;		
		//zmiana jasnosci podswietlenia:
		lcdContext = getLcdWorkQueueFree();
		if(lcdContext != NULL)
		{
			lcdContext->state = LCD_STATE_BRIGHTNESS_JOB;
			lcdContext->substate = LCD_SUBSTATE_BRIGHTNESS_JOB_START;
			lightLevel = readModbus(MODBUS_SYSTEM_BACKLIGHT_OFF);
			if(lightLevel > 100)
				lightLevel = 100;
			lcdContext->data[0] = (uint8_t)lightLevel;
			enqueueSingleReplaceLcdWork();
			brightState = LCD_BACKLIGHT_DARK_STATE;
		}
		
		if(lcdButtonData.sceneSwitchFlag && lcdButtonData.sceneSwitchNumber != lcdButtonData.scene.activeScene)
		{
			//wyswietlenie sceny domyslnej uzytkownika:
			lcdContext = getLcdWorkQueueFree();
			if(lcdContext != NULL)
			{
				lcdContext->data[0] = lcdButtonData.sceneSwitchNumber;
				lcdContext->data[1] = 1;
				lcdContext->state = LCD_STATE_SCENE_ACTIVE;
				lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_START;
				enqueueLcdWork();
			}
		}
		lcdButtonData.modbusSpace[MODBUS_READ_ONLY_BIT_DATA_2] &= ~(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_URZADZENIE);
	}
	return result;
}
//////////////////////////////////////////////////////////////////////////
void alarmBuzzerActive(void)
{
	alarmBuzzerState = ALARM_BUZZER_ACTIVE;
}
//////////////////////////////////////////////////////////////////////////
void alarmBuzzerDeactive(void)
{
	alarmBuzzerState = ALARM_BUZZER_DEACTIVE;
}
//////////////////////////////////////////////////////////////////////////
void alarmBuzzerService(void)
{
	//do wykrywania polecen zwiazanych z dzialaniem alarmu:
	uint16_t controlValue = readModbus(MODBUS_VOLATILE_CONTROL);
	
	//wykrywanie sytuacji alarmowej:
	uint16_t bitDataValue;
	if(alarmBuzzerState == ALARM_BUZZER_IDLE)
	{
		bitDataValue = readModbus(MODBUS_SYSTEM_BIT_DATA);
		if(bitDataValue & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_ALARM_SET))
		{
			if(readModbus(MODBUS_READ_ONLY_CLOCK_ALARM_H) == readModbus(MODBUS_VOLATILE_CLOCK_0_H))
			{
				if(readModbus(MODBUS_READ_ONLY_CLOCK_ALARM_M) == readModbus(MODBUS_VOLATILE_CLOCK_0_M))
				{
					alarmBuzzerActive();
				}
			}
		
		}
	}
			
	//obsluga alarmu:
	switch(alarmBuzzerState)
	{
		case ALARM_BUZZER_IDLE:
			if(controlValue & (1<<BIT_MODBUS_VOLATILE_ALARM_ACTIVE))
				alarmBuzzerActive();
			break;
		case ALARM_BUZZER_ACTIVE:
			writeSpecialModbus(MODBUS_READ_ONLY_BIT_DATA_2, readModbus(MODBUS_READ_ONLY_BIT_DATA_2) | (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_ALARM));
			alarmBuzzerState = ALARM_BUZZER_LONG_FLASH_OFF;
			break;
		case ALARM_BUZZER_LONG_FLASH_OFF:
			if(!getTimerShort(TMR_SHORT_BUZZER_ALARM))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendBrightness(readModbus(MODBUS_SYSTEM_BACKLIGHT_OFF));
					alarmBuzzerState = ALARM_BUZZER_LONG_BEEP;
					setTimerShort(TMR_SHORT_BUZZER_ALARM, 1);
				}
			}
			break;
		case ALARM_BUZZER_LONG_BEEP:
			if(!getTimerShort(TMR_SHORT_BUZZER_ALARM))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendTouchBuzzorOn(10);
					alarmBuzzerState = ALARM_BUZZER_LONG_FLASH_ON;
					setTimerShort(TMR_SHORT_BUZZER_ALARM, 20);
				}
			}
			break;
		case ALARM_BUZZER_LONG_FLASH_ON:
			if(!getTimerShort(TMR_SHORT_BUZZER_ALARM))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendBrightness(readModbus(MODBUS_SYSTEM_BACKLIGHT_ON));
					alarmBuzzerState = ALARM_BUZZER_SHORT_FLASH_OFF;
					setTimerShort(TMR_SHORT_BUZZER_ALARM, 10);
				}
			}
			break;
		case ALARM_BUZZER_SHORT_FLASH_OFF:
			if(!getTimerShort(TMR_SHORT_BUZZER_ALARM))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendBrightness(readModbus(MODBUS_SYSTEM_BACKLIGHT_OFF));
					alarmBuzzerState = ALARM_BUZZER_SHORT_BEEP;
					setTimerShort(TMR_SHORT_BUZZER_ALARM, 1);
				}
			}
			break;
		case ALARM_BUZZER_SHORT_BEEP:
			if(!getTimerShort(TMR_SHORT_BUZZER_ALARM))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendTouchBuzzorOn(10);
					alarmBuzzerState = ALARM_BUZZER_SHORT_FLASH_ON;
					setTimerShort(TMR_SHORT_BUZZER_ALARM, 20);
				}
			}
			break;
		case ALARM_BUZZER_SHORT_FLASH_ON:
			if(!getTimerShort(TMR_SHORT_BUZZER_ALARM))
			{
				if(!uartLCD->isBusy())
				{
					lcdSendBrightness(readModbus(MODBUS_SYSTEM_BACKLIGHT_ON));
					alarmBuzzerState = ALARM_BUZZER_LONG_FLASH_OFF;
					setTimerShort(TMR_SHORT_BUZZER_ALARM, 200);
				}
			}
			break;
		case ALARM_BUZZER_DEACTIVE:
			writeSpecialModbus(MODBUS_READ_ONLY_BIT_DATA_2, readModbus(MODBUS_READ_ONLY_BIT_DATA_2) & ~(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_ALARM));
			bitDataValue = readModbus(MODBUS_SYSTEM_BIT_DATA);
			bitDataValue &= ~(1<<BIT_MODBUS_SYSTEM_BIT_DATA_ALARM_SET);
			writeModbus(MODBUS_SYSTEM_BIT_DATA, bitDataValue);
			lcdBacklightService(LCD_BACKLIGHT_ON_CMD);
			alarmBuzzerState = ALARM_BUZZER_IDLE;
			break;
		default:
			alarmBuzzerState = ALARM_BUZZER_IDLE;
	}
	
	if(controlValue & (1<<BIT_MODBUS_VOLATILE_ALARM_DEACTIVE))
		alarmBuzzerDeactive();
	uint16_t erasedControlValue = controlValue & ~(1<<BIT_MODBUS_VOLATILE_ALARM_ACTIVE);
	if(erasedControlValue != controlValue)
		writeModbus(MODBUS_VOLATILE_CONTROL, erasedControlValue);
}
//////////////////////////////////////////////////////////////////////////
_lcdStateContext * getLcdWorkQueuePeek(void)
{
	if(!lcdWorkQueueNotEmptyFlag)
		return NULL;
	return &lcdWorkQueue[lcdWorkQueueFirstIndex];
}
//////////////////////////////////////////////////////////////////////////
_lcdStateContext * getLcdWorkQueueFree(void)
{
	if(!lcdWorkQueueNotEmptyFlag)
		return &lcdWorkQueue[0];
	
	uint8_t freeIndex = (lcdWorkQueueLastIndex+1)%LCD_WORK_QUEUE_MAX;
	if(freeIndex == lcdWorkQueueFirstIndex)
		return NULL;
	
	return &lcdWorkQueue[freeIndex];
}
//////////////////////////////////////////////////////////////////////////
_lcdStateContext * getLcdWorkQueueHiPrio(void)
{
	if(!lcdWorkQueueNotEmptyFlag)
		return &lcdWorkQueue[0];
	
	uint8_t freeIndex = 0;
	if(lcdWorkQueueFirstIndex==0)
		freeIndex = LCD_WORK_QUEUE_MAX-1;
	else
		freeIndex = lcdWorkQueueFirstIndex-1;
	
	return &lcdWorkQueue[freeIndex];
}
//////////////////////////////////////////////////////////////////////////
uint8_t enqueueLcdWorkHiPrio(void)
{
	if(!lcdWorkQueueNotEmptyFlag)
	{
		lcdWorkQueueFirstIndex = 0;
		lcdWorkQueueLastIndex = 0;
		lcdWorkQueueNotEmptyFlag = 1;
		return 1;
	}
	
	uint8_t freeIndex = 0;
	if(lcdWorkQueueFirstIndex==0)
		freeIndex = LCD_WORK_QUEUE_MAX-1;
	else
		freeIndex = lcdWorkQueueFirstIndex-1;
	
	lcdWorkQueueFirstIndex = freeIndex;
	return 1;
}
//////////////////////////////////////////////////////////////////////////
uint8_t enqueueLcdWork(void)
{
	if(!lcdWorkQueueNotEmptyFlag)
	{
		lcdWorkQueueFirstIndex = 0;
		lcdWorkQueueLastIndex = 0;
		lcdWorkQueueNotEmptyFlag = 1;
		return 1;
	}		
	
	uint8_t freeIndex = (lcdWorkQueueLastIndex+1)%LCD_WORK_QUEUE_MAX;
	if(freeIndex == lcdWorkQueueFirstIndex)
		return 0;
		
	lcdWorkQueueLastIndex = freeIndex;
	return 1;
}
//////////////////////////////////////////////////////////////////////////
int8_t findWorkInQueue(_lcdWorkState work)
{
	if(lcdWorkQueueNotEmptyFlag)
	{
		uint8_t addingItemIndex = (lcdWorkQueueLastIndex+1)%LCD_WORK_QUEUE_MAX;
		for(int i=0; i<LCD_WORK_QUEUE_MAX; i++)
		{
			if(i != addingItemIndex && lcdWorkQueue[i].state == work)
			{
				//wykryto w kolejce inna instację tego zadania:
				return i;
			}
		}
	}	
	return -1;
}	
//////////////////////////////////////////////////////////////////////////
uint8_t enqueueSingleReplaceLcdWork(void)
{
	uint8_t addingItemIndex = (lcdWorkQueueLastIndex+1)%LCD_WORK_QUEUE_MAX;
	int8_t workSearch = findWorkInQueue(lcdWorkQueue[addingItemIndex].state);
	if(workSearch < 0)
		return enqueueLcdWork();
	else
	{
		//wykryto w kolejce inna instację tego zadania, nadpisac dane zadan:
		lcdWorkQueue[workSearch].substate = lcdWorkQueue[addingItemIndex].substate;
		memmove(lcdWorkQueue[workSearch].data, lcdWorkQueue[addingItemIndex].data, LCD_WORK_CONTEXT_DATA_SIZE);
		lcdWorkQueue[addingItemIndex].state = LCD_STATE_NONE;
		lcdWorkQueue[addingItemIndex].substate = LCD_SUBSTATE_NONE;
		return 1;
	}
}	
//////////////////////////////////////////////////////////////////////////
void dequeueLcdWork(void)
{
	if(!lcdWorkQueueNotEmptyFlag)
	{
		return;
	}		
	
	lcdWorkQueue[lcdWorkQueueFirstIndex].state = LCD_STATE_NONE;
	lcdWorkQueue[lcdWorkQueueFirstIndex].substate = LCD_SUBSTATE_NONE;
	if(lcdWorkQueueFirstIndex == lcdWorkQueueLastIndex)
	{
		lcdWorkQueueNotEmptyFlag = 0;
		return;
	}
	lcdWorkQueueFirstIndex = (lcdWorkQueueFirstIndex+1)%LCD_WORK_QUEUE_MAX;
}
//////////////////////////////////////////////////////////////////////////
void clearLcdWorkQueue(void)
{
	lcdWorkQueueNotEmptyFlag = 0;
}
//////////////////////////////////////////////////////////////////////////
uint8_t getJobCount(void)
{
	if(!lcdWorkQueueNotEmptyFlag)
		return 0;
	uint8_t endIndex = lcdWorkQueueLastIndex;
	if(lcdWorkQueueFirstIndex > lcdWorkQueueLastIndex)
		endIndex += LCD_WORK_QUEUE_MAX;
	return endIndex - lcdWorkQueueFirstIndex + 1;
}
//////////////////////////////////////////////////////////////////////////
void lcdService(void)
{	
	//obsluga glownych zadan:
	lcdActuallContext = getLcdWorkQueuePeek();
	if(lcdActuallContext != NULL)
	{		
		switch(lcdActuallContext->state)
		{
			case LCD_STATE_INIT:
				lcdStateInit();
				break;
			case LCD_STATE_WAGO_NO_CONNECT:
				lcdStateWagoNoConnect();
				break;
			
			//scena uzytkownika:
			case  LCD_STATE_SCENE_ACTIVE:
				lcdStateSceneActive(0);
				break;
			case  LCD_STATE_PRINT_BARS:
				lcdStatePrintObjects(LCD_OBJECT_BAR, 0);
				break;
			case  LCD_STATE_PRINT_STATES:
				lcdStatePrintObjects(LCD_OBJECT_STATE, 0);
				break;
			case  LCD_STATE_PRINT_NUMBERS:
				lcdStatePrintObjects(LCD_OBJECT_NUMBER, 0);
				break;
			case  LCD_STATE_PRINT_BUTTONS:
				lcdStatePrintObjects(LCD_OBJECT_BUTTON, 0);
				break;
			//scena serwisowa:
			case  LCD_STATE_SERVICE_SCENE_ACTIVE:
				lcdStateSceneActive(1);
				break;
			case  LCD_STATE_SERVICE_PRINT_BARS:
				lcdStatePrintObjects(LCD_OBJECT_BAR, 1);
				break;
			case  LCD_STATE_SERVICE_PRINT_STATES:
				lcdStatePrintObjects(LCD_OBJECT_STATE, 1);
				break;
			case  LCD_STATE_SERVICE_PRINT_NUMBERS:
				lcdStatePrintObjects(LCD_OBJECT_NUMBER, 1);
				break;
			case  LCD_STATE_SERVICE_PRINT_BUTTONS:
				lcdStatePrintObjects(LCD_OBJECT_BUTTON, 1);
				break;
			
			case  LCD_STATE_SCENE_ACTIVE_DONE:
				lcdButtonData.scene.sceneReady = 1;
				dequeueLcdWork();
				break;
		
			case LCD_STATE_TOUCH:
 				lcdStateClick();
				break;
				
			case LCD_STATE_BRIGHTNESS_JOB:
				lcdBrightness();
				break;
				
			case LCD_STATE_INIT_DEVICE_WAKE_UP:
				lcdInitWakeUp();
				break;
				
			case LCD_STATE_INIT_PID:
				lcdInitPid();
				break;
				
			case LCD_STATE_PROJECT_CLEAR_NOTUSE:
				lcdClearNotuseProjekt();
				break;							
			
			case LCD_STATE_VALID_MODBUS_DATA:
				lcdValidModbusData();
				break;
				
									
			default:
				if(lcdActuallContext->state >= LCD_STATE_ALL)
				{
					dequeueLcdWork();	
				}
				break;
		}
	}		
}

//////////////////////////////////////////////////////////////////////////
void lcdServiceHighPriority(void)
{
	//obsluga zadan o podwyzszonym priorytecie:
	lcdActuallContext = getLcdWorkQueuePeek();
	if(lcdActuallContext != NULL)
	{
		switch(lcdActuallContext->state)
		{
			case LCD_STATE_CLEAR:
				lcdStateClear();
				break;
			case LCD_STATE_PRINT:
				lcdStatePrint();
				break;
			case LCD_STATE_IMAGE_SAVE:
				lcdStateSaveImage();
				break;
			case  LCD_STATE_PROJECT_SAVE:
				lcdStateProjectSave();
				break;
			case  LCD_STATE_IMAGE_CRC:
				lcdStateImageCrc();
				break;
			default:
				//zdjecie z kolejki wszystkich zadan nie nalezacych do grupy o podwyzszonym priorytecie
				dequeueLcdWork();
			break;
		}
	}
}