﻿/*!
 *  @file lcd.h
 *	@brief Obsługa wyświetlacza LCD.
 */ 

#ifndef LCD_H_
#define LCD_H_

#include "uart.h"
#include "lcdButton.h"

/*! 
	@def MAX_REQUEST_COUNT
    @brief Maksymalna ilość prób pobierania danych z LCD
*/
#define MAX_REQUEST_COUNT	3

/*!
	@enum _lcdWorkState
	@brief Tryb pracy wyświetlacza LCD
*/
typedef enum {
	LCD_STATE_NONE = 0,					/*< Zadanie puste */
	LCD_STATE_INIT,						/*< Inicjalizacja wyświetlacza */
	LCD_STATE_WAGO_NO_CONNECT,			/*< Wyswietlanie grafik o braku polaczenia komunikacji i ponownym nawiazywaniu komunikacji */
	LCD_STATE_CLEAR,					/*< Wyczyszczenie ekranu podaną wartością */
	LCD_STATE_PRINT,					/*< Wyswietlenie pikseli */
	LCD_STATE_IMAGE_SAVE,				/*< Zapisanie aktualnego obrazu na wyświetlaczu */
	LCD_STATE_IMAGE_CRC,				/*< Wyswietlenie obrazu i podanie jego crc */
	LCD_STATE_PROJECT_SAVE,				/*< Zapisanie komunikatu pochodzącego z pliku konfiguracyjnego */
	LCD_STATE_SCENE_ACTIVE,				/*< Aktywacja sceny */
	LCD_STATE_PRINT_BARS,				/*< Wyswietlenie obiektow BAR */
	LCD_STATE_PRINT_STATES,				/*< Wyswietlenie obiektow STATE */
	LCD_STATE_PRINT_NUMBERS,			/*< Wyswietlenie obiektow NUMBER */
	LCD_STATE_PRINT_BUTTONS,			/*< Wyswietlenie przyciskow */
	LCD_STATE_SERVICE_SCENE_ACTIVE,		/*< Aktywacja belki (sceny) serwisowej */
	LCD_STATE_SERVICE_PRINT_BARS,		/*< Wyswietlenie obiektow BAR belki (sceny) serwisowej */
	LCD_STATE_SERVICE_PRINT_STATES,		/*< Wyswietlenie obiektow STATE belki (sceny) serwisowej */
	LCD_STATE_SERVICE_PRINT_NUMBERS,	/*< Wyswietlenie obiektow NUMBER belki (sceny) serwisowej */
	LCD_STATE_SERVICE_PRINT_BUTTONS,	/*< Wyswietlenie przyciskow belki (sceny) serwisowej */
	LCD_STATE_SCENE_ACTIVE_DONE,		/*< Ustawienie flagi mówiącej o tym że nowa scena została już wgrana */
	LCD_STATE_TOUCH,					/*< Kliknięcie w wywietlacz LCD */
	LCD_STATE_DS_JOB,					/*< zadanie dla DS */
	LCD_STATE_UART_JOB,					/*< zmiana predkosci uartu */
	LCD_STATE_BRIGHTNESS_JOB,			/*< ustawienie jasnosci */
	LCD_STATE_INIT_DEVICE_WAKE_UP,		/*< Pobranie danych konfiguracyjnych dotyczacym aktywacji urządzenia */
	LCD_STATE_INIT_PID,					/*< Pobranie danych konfiguracyjnych dotyczacym algorytmow PID */
	LCD_STATE_PROJECT_CLEAR_NOTUSE,		/*< Wykasowanie stron pamieci flash z wyswietlacza, w ktorych moga znajdowac sie stare, niewykorzystywane obecnie sceny */
	LCD_STATE_VALID_MODBUS_DATA,		/*< Na podstawie skrajnych wartosci elementow zapisach na wyswietlaczu sprawdzana jest przestrze modbus */
	/*! Liczba wszystkich zadan */	
	LCD_STATE_ALL,
}_lcdWorkState;

/*!
	@enum _lcdWorkSubstate
	@brief Podtryb pracy wyświetlacza LCD
*/
typedef enum {
	LCD_SUBSTATE_NONE = 0,		/*! < Podzadanie puste */
	
	/// Inicjalizacja wyświetlacza: rozpoczęcie odliczania czasu na inicjalizację wyświetlacza LCD
	LCD_SUBSTATE_INIT_START,
	/// Inicjalizacja wyświetlacza: spytaj o wersję programu i ustawienia wyświetlacza
	LCD_SUBSTATE_INIT_ASK,
	/// Inicjalizacja wyświetlacza: oczekiwanie na odebranie komunikatu
	LCD_SUBSTATE_INIT_RESP_WAIT,
	/// Inicjalizacja wyświetlacza: wykryto urządzenie
	LCD_SUBSTATE_INIT_FOUND,
	/// Inicjalizacja wyświetlacza: zmiana prędkości komunikacji odnalezionego wyświetlacza
	LCD_SUBSTATE_INIT_SET_BAUD_RATE,
	/// Inicjalizacja wyświetlacza: błąd inicjalizacji - wyświetlacz LCD nie został wykryty
	LCD_SUBSTATE_INIT_ERR,
	
	
	LCD_SUBSTATE_WAGO_NO_CONNECT_START,				/*!< Wyswietlanie grafik laczenia z WAGO: start */
	LCD_SUBSTATE_WAGO_NO_CONNECT_CONTROL_DATA,
	LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_INIT_DATA,
	LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_IMAGE_GRAPHIC,
	LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_TEXT_GRAPHIC,
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_LOGO,			/*!< Pobranie grafik laczenia z WAGO:  */
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECTING,			/*!< Pobranie grafik laczenia z WAGO:  */
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_OK,			/*!< Pobranie grafik laczenia z WAGO:  */	
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_CONNECT_ERROR,			/*!< Pobranie grafik laczenia z WAGO:  */
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_EXIT,			/*!< Pobranie grafik laczenia z WAGO:  */
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_LOGO,			/*!< */
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECTING,			/*!< */	
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_OK,			/*!< */
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_CONNECT_ERROR,			/*!< */	
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_LOGO,			/*!< Odczytywanie tekstu podczas imitacji inicjalizacji polączenia z serwerem (MODBUS)*/
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECTING,			/*!< Odczytywanie tekstu podczas imitacji inicjalizacji polączenia z serwerem (MODBUS)*/	
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_OK,			/*!< Odczytywanie tekstu podczas imitacji inicjalizacji polączenia z serwerem (MODBUS)*/
	LCD_SUBSTATE_WAGO_NO_CONNECT_READ_TEXT_CONNECT_ERROR,			/*!< Odczytywanie tekstu podczas imitacji inicjalizacji polączenia z serwerem (MODBUS)*/
	LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_LOGO,					/*!< */
	LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECTING,			/*!< */
	LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECT_OK,			/*!< */
	LCD_SUBSTATE_WAGO_NO_CONNECT_WAIT_CONNECT_ERROR,	
	LCD_SUBSTATE_WAGO_NO_CONNECT_CHECK_MODBUS,			/*!< */
	LCD_SUBSTATE_WAGO_NO_CONNECT_PRINT,		/*!< Wyswietlanie grafik laczenia z WAGO:  */
	LCD_SUBSTATE_WAGO_NO_CONNECT_SEND_TEXT,
	LCD_SUBSTATE_WAGO_NO_CONNECT_ERROR,
	LCD_SUBSTATE_WAGO_NO_CONNECT_EXIT,
	
	
	LCD_SUBSTATE_CLEAR_START,		/*!< Czyszczenie wyświetlacza: start */
	LCD_SUBSTATE_CLEAR_EXECUTE,		/*!< Czyszczenie wyświetlacza: wysłanie komendy wypełnienia podanym kolorem */
	LCD_SUBSTATE_CLEAR_EXIT,		/*!< Czyszczenie wyświetlacza: koniec */
	
	/// Wyswietlenie pikseli: start
	LCD_SUBSTATE_PRINT_START,
	/// Wyswietlenie pikseli: koniec
	LCD_SUBSTATE_PRINT_EXIT,
	
	/// Zapisanie obrazu: start
	LCD_SUBSTATE_IMAGE_SAVE_START,
	/// Zapisanie obrazu: koniec
	LCD_SUBSTATE_IMAGE_SAVE_EXIT,
	
	/// Podanie CRC obrazu: start
	LCD_SUBSTATE_IMAGE_CRC_START,
	LCD_SUBSTATE_IMAGE_CRC_ASK,
	/// Podanie CRC obrazu: koniec
	LCD_SUBSTATE_IMAGE_CRC_EXIT,
	
	LCD_SUBSTATE_PROJECT_SAVE_START,
	LCD_SUBSTATE_PROJECT_SAVE_WORKING,
	LCD_SUBSTATE_PROJECT_SAVE_EXIT,
	
	
	/// Przygotowanie danych do odczytu danej sekcji
	LCD_SUBSTATE_SCENE_ACTIVE_START,
	/// Wyslanie pytania o dane startowe dla sceny
	LCD_SUBSTATE_SCENE_ACTIVE_READ_INIT_DATA,
	/// Oczekiwanie na odebranie danych startowych dla sceny	
	LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_INIT_DATA,
	/// Przygotowanie danych do odczytu danej sekcji
	LCD_SUBSTATE_SCENE_ACTIVE_PREPARE_CONTROL_DATA,
	/// Wyslanie pytania o dane sterujace kolejnych obiektow znajdujacych sie na scenie (BAR, STATE, NUMBER lub BUTTON)
	LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA,
	/// Trwa oczekiwanie na odebranie danych sterujących dotyczących kolejnego elementu
	LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_CONTROL_DATA,
	/// Wysłanie komendy wyswietlenia grafiki, kontynuacja pracy od podzadania zapisanego w przestrzeni danych aktualnego zadania
	LCD_SUBSTATE_SCENE_ACTIVE_SEND_GRAPHIC,
	/// Wyslanie pytania o grafikę kolejnych obiektow LABEL znajdujących się na scenie
	LCD_SUBSTATE_SCENE_ACTIVE_READ_IMAGE,
	/// Trwa oczekiwanie na odebranie danych dotyczących grafiki obiektu LABEL
	LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_IMAGE_GRAPHIC,
	/// Oczekiwanie na zakonczenie wysyłania komendy wywietlającej tekst i wyslanie komendy z prosba o dane kolejnego tekstu
	LCD_SUBSTATE_SCENE_ACTIVE_READ_TEXT,
	/// Trwa oczekiwanie na odebranie danych dotyczących tekstu
	LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_TEXT,
	/// Wysłanie komendy wyswietlenia tekstu
	LCD_SUBSTATE_SCENE_ACTIVE_SEND_TEXT,
	
	//LCD_STATE_PRINT_BUTTONS:
	LCD_SUBSTATE_PRINT_OBJECTS_START,				/*!< Podzadanie rozpoczynające wyświeltanie obiektów BUTTON */
	LCD_SUBSTATE_PRINT_OBJECTS_PREPARE_NUMBER,		/*!< Przygotowanie danych do wyswietlenia obiektu NUMBER */
	LCD_SUBSTATE_PRINT_OBJECTS_READ_GRAPHIC,		/*!< Wyslanie pytania o grafikę kolejnego obiektu BUTTON */
	LCD_SUBSTATE_PRINT_OBJECTS_WATCHDOG,	/*!< Oczekiwanie na odebranie danych dotyczących grafiki obiektu BUTTON */
	LCD_SUBSTATE_PRINT_OBJECTS_EXEC_GRAPHIC,		/*!< Wysłanie komendy wyświetlenia grafiki obiektu BUTTON */
	LCD_SUBSTATE_PRINT_OBJECTS_READ_TEXT,			/*!< Wyslanie pytania o tekst kolejnego obiektu BUTTON */
	LCD_SUBSTATE_PRINT_OBJECTS_EXEC_TEXT,			/*!< Wysłanie komendy wyświetlenia tekstu obiektu BUTTON */
	LCD_SUBSTATE_PRINT_OBJECTS_NEXT,				/*!< Przejście do wyświetlania kolejnego obiektu BUTTON */
	LCD_SUBSTATE_PRINT_OBJECTS_EXIT,				/*!< Koniec zadania */

	LCD_SUBSTATE_TOUCH_START,
	LCD_SUBSTATE_TOUCH_OFF,
	LCD_SUBSTATE_TOUCH_ON,
	LCD_SUBSTATE_TOUCH_PRESS_ON,
	LCD_SUBSTATE_TOUCH_PRESS_OFF,
	LCD_SUBSTATE_TOUCH_EXIT,
		
	
	LCD_SUBSTATE_SCENE_ACTIVE_EXIT,	
	LCD_SUBSTATE_SCENE_ACTIVE_ERROR,
		
	LCD_SUBSTATE_BRIGHTNESS_JOB_START,
	LCD_SUBSTATE_BRIGHTNESS_JOB_EXIT,
	
	//LCD_STATE_INIT_DEVICE_WAKE_UP:
	LCD_SUBSTATE_INIT_WAKE_UP_START,
	LCD_SUBSTATE_INIT_WAKE_UP_SEND_CMD,
	LCD_SUBSTATE_INIT_WAKE_UP_WATCHDOG,
	LCD_SUBSTATE_INIT_WAKE_UP_END,
	
	//LCD_STATE_INIT_PID:
	LCD_SUBSTATE_INIT_PID_START,
	LCD_SUBSTATE_INIT_PID_SEND_CMD,
	LCD_SUBSTATE_INIT_PID_WATCHDOG,
	LCD_SUBSTATE_INIT_PID_END,
	
	//LCD_STATE_PROJECT_CLEAR_NOTUSE:
	LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_START,
	LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_CMD,
	LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_WATCHDOG,
	LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_NEXT,
	LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_END,
	
	//LCD_STATE_VALID_MODBUS_DATA:
	LCD_SUBSTATE_VALID_MODBUS_START,
	LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA,
	LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_INIT_DATA,
	LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA_OK,
	
	LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA,
	LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BAR_CONTROL_DATA,
	LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA_OK,
	
	LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA,	
	LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_NUMBER_CONTROL_DATA,
	LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA_OK,
	
	LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA,
	LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BUTTON_CONTROL_DATA,
	LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA_OK,
	
	LCD_SUBSTATE_VALID_MODBUS_CHANGE_TO_NEXT_SCENE,
	
	LCD_SUBSTATE_VALID_MODBUS_END,
	LCD_SUBSTATE_VALID_MODBUS_ERR,
	
	/// Przejście do domyślnego etapu dla danego trybu pracy
	LCD_SUBSTATE_DEFAULT = 0xFF,
}_lcdWorkSubstate;



/*!
	@enum _lcdObjectType
	@brief Typ obiektu, dla którego wykonywane jest jakieś zadanie
*/
typedef enum {
	LCD_OBJECT_IMAGE,
	LCD_OBJECT_BAR,
	LCD_OBJECT_STATE,
	LCD_OBJECT_NUMBER,
	LCD_OBJECT_BUTTON,
}_lcdObjectType;


/*!
	@enum _lcdBacklightCmd
	@brief Zbiór poleceń związanych z kontrolą podswietlenia LCD
*/
typedef enum{
	LCD_BACKLIGHT_NORMAL_CMD,			/*! < normalny tryb pracy */
	LCD_BACKLIGHT_ON_CMD,				/*! < wymuszenie włączenia podswietlenia i rozpoczecie na nowo odliczanie czasu do automatycznego wygaszenia*/
	LCD_BACKLIGHT_REFRESH_CMD,			/*! < odswierzenie wyswietlanej jasnosci ekranu (bez zmiany czasu do automatycznego wygaszenia) */
}_lcdBacklightCmd;

/*!
	@enum _lcdBacklightState
	@brief Typ określający stan aktualnego podświetlenia
*/
typedef enum{
	LCD_BACKLIGHT_LIGHT_STATE,			/*! < Podswietlenie włączone */
	LCD_BACKLIGHT_DARK_STATE,			/*! < Podswietlenie wyłączone */
	LCD_BACKLIGHT_TO_LIGHT_STATE,		/*! < Włącz podświetlenie */
	LCD_BACKLIGHT_TO_DARK_STATE,		/*! < Wyłącz podświetlenie */
}_lcdBacklightState;

/*!
	@enum _alarmBuzzerState
	@brief Etapy wytwarzania alarmu lub oczekiwanie na niego
*/
typedef enum
{
	ALARM_BUZZER_IDLE,				/*!< Oczekiwanie na wywołanie alarmu */
	ALARM_BUZZER_ACTIVE,			/*!< włączenie alarmu */
	ALARM_BUZZER_LONG_BEEP,			/*!< Alarm aktywny, wydanie dźwięku po dłuższej przerwie */
	ALARM_BUZZER_LONG_FLASH_OFF,	/*!< Alarm aktywny, mrugnięcie po dłuższej przerwie */
	ALARM_BUZZER_LONG_FLASH_ON,		/*!< Alarm aktywny, mrugnięcie po dłuższej przerwie */
	ALARM_BUZZER_SHORT_BEEP,		/*!< Alarm aktywny, wydanie dźwięku po krótszej przerwie */
	ALARM_BUZZER_SHORT_FLASH_OFF,	/*!< Alarm aktywny, mrugnięcie po krótszej przerwie */
	ALARM_BUZZER_SHORT_FLASH_ON,	/*!< Alarm aktywny, mrugnięcie po krótszej przerwie */
	ALARM_BUZZER_DEACTIVE,			/*!< wyłączenie alarmu */
}_alarmBuzzerState;
	
/*!
	@struct _lcdData
	@brief Dane wyświetlacza LCD
	
	Struktura przechowująca dane wyświetlacza LCD.
*/
typedef struct{	
	/// Wersja wyświetlacza LCD
	uint8_t version[2];
	
	/// TFT_ID
	int8_t id;
	
	/// Prędkość komunikacji
	_baudRate baudRate;
	
	/// Opcje dotyczące pracy wyświetlacza LCD
	uint8_t options;	
}_lcdData;


/*! 
	@def LCD_WORK_QUEUE_MAX
    @brief Pojemność kolejki na zadania.
*/
#define LCD_WORK_QUEUE_MAX			LCD_STATE_ALL


/*!
	@def LCD_WORK_CONTEXT_DATA_SIZE
	@brief Wielkość przestrzeni pamięci danych jaka jest przeznaczona dla każdego podzadania
*/
#define LCD_WORK_CONTEXT_DATA_SIZE	30

/*!
	@def LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX
	@brief Do wyświetlenia obiektu NUMBER
	@details Indeks danych roboczych aktualnego podzadania pod którym znajduje się
	5-cioelementowa tablica uint8_t przechowująca kolejne cyfry/znaki wyświetlanej liczby
*/
#define LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX (LCD_WORK_CONTEXT_DATA_SIZE-5)

/*!
	@def LCD_WORK_CONTEXT_NUMBER_DIGIT_COUNT_INDEX 
	@brief Do wyświetlenia obiektu NUMBER
	@details Indeks danych roboczych aktualnego podzadanie pod którym znajduje się
	zmienna typu uint8_t określająca ilość cyfr/znaków które składają się na prezentację danego obiektu NUMBER
*/
#define LCD_WORK_CONTEXT_NUMBER_DIGIT_COUNT_INDEX (LCD_WORK_CONTEXT_NUMBER_ARRAY_INDEX-1)

/*!
	@def LCD_WORK_CONTEXT_NUMBER_DIGIT_ACTUAL_INDEX 
	@brief Do wyświetlenia obiektu NUMBER
	@details Indeks danych roboczych aktualnego podzadanie pod którym znajduje się
	zmienna typu uint8_t określająca która obecnie cyfra/znak jest wyświetlana
*/
#define LCD_WORK_CONTEXT_NUMBER_DIGIT_ACTUAL_INDEX (LCD_WORK_CONTEXT_NUMBER_DIGIT_COUNT_INDEX-1)

/*!
	@def LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX
	@brief Do wyświetlenia obiektu NUMBER
	@details Indeks danych roboczych aktualnego podzadanie pod którym znajduje się
	zmienna typu uint16_t przechowująca współrzędną X dla grafiki kolejnej cyfry/znaku
*/
#define LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX (LCD_WORK_CONTEXT_NUMBER_DIGIT_ACTUAL_INDEX-2)

/*!
	@def LCD_WORK_CONTEXT_NUMBER_DELTA_X_PREVIOUS_INDEX
	@brief Do wyświetlenia obiektu NUMBER
	@details Indeks danych roboczych aktualnego podzadanie pod którym znajduje się
	zmienna typu uint16_t przechowująca szerokosc poprzednio wyswietlanego znaku/cyfry.
*/
#define LCD_WORK_CONTEXT_NUMBER_DELTA_X_PREVIOUS_INDEX (LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX-2)

/*!
	@def LCD_WORK_CONTEXT_BACKGROUND_INDEX
	@brief Do wyświetlenia tła sceny
	@details Indeks danych roboczych aktualnego podzadanie pod którym znajduje się
	dane dotyczące grafiki tła aktualnej sceny
*/
#define LCD_WORK_CONTEXT_BACKGROUND_INDEX (LCD_WORK_CONTEXT_DATA_SIZE-9)

/*!
	@struct _lcdStateContext
	@brief Kontekst zadania
	
	Kotekst służy do przechowania zadania w kolejce zadań. Zawiera określenie jakie zadanie ma być wykonane,
	oraz przestrzeń na przechowanie danych potrzebnych dla jego wykonania.
*/
typedef struct{
	/// Rodzaj zadania do wykonania
	_lcdWorkState state;
	/// Aktualny rodzaj działania w ramach danego zadania
	_lcdWorkSubstate substate;
	/// Przestrzeń na dane do przechowania
	uint8_t data[LCD_WORK_CONTEXT_DATA_SIZE];	
	/// Licznik do odliczania nieudanych prob realizacji podzadania
	uint8_t backupCounter;
	/// Podzadanie realizowane w przypadku ponowienia próby realizacji jakiegoś podzadania
	_lcdWorkSubstate backupSubstate;
}_lcdStateContext;


/**
 * @brief Funkcja kontrolująca klikanie w wywietlacz LCD
 */
void lcdTouchService(void);

/**
 * @brief Funkcja kontrolująca podswietlenie LCD (w tym automatyczne wygaszanie ekranu)
 * @param cmd polecenie sterujące działaniem funkcji
 * @return 0 - gdy cmd==LCD_BACKLIGHT_NORMAL_CMD, lub gdy funkcja nie dokonała zmiany jasności swiecenia;
 		  !0 - gdy dla cmd==LCD_BACKLIGHT_ON_CMD lub cmd==LCD_BACKLIGHT_REFRESH_CMD funkcja dokonała zmiany świecenia
 */
uint8_t lcdBacklightService(_lcdBacklightCmd cmd);


/*!
	@brief Włączenie alarmu
*/
void alarmBuzzerActive(void);

/*!
	@brief Wyłączenie alarmu
*/
void alarmBuzzerDeactive(void);

/*!
	@brief Obsługa alarmu dźwiękowego dla budzika
*/
void alarmBuzzerService(void);


/*!
	@brief Główna funkcja realizująca logikę współpracy z wyświetlaczem LCD
*/
void lcdService(void);	


/*!
	@brief Funkcja realizująca logikę współpracy z wyświetlaczem LCD podczas zadan o podwyzszonym priorytecie
	(wgrywanie grafik lub konfiguracji)
*/
void lcdServiceHighPriority(void);

/*!
	@brief Pobranie kontekstu zadania z wierzchołka kolejki
	@return Wskaźnik na kontekst zadania, które jest aktualnie na początku kolejki
*/
_lcdStateContext * getLcdWorkQueuePeek(void);

/*!
	@brief Pobranie wolnego obszaru na kontekst
	@details Metoda służy po to żeby zapisać zbiór potrzebnych danych dla kolejnego kontekstu od razu do właściwej przestrzeni na ten kontekst
	@return Zwracany jest wskaźnik na wolną (kolejna po ostatnim używanej kontekstem)
		przestrzeń.
*/
_lcdStateContext * getLcdWorkQueueFree(void);


/*!
	@brief Pobranie wolnego obszaru na kontekst dla zadania o wysokim priorytecie!
	@details zadanie bedzie obslugiwane poza kolejnoscia
	@return Zwracany jest wskaźnik na wolną
*/
_lcdStateContext * getLcdWorkQueueHiPrio(void);

/*!
	@brief Zwiększenie (jeśli to możliwe) ilość zadań w kolejce
	@details Dane dla nowego zadania powinny być zapisane przy wykorzystaniu metody @ref getLcdWorkQueueFree(void)
	@return 0 - kolejka na zadania LCD jest pełna, nie udało się dodać kolejnego zadania;
			@n !0 - wstawienie zadania do kolejki udało się;
*/
uint8_t enqueueLcdWork(void);

uint8_t enqueueLcdWorkHiPrio(void);


/*!
	@brief
		Funkcja wyszukuje czy dane zadanie jest aktualnie w kolejce do wykonania
	@return
		Indeks w tablicy gdzie znaleziono zadanie lub -1 w przypadku gdy takiego nie znaleziono
*/
int8_t findWorkInQueue(_lcdWorkState work);

/*!
	@brief Dodanie zadania do kolejki.
	Działanie takie jak funkcji enqueueLcdWork(), z tym, że przed dodaniem nowego zadania sprawdzana jest lista oczekujących zadan.
	Jeśli w kolejce zadań znajduje się już takie zadanie, to nowe nie jest dodawane, tylko zmieniane są dane wywołania tego zadania,
	czyli dla instancji zadania wczesniej dodanego, wstawiane są dane aktualnie wprowadzanego zadania
*/
uint8_t enqueueSingleReplaceLcdWork(void);

/*!
	@brief Zdjęcie z kolejki pierwszego zadania
*/
void dequeueLcdWork(void);

/*!
	@brief Usunięcie wszystkich zadań które znajdują się w kolejce zadań.
*/
void clearLcdWorkQueue(void);

/*!
	@brief Zwraca ilość zadań znajdujących się w kolejce do wywołania
	@return Ilość zadań czekających w kolejce
*/
uint8_t getJobCount(void);

/*!
	@brief Funkcja sprawdza czy należy odświeżyć liczbę
	@details Funkcja uwzględnia to, czy wartość zapisana w przestrzeni modbus wykracza poza zakres wyznaczony przez wartości min i max, uwzględnia flagę blokady wymuszonego nadpisania gdy wystąpi takie wykroczenia poza zakres, uwzględniona jest też blokada zapisu w przestrzeni modbus tylko do odczytu
	@param number wskaźnik na obiekt
	@return 1-obiekt należy zaktualizować; 0-obiektu nie należy aktualizować
*/
uint8_t isNumberToRefresh(_numberData * number);

_lcdData lcdData;

#endif /* LCD_H_ */
