﻿/*!
 *  @file lcdButton.c
 */

#include <avr/io.h>
#include <stdlib.h> 

#include "lcdButton.h"
#include "processor.h"
#include "main.h"
#include "eemem.h"
#include "ds.h"


void initData(void)
{
	initEepromData();
	writeSpecialModbus(MODBUS_READ_ONLY_BIT_DATA_2, 0);
	writeSpecialModbus(MODBUS_READ_ONLY_CLOCK_ALARM_H, MODBUS_DEF_VAL_CLOCK_ALARM_INACTIVE);
	writeSpecialModbus(MODBUS_READ_ONLY_CLOCK_ALARM_M, MODBUS_DEF_VAL_CLOCK_ALARM_INACTIVE);
	writeSpecialModbus(MODBUS_READ_ONLY_TEMP_0, 0);
	writeSpecialModbus(MODBUS_READ_ONLY_TEMP_1, 0);
	writeSpecialModbus(MODBUS_READ_ONLY_LIGHT, 0);
	writeSpecialModbus(MODBUS_READ_ONLY_PASS_DIGITS, 0);
	writeSpecialModbus(MODBUS_READ_ONLY_PASSWORD, 0);		
	writeModbus(MODBUS_VOLATILE_ACTIVE_SCENE, readModbus(MODBUS_SYSTEM_DEFAULT_SCENE));
	writeModbus(MODBUS_VOLATILE_CONTROL, 0);
	writeModbus(MODBUS_VOLATILE_SAVE_LIGHT_SCENE, 0);	
	writeModbus(MODBUS_VOLATILE_DATE_YAER, 2012);
	writeModbus(MODBUS_VOLATILE_DATE_MONTH, 1);
	writeModbus(MODBUS_VOLATILE_DATE_DAY, 1);
	writeModbus(MODBUS_VOLATILE_CLOCK_0_H, 12);
	writeModbus(MODBUS_VOLATILE_CLOCK_0_M, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_1_H, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_1_M, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_2_H, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_2_M, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_3_H, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_3_M, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_4_H, 0);
	writeModbus(MODBUS_VOLATILE_CLOCK_4_M, 0);	
	for(int i = MODBUS_SYSTEM_BEGIN; i < MODBUS_SIZE; i++)
	{
		writeModbus(i, readModbus(i));
	}
	lcdButtonData.modbusSpace[MODBUS_VOLATILE_ACTUALIZATION] = MODBUS_DEF_VAL_ACTUALIZATION_STATE;
	lcdButtonData.modbusSpace[MODBUS_READ_ONLY_BIT_DATA_2] |= (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_URZADZENIE);
}

uint16_t readModbus(uint16_t address)
{
	if(address < MODBUS_SIZE)
	{
		if(address == MODBUS_READ_ONLY_BIT_DATA_1)
		{
			uint16_t tmp = lcdButtonData.modbusSpace[address];
			lcdButtonData.modbusSpace[address] &= ~((1<<BIT_MODBUS_READ_ONLY_BIT_DATA_1_RISING_DIO1)|
													(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_1_RISING_DIO2)|
													(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_1_FALLING_DIO1)|
													(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_1_FALLING_DIO2));
			return tmp;													
		}
		return lcdButtonData.modbusSpace[address];
	}		
	else
		return 0;
}

uint16_t readInputRegister(uint16_t address)
{
	if(address == INPUT_REGISTER_BOOT_VER)
	{
		return lcdButtonData.bootSoftVersion;		
	}
	else if(address == INPUT_REGISTER_SOFT_VER)
	{
		return MAKE_VERSION(__APP_VER_MAIN__, __APP_VER_REV__);
	}
	return 0;
}

void writeSpecialModbus(uint16_t address, uint16_t value)
{
	lcdButtonData.modbusSpace[address] = value;
}

void writeModbus(uint16_t address, uint16_t value)
{
	static uint16_t hourTmpValue;			//zmienna przechowujące dwa bajty MSB z liczby 4-bajtowej zapisywanej pod adresami systemowymi właściwymi dla godziny; konwersja liczby 4-bajtowej (zliczającej milisekundy od początku doby) na dwie wartości 2-bajtowe (godziny i minuty) następuje podczas zapisu do minut
	static uint8_t lastHourValueSave;		//zmienna przechowuje informacje dotyczącą tego, która godzina (MODBUS_VOLATILE_CLOCK_0_H - MODBUS_VOLATILE_CLOCK_5_H) zostala zapisana w w zmiennej hourTmpValue;
	uint8_t writeLock = false;				//flaga blokuje zapis do przestrzeni modbus
	_lcdStateContext * lcdStateContext;
	uint8_t indeks;
	uint16_t mask;
	if(address >= MODBUS_VOLATILE_BEGIN)
	{
		if(address < MODBUS_VOLATILE_END)
		{
			uint32_t time;
			uint16_t hours;
			uint16_t minutes;
			switch(address)
			{
				case MODBUS_VOLATILE_ACTUALIZATION:
					//brak dzialania
				break;
				case MODBUS_VOLATILE_ACTIVE_SCENE:
					//zapis nowej sceny zostanie zrobiony podczas realizacji zadania wyświetlenia nowej sceny
					if(value < SCENES_COUNT && value != lcdButtonData.modbusSpace[MODBUS_VOLATILE_ACTIVE_SCENE])
					{
						lcdStateContext = getLcdWorkQueueFree();
						if(lcdStateContext != NULL)
						{
							lcdStateContext->data[0] = value;
							lcdStateContext->data[1] = 1;
							lcdStateContext->state = LCD_STATE_SCENE_ACTIVE;
							lcdStateContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_START;
							enqueueLcdWork();
						}
					}
					break;
				case MODBUS_VOLATILE_CONTROL:
					if(value & (1<<BIT_MODBUS_VOLATILE_DS_CONTROL_DELETE))
					{
						dsDeleteRomCodes(value);
					}
					if(value & (1<<BIT_MODBUS_VOLATILE_DS_CONTROL_SWAP))
					{
						dsSwapRomCodes(value);
					}
					if(value & (1<<BIT_MODBUS_VOLATILE_ALARM_ACTIVE))
					{
						alarmBuzzerActive();
					}
					if(value & (1<<BIT_MODBUS_VOLATILE_ALARM_DEACTIVE))
					{
						alarmBuzzerDeactive();
					}
					break;
				case MODBUS_VOLATILE_SAVE_LIGHT_SCENE:
					mask = 0x01;
					if(value!=0)
					{
						for(indeks = 0; indeks < LIGHT_SCENE_COUNT; indeks++)
						{
							if(value & mask)
							{
								eepromWriteLightScene(indeks, getLightSceneDataAddress());
								break;
							}
							mask <<= 1;
						}
					}						
					break;
				case MODBUS_VOLATILE_DATE_YAER:
					lcdButtonData.modbusSpace[address] = value;
					break;
				case MODBUS_VOLATILE_DATE_MONTH:
					if(value >= 1 && value <= 12)
						lcdButtonData.modbusSpace[address] = value;
					break;
				case MODBUS_VOLATILE_DATE_DAY:
					if(value >= 1 && value <= 31)
						lcdButtonData.modbusSpace[address] = value;
					break;
				case MODBUS_VOLATILE_CLOCK_0_H:
				case MODBUS_VOLATILE_CLOCK_1_H:
				case MODBUS_VOLATILE_CLOCK_2_H:
				case MODBUS_VOLATILE_CLOCK_3_H:
				case MODBUS_VOLATILE_CLOCK_4_H:
					hourTmpValue = value;
					lastHourValueSave = address;
					break;
				case MODBUS_VOLATILE_CLOCK_0_M:
				case MODBUS_VOLATILE_CLOCK_1_M:
				case MODBUS_VOLATILE_CLOCK_2_M:
				case MODBUS_VOLATILE_CLOCK_3_M:
				case MODBUS_VOLATILE_CLOCK_4_M:
					if(lastHourValueSave == address-1)
					{
						time = value;
						time <<= 16;
						time |= hourTmpValue;
						time = time / 1000 / 60;	//czas w minutach
						hours = time / 60;
						minutes = time - hours * 60;					
						lcdButtonData.modbusSpace[address - 1] = hours;
						lcdButtonData.modbusSpace[address] = minutes;
					}
					lastHourValueSave = 0;
					break;				
			}
		}
		else if(address <= MODBUS_SYSTEM_LAST)
		{
			switch(address)
			{
				case MODBUS_SYSTEM_CLOCK_ALARM_H_SET:
				case MODBUS_SYSTEM_CLOCK_ALARM_M_SET:
					break;
				case MODBUS_SYSTEM_BACKLIGHT_ON:
				case MODBUS_SYSTEM_BACKLIGHT_OFF:
					lcdBacklightService(LCD_BACKLIGHT_REFRESH_CMD);
					break;
				case MODBUS_SYSTEM_DARKER_TIME:
					if(readModbus(MODBUS_SYSTEM_DARKER_TIME) != value)
					{
						if(getTimer100msec(TMR_100MSEC_DARKER))
							setTimer100msec(TMR_100MSEC_DARKER, value);
					}
					break;
				case MODBUS_SYSTEM_BUZZER_TIME:
					if(value > 255)
						value = 255;
					lcdButtonData.buzzerTime = value;
					break;
				case MODBUS_SYSTEM_DEFAULT_SCENE:
					if(value >= SCENES_COUNT)
						writeLock = true;
					break;
				case MODBUS_SYSTEM_LANG_USER:
					if(value >= MAX_LANG_COUNT)
						writeLock = true;
					else
					{
						if(lcdButtonData.scene.activeLang != value)
						{
							lcdButtonData.scene.activeLang = value;	
							
							//dodanie do kolejki zadan wyswietlenie na nowo belki systemowej i sceny uzytkownika:
							lcdStateContext = getLcdWorkQueueFree();
							if(lcdStateContext != NULL)
							{
								lcdStateContext->state = LCD_STATE_SERVICE_SCENE_ACTIVE;
								lcdStateContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_START;
								enqueueLcdWork();
							}
						}
					}						
					break;
				case MODBUS_SYSTEM_BIT_DATA:
					if(value & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_ALARM_SET))
					{
						writeSpecialModbus(MODBUS_READ_ONLY_CLOCK_ALARM_H, readModbus(MODBUS_SYSTEM_CLOCK_ALARM_H_SET));
						writeSpecialModbus(MODBUS_READ_ONLY_CLOCK_ALARM_M, readModbus(MODBUS_SYSTEM_CLOCK_ALARM_M_SET));
					}
					else
					{
						writeSpecialModbus(MODBUS_READ_ONLY_CLOCK_ALARM_H, MODBUS_DEF_VAL_CLOCK_ALARM_INACTIVE);
						writeSpecialModbus(MODBUS_READ_ONLY_CLOCK_ALARM_M, MODBUS_DEF_VAL_CLOCK_ALARM_INACTIVE);
					}
					if(value & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_0))
						DO_0_HI();
					else
						DO_0_LOW();
				
					if(value & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_1))					
						DO_1_HI();
					else
						DO_1_LOW();
										
					if(value & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_2))
						DO_2_HI();
					else
						DO_2_LOW();
					
					if(value & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_3))
						DO_3_HI();
					else
						DO_3_LOW();
					break;
				case MODBUS_SYSTEM_UART_SPEED:
					if(ifBaudRateCorrect(value))
					{
						lcdButtonData.newModbusBaudRate = value;
					}
					else
					{
						writeLock = 1;	
					}
					break;
				case MODBUS_SYSTEM_MODBUS_SOFT_ADDR:
					if(!MODBUS_ADDRESS_MODIF_TEST || value == 0 || value > MODBUS_ADDRESS_MAX_VALUE)
					{
						writeLock = 1;
					}
					else
					{
						lcdButtonData.newModbusAddress = (uint8_t)value;
					}					
					break;
				case MODBUS_SYSTEM_SHOW_LIGHT_SCENE:
					if(value == 0)
					{
						for(int i = MODBUS_SYSTEM_LIGHT_SCENE_REG_FIRST; i <= MODBUS_SYSTEM_LIGHT_SCENE_REG_LAST; i++)
						{
							lcdButtonData.modbusSpace[i] = 0;
						}
					}
					else
					{
						mask = 0x01;
						for(indeks = 0; indeks < LIGHT_SCENE_COUNT; indeks++)
						{
							if(value & mask)
							{
								eepromReadLightScene(indeks, getLightSceneDataAddress());
								break;
							}
							mask <<= 1;
						}
					}
					break;
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_0:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_1:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_2:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_3:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_4:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_5:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_6:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_7:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_8:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_9:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_A:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_B:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_C:
				case MODBUS_SYSTEM_LIGHT_SCENE_REG_D:
					break;
				case MODBUS_SYSTEM_PWM_0_VALUE:
					if(value > 255)
						value = 255;
					OCR0A = (uint8_t)value;
					break;
				case MODBUS_SYSTEM_PWM_1_VALUE:
					if(value > 255)
					value = 255;
					OCR0B = (uint8_t)value;
					break;
				case MODBUS_READ_ONLY_EXTRA_BIT_DATA:
					writeLock = 1;
					break;
			}
			if(!writeLock)
			{
				lcdButtonData.modbusSpace[address] = value;
				eepromService(LCD_EEPROM_SAVE);
			}
		}				
		else if(address < MODBUS_SIZE)
		{
			lcdButtonData.modbusSpace[address]  = value;					
			eepromService(LCD_EEPROM_SAVE);
		}
	}			
}

uint16_t * getLightSceneDataAddress()
{
	return &lcdButtonData.modbusSpace[MODBUS_SYSTEM_LIGHT_SCENE_REG_0];
}

void buttonsIntegrity(void)
{
	_lcdStateContext * lcdStateContext = getLcdWorkQueueFree();
	if(lcdButtonData.scene.sceneReady && lcdStateContext != NULL)
	{	
		uint8_t minIndex, maxIndex;
		minIndex = BIG_OBJECTS_MAX_COUNT;
		maxIndex = 0;
		_buttonData * button;		
		button = lcdButtonData.scene.user.firstButton;
		for(int i=0; i<lcdButtonData.scene.user.buttonsCount; i++)
		{
			uint16_t regValue;
			if(button->type == BUT_CODE_INDEP ||
				button->type == BUT_CODE_INDEP_OFF ||
				button->type == BUT_CODE_BLIND)
			{
				regValue = readModbus(button->data[0]);
				if(button->value != (regValue & ((uint16_t)1<<button->data[1])))
				{				
					button->value = (regValue & ((uint16_t)1<<button->data[1]));
					if(i<minIndex)
						minIndex = i;
					if(i>maxIndex)
						maxIndex = i;
				}
			}
			else if(button->type == BUT_CODE_VALUE)
			{
				regValue = readModbus(button->data[0]);
				uint16_t tmpVal = button->data[1] | (((uint16_t)(button->data[2]))<<8);
				regValue = regValue == tmpVal;
				if(button->value != regValue)
				{
					button->value = regValue;
					if(i<minIndex)
						minIndex = i;
					if(i>maxIndex)
						maxIndex = i;
				}
			}
			button++;
		}
		if(minIndex<=maxIndex)
		{
			lcdStateContext->data[0] = minIndex;								//indeks obiektu od ktorego rozpoczac wyswietlanie
			lcdStateContext->data[1] = maxIndex-minIndex+1;								//ilosc obiektow do wyswietlenia
			lcdStateContext->state = LCD_STATE_PRINT_BUTTONS;
			lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
			enqueueLcdWork();
		}
    }		
} 

void barsIntegrity(void)
{
	_lcdStateContext * lcdStateContext;
	uint8_t minIndex, maxIndex;
	_barData * barData;
	//przestrzen systemowa:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdButtonData.scene.sceneReady && lcdStateContext != NULL)
	{
		minIndex = BIG_OBJECTS_MAX_COUNT;
		maxIndex = 0;
		barData = lcdButtonData.scene.system.firstBar;
		for(int i=0; i<lcdButtonData.scene.system.barsCount; i++)
		{
			uint16_t regValue = readModbus(barData->mbWord);
			if(barData->value != regValue)
			{
				barData->value = regValue;
				if(i<minIndex)
				minIndex = i;
				if(i>maxIndex)
				maxIndex = i;
			}
			barData++;
		}
		if(minIndex<=maxIndex)
		{
			lcdStateContext->data[0] = minIndex;								//indeks obiektu od ktorego rozpoczac wyswietlanie
			lcdStateContext->data[1] = maxIndex-minIndex+1;								//ilosc obiektow do wyswietlenia
			lcdStateContext->state = LCD_STATE_SERVICE_PRINT_BARS;
			lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
			enqueueLcdWork();
		}
	}
	//przestrzen uzytkownika:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdButtonData.scene.sceneReady && lcdStateContext != NULL)
	{
		minIndex = BIG_OBJECTS_MAX_COUNT;
		maxIndex = 0;
		barData = lcdButtonData.scene.user.firstBar;
		for(int i=0; i<lcdButtonData.scene.user.barsCount; i++)
		{
			uint16_t regValue = readModbus(barData->mbWord);
			if(barData->value != regValue)
			{
				barData->value = regValue;
				if(i<minIndex)
					minIndex = i;
				if(i>maxIndex)
					maxIndex = i;
			}
			barData++;
		}
		if(minIndex<=maxIndex)
		{
			lcdStateContext->data[0] = minIndex;								//indeks obiektu od ktorego rozpoczac wyswietlanie
			lcdStateContext->data[1] = maxIndex-minIndex+1;								//ilosc obiektow do wyswietlenia
			lcdStateContext->state = LCD_STATE_PRINT_BARS;
			lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
			enqueueLcdWork();
		}
	}		
} 


void statesIntegrity(void)
{
	uint8_t minIndex, maxIndex;
	_lcdStateContext * lcdStateContext;
	_stateData * stateData;
	//przestrzen systemowa:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdButtonData.scene.sceneReady && lcdStateContext != NULL)
	{
		minIndex = BIG_OBJECTS_MAX_COUNT;
		maxIndex = 0;
		stateData = lcdButtonData.scene.system.firstState;
		for(int i=0; i<lcdButtonData.scene.system.statesCount; i++)
		{
			uint16_t regValue = ((readModbus(stateData->mbWord) & (1<<(stateData->mbBit))) != 0);
			if(stateData->value != regValue)
			{
				stateData->value = regValue;
				if(i<minIndex)
					minIndex = i;
				if(i>maxIndex)
					maxIndex = i;
			}
			stateData++;
		}
		if(minIndex<=maxIndex)
		{
			lcdStateContext->data[0] = minIndex;								//indeks obiektu od ktorego rozpoczac wyswietlanie
			lcdStateContext->data[1] = maxIndex-minIndex+1;								//ilosc obiektow do wyswietlenia
			lcdStateContext->state = LCD_STATE_SERVICE_PRINT_STATES;
			lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
			enqueueLcdWork();
		}
	}
	//przestrzen użytkownika:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdButtonData.scene.sceneReady && lcdStateContext != NULL)
	{	
		minIndex = BIG_OBJECTS_MAX_COUNT;
		maxIndex = 0;
		stateData = lcdButtonData.scene.user.firstState;
		for(int i=0; i<lcdButtonData.scene.user.statesCount; i++)
		{
			uint16_t regValue = ((readModbus(stateData->mbWord) & (1<<(stateData->mbBit))) != 0);
			if(stateData->value != regValue)
			{
				if(stateData->mbWord >= MODBUS_READ_ONLY_END)
				{
					stateData->value = regValue;
					if(i<minIndex)
						minIndex = i;
					if(i>maxIndex)
						maxIndex = i;	
				}
			}
			stateData++;
		}
		if(minIndex<=maxIndex)
		{
			lcdStateContext->data[0] = minIndex;								//indeks obiektu od ktorego rozpoczac wyswietlanie
			lcdStateContext->data[1] = maxIndex-minIndex+1;								//ilosc obiektow do wyswietlenia
			lcdStateContext->state = LCD_STATE_PRINT_STATES;
			lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
			enqueueLcdWork();
		}
	}		
} 


void numbersIntegrity()
{
	_lcdStateContext * lcdStateContext;
	uint8_t minIndex, maxIndex;
	_numberData * numberData;
	int16_t modbusValue;
	//przestrzen systemowa:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdButtonData.scene.sceneReady && lcdStateContext != NULL)
	{		
		minIndex = BIG_OBJECTS_MAX_COUNT;
		maxIndex = 0;
		numberData = lcdButtonData.scene.system.firstNumber;		
		for(int i=0; i<lcdButtonData.scene.system.numbersCount; i++)
		{
			modbusValue = readModbus(numberData[i].mbWord);
			if(numberData[i].value != modbusValue)
			{
				numberData[i].value = modbusValue;
				if(i<minIndex)
					minIndex = i;
				if(i>maxIndex)
					maxIndex = i;	
			}
		}
		if(minIndex<=maxIndex)
		{
			lcdStateContext->data[0] = minIndex;								//indeks obiektu od ktorego rozpoczac wyswietlanie
			lcdStateContext->data[1] = maxIndex-minIndex+1;						//ilosc obiektow do wyswietlenia
			lcdStateContext->state = LCD_STATE_SERVICE_PRINT_NUMBERS;
			lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
			enqueueLcdWork();
		}
	}
	//przestrzen uzytkownika:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdButtonData.scene.sceneReady && lcdStateContext != NULL)
	{	
		minIndex = BIG_OBJECTS_MAX_COUNT;
		maxIndex = 0;
		numberData = lcdButtonData.scene.user.firstNumber;
		for(int i=0; i<lcdButtonData.scene.user.numbersCount; i++)
		{
			modbusValue = readModbus(numberData[i].mbWord);
			if(numberData[i].value != modbusValue)
			{
				numberData[i].value = modbusValue;
				if(i<minIndex)
					minIndex = i;
				if(i>maxIndex)
					maxIndex = i;	
			}
		}
		if(minIndex<=maxIndex)
		{
			lcdStateContext->data[0] = minIndex;								//indeks obiektu od ktorego rozpoczac wyswietlanie
			lcdStateContext->data[1] = maxIndex-minIndex+1;						//ilosc obiektow do wyswietlenia
			lcdStateContext->state = LCD_STATE_PRINT_NUMBERS;
			lcdStateContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_START;
			enqueueLcdWork();
		}
	}		
} 

uint32_t seedRandom = DEFAULT_SEED;

uint8_t getRandomInt(uint8_t upperLimit)
{	
	static uint8_t seedInitialize = 0;		/*!< flaga sygnalizujaca, ze zarodek zostal zainicjalizowany*/

	if(seedInitialize == 0)
	{
		srandom(seedRandom);
		seedInitialize = 1;
	}
	return random() %  (uint32_t)upperLimit;
}