﻿/*!
 *  @file lcdButton.h
 *	@brief Deklaracje głównych typów danych
 */ 
#ifndef LCDBUTTON_H_
#define LCDBUTTON_H_

#include <avr/io.h>
#include "protocolWago.h"
#include "uart.h"
#include "pid.h"

/*! 
	@def DEFAULT_SEED
    @brief Wartość początkowa dla inicjalizacji zarodka generatora liczb pseudolosowych
*/
#define DEFAULT_SEED	547899481

/*! 
	@def SCENES_COUNT
    @brief Ilość możliwych do zaprogramowania scen
*/
#define SCENES_COUNT 64

/*!
	@def BIG_OBJECTS_MAX_COUNT
	@brief Maksymalna ilość większych obiektów (LABEL, BAR, NUMBER, BUTTON) jakie mogą zmieścić się na jednej scenie
*/
#define BIG_OBJECTS_MAX_COUNT 63

/*!
	@def SMALL_OBJECTS_MAX_COUNT
	@brief Maksymalna ilość mniejszych obiektów (IMAGE) jakie mogą zmieścić się na jednej scenie
*/
#define SMALL_OBJECTS_MAX_COUNT 4*BIG_OBJECTS_MAX_COUNT

/*!
	@def MEDIUM_OBJECTS_MAX_COUNT
	@brief Maksymalna ilość średnich obiektów (STATE) jakie mogą zmieścić się na jednej scenie
*/
#define MEDIUM_OBJECTS_MAX_COUNT 2*BIG_OBJECTS_MAX_COUNT

/*! 
	@def MAX_LANG_COUNT
    @brief Maksymalna ilość wersji językowych
*/
#define MAX_LANG_COUNT 8

/*! 
	@def LIGHT_SCEN_COUNT
    @brief Ilość scen świetlnych
*/
#define LIGHT_SCENE_COUNT 16

/*! 
	@def WAKE_UP_REGISTERS_COUNT
    @brief Ilość scen rejestrow ktore sa aktualizowane na skutek wybudzenia urzadzenia
*/
#define WAKE_UP_REGISTERS_COUNT 16

/*!
	@enum _clickedState
	@brief Stany związane z wciśnięciem/nie wciśnięciem przycisku.
*/
typedef enum {
	CLICK_NONE = 0,		//!< Brak danych nt kliknięcia lub brak kliknięcia
	CLICK_EMPTY,		//!< Kliknięcie w obszar nie należący do żadnego przycisku
	CLICK_SHORT,		//!< Pojedyncze klikniecie
	CLICK_LONG,			//!< Przytrzymanie przez dluzsza chwile przycisku
}_clickedState;

/*!
	@enum _buttonType
	@brief Typy przycisków
*/
typedef enum {
	BUT_CODE_SCENE = 0xB0,        //SceneButton
	BUT_CODE_PASS_NUM = 0xB1,     //PassButton
	BUT_CODE_PASS_CLR = 0xB2,     //PassClearButton
	BUT_CODE_PASS_SET = 0xB3,     //PassConfirmButton
	BUT_CODE_EMPTY = 0xB4,        //EmptyButton
	BUT_CODE_INDEP = 0xB5,        //IndepButton
	BUT_CODE_INDEP_OFF = 0xB6,    //IndepAutoOffButton
	BUT_CODE_VALUE = 0xB7,        //ValueButton - wlaczenie danego przycisku powoduje ustawienie we wskazanym rejestrze właściwej dla danego przycisku wartości
	BUT_CODE_BLIND = 0xB8,		  //BlindDownButton
	BUT_CODE_ROCKER = 0xBA,		  //RockerDownButton
	
}_buttonType;


/*!
	@struct _buttonData
	@brief Podstawowe dane pojedynczego obiektu BUTTON.
	
	Struktura przechowująca podstawowe dane dotyczące pojedynczygo obiektu BUTTON.
*/
typedef struct{
	/// Współrzędna X lewego górnego rogu
	uint16_t	xLGR;
	/// Współrzędna Y lewego górnego rogu
	uint16_t yLGR;
	/// Współrzędna X prawego dolnego rogu
	uint16_t xPDR;
	/// Współrzędna Y prawego dolnego rogu
	uint16_t yPDR;
	// TODO: zamienic na enum!
	/// Typ przycisku
	uint8_t type;
	/// Dane zależne od pola type
	uint8_t data[9];
	/// Timer odliczajacy czas specyficzny dla danego przycisku
	uint16_t timedown;
	uint8_t timedown2;
	uint16_t value;
}_buttonData;

/*!
	@struct _numberData
	@brief Podstawowe dane pojedynczego obiektu NUMBER.
	
	Struktura przechowująca podstawowe dane dotyczące pojedynczygo obiektu NUMBER.
*/
typedef struct{
	/// Współrzędna X
	uint16_t x;
	/// Współrzędna Y
	uint16_t y;
	// TODO: zamienić na pole typu enum
	/// Status obiektu
	uint8_t status;
	/// Minimalna wartość jaką może mieć dany obiekt
	int16_t min;
	/// Maksymalna wartość jaką może mieć dany obiekt
	int16_t max;
	/// Identyfikator grafiki
	uint16_t graphId;
	/// Adres modbus rejestru
	uint8_t mbWord;
	int16_t value;
}_numberData;

/*!
	@struct _barData
	@brief Podstawowe dane pojedynczego obiektu BAR.
	
	Struktura przechowująca podstawowe dane dotyczące pojedynczygo obiektu BAR.
*/
typedef struct{
	/// Współrzędna X
	uint16_t x;
	/// Współrzędna Y
	uint16_t y;
	// TODO: zamienić na pole typu enum
	/// niewykorzystywane
	uint8_t status;
	/// Minimalna wartość jaką może mieć dany obiekt
	int16_t min;
	/// Maksymalna wartość jaką może mieć dany obiekt
	int16_t max;
	/// Identyfikator grafiki
	uint16_t graphId;
	/// Adres modbus rejestru
	uint8_t mbWord;
	int16_t value;
}_barData;

/*!
	@struct _stateData
	@brief Podstawowe dane pojedynczego obiektu STATE.
	
	Struktura przechowująca podstawowe dane dotyczące pojedynczygo obiektu STATE.
*/
typedef struct{
	/// Współrzędna X
	uint16_t x;
	/// Współrzędna Y
	uint16_t y;
	// TODO: zamienić na pole typu enum
	/// niewykorzystywane
	uint8_t status;
	/// Identyfikator grafiki
	uint16_t graphId;
	/// Adres modbus rejestru
	uint8_t mbWord;
	/// Numer bitu: 0-LSB; 15-MSB
	uint8_t mbBit;
	/// true - stan on, false - stan off
	uint8_t value;
}_stateData;


/*!
	@struct _lcdButtonScene
	@brief Struktura przechowująca dane wykorzystywane przy wyświetlaniu sceny
*/
typedef struct
{	
	uint8_t labelsCount;			/**< ilość grafik na scenie*/
	uint8_t textsCount;				/**< ilość tekstów na scenie*/
	uint8_t barsCount;				/**< ilość obiektów BAR na scenie*/
	uint8_t statesCount;			/**< ilość obiektów STATE na scenie*/
	uint8_t numbersCount;			/**< ilość obiektów NUMBER na scenie*/
	uint8_t buttonsCount;			/**< ilość obiektów BUTTON na scenie*/
	uint8_t textsLengths[BIG_OBJECTS_MAX_COUNT];		/**< maksymalne dlugosci tekstow (najpierw teksty ogólne, później teksty przycisków)*/

	_barData * firstBar;			/**< wskaźnik na pierwszy obiekt BAR*/
	_stateData * firstState;		/**< wskaźnik na pierwszy obiekt STATE*/
	_numberData * firstNumber;		/**< wskaźnik na pierwszy obiekt NUMBER*/
	_buttonData * firstButton;		/**< wskaźnik na pierwszy obiekt BUTTON*/
	_buttonData workData[BIG_OBJECTS_MAX_COUNT];	/**< tablica zawierająca dane sterujące wyświetlanej sceny*/
}_lcdButtonScene;

/*!
	@enum _workState
	@brief Zbiór wartości oznaczających tryb pracy bootloadera.
	@details Zdefiniowane tu wartości można zapisywać poprzez Modbus do rejestru o adresie 13 (funkcja 6)
*/
typedef enum
{
	MODE_NONE			= 0xFFFF,		/*!< Stan nieustalony */
	MODE_ACT_ERR		= 0xF00F,		/*!< Normalny stan bootloadera/blad aktualizacji (szybkie mruganie diody) */
	MODE_ACT			= 0x00FF,		/*!< Aktualizacja oprogramowania (powolne mruganie diody) */
	MODE_ACT_END		= 0xFF00,		/*!< Wyjście z trybu bootloadera */
}_workState;


/*!
	@struct _lcdButtonData
	@brief Struktura zawiera dane
*/
typedef struct
{
	//dane podstawowe
	uint8_t dipswitchValue;				/*!< adres wybrany na dipswitchu */
	_workState deviceWorkState;			/*!< Aktualny tryb pracy urządzenia */
	uint8_t bootDetect;					/*!< 0 - nie wykryto bootloadera; 1 - bootloader wykryty */
	uint8_t appLabel;					/*!< zmienna wymuszajaca umieszczenie znacznika aplikacji w pamieci */
	uint16_t bootSoftVersion;			/*!< wersja programu dzialajacego w ApplicationSection */
	
	uint8_t buzzerTime;					/**< czas pracy buzzera po kliknięciu w przycisk (*10ms)*/
	uint16_t passwordBuild;				/**< zmienna tymczasowa do przechowywania hasla przed jego potwierdzeniem*/
	
	uint8_t tekst[102];
	
	//dane zwiazane z komunikacja z WAGO (protokół modbus)
	uint8_t modbusActive;				/*!< flaga okreslajay czy zostala wykryta komunikacja z modbusem*/
	uint8_t modbusAddress;				/*!< Adres pod jakim urządzenie jest dostępne dla protokołu Modbus */
	uint8_t newModbusAddress;			/*!< Nowy adres urzadzenia jaki zostanie ustawiony po zakonczeniu wysylania odpowiedzi */
	uint16_t modbusSpace[MODBUS_SIZE];	/*!< Przestrzen danych Modbus */
	_baudRate newModbusBaudRate;		/*!< Przechowuje nowa predkosc transmisji jaka zostala ustawiona do momentu zakonczenia wysylania odpowiedzi */
	_baudRate actualModbusBaudRate;		/*!< aktualna predkosc modbus */
		
	//dane zwiazane ze scenami:	
	struct  
	{
		uint8_t sceneReady;				/*!< flaga oznaczająca zakończenie wgrywania nowej sceny i pozwalająca na kontrolę poprzez buttonsIntegrity()*/
		uint8_t activeScene;			/*!< aktualnie aktywna scena*/
		uint8_t activeLang;				/*!< aktualnie wybrany język*/
		uint16_t logoHeight;			/*!< wysokość (w pikselach) grafiki systemowej*/
		_lcdButtonScene user;			/*!< scena/przestrzeń użytkownika*/
		_lcdButtonScene system;			/*!< scena/przestrzeń systemowa*/
	}scene;
	
	//dane zwiazane z dotykiem:
	struct
	{
		//nie uwzgledniono możliwości wystąpienia przycisku na belce systemowej
		uint8_t clickedButtonIndex;		/**< indeks wciśnietego przycisku*/
		_clickedState clickedState;		/**< flaga aktywnego wcisnięcia wyswietlacza*/
	}button;
	
	//wybudzenie urzadzenia:
	uint8_t sceneSwitchFlag;			/*! < Flaga oznaczajaca czy wybudzenie urzadzenia zwiazane jest z przelaczeniem sceny na sceneSwitchNumber */
	uint8_t sceneSwitchNumber;
	uint8_t lightSceneSwitchFlag;		/*! < Flaga oznaczajaca czy wybudzenie urzadzenia zwiazane jest z przelaczeniem sceny swietlnej*/
	uint8_t lightSceneSwitchMask;
	uint8_t newValuesCount;				/*! < Ilosc rejestrow modbus, ktorych wartosc jest zmieniana na skutek wybudzenia*/
	uint8_t wakeUpMbAddresses[WAKE_UP_REGISTERS_COUNT];
	uint16_t wakeUpMbValues[WAKE_UP_REGISTERS_COUNT];
	
	//PID:
	uint8_t pidCount;					/*! < Ilosc uzywanych algorytmow PID */
	SPidData pidData[PID_MAX_COUNT];
	
	//dane zwiazane z wgrywaniem konfiguracji
	struct
	{
		uint8_t newUserConfigFlag;			/*!< flaga oznacza ze rozpoczeto wgrywanie nowej konfiguracji uzytkownika */
		int8_t lastUserSceneSave;			/*!< indeks ostatnio wgrywanej konfiguracji dla sceny uzytkownika */
	}config;
}_lcdButtonData;

/*!
	@brief Inicjalizacja danych
*/
void initData(void);

/*!
	@brief Odczytanie wartości z przestrzenii MODBUS
	@param address Adres MODBUS z którego należy odczytać dane
	@return Odczytane dane
*/
uint16_t readModbus(uint16_t address);

/*!
	@brief Odczytanie wartości z przestrzenii MODBUS dla Input Registers
	@param address Adres MODBUS (Input Register) z którego należy odczytać dane
	@return Odczytane dane
*/
uint16_t readInputRegister(uint16_t address);

/*!
	@brief Zapis do chronionej przestrzeni modbus (z zewnatrz tylko do odczytu)
	@param address Adres rejestru do którego należy dokonać zapisu
	@param val Wartość jaka ma być zapisana pod wskazanym adresem 
*/
void writeSpecialModbus(uint16_t address, uint16_t value);

/*!
	@brief Zapis do przestrzeni modbus
	@param address Adres rejestru do którego należy dokonać zapisu
	@param val Wartość jaka ma być zapisana pod wskazanym adresem 
*/
void writeModbus(uint16_t address, uint16_t val);

/*!
	@brief Zwraca adres do adresu RAM przestrzeni modbus gdzie przechowywane są dane aktualnej sceny świetlnej
	@return adres pamięci RAM
*/
uint16_t * getLightSceneDataAddress();

/*!
	@brief Badanie spójności obiektów BUTTON z wyświetlonym obrazem
*/
void buttonsIntegrity(void);

/*!
	@brief Badanie spójności obiektów RAR z wyświetlonym obrazem
*/
void barsIntegrity(void);

/*!
	@brief Badanie spójności obiektów NUMBER z wyświetlonym obrazem
*/
void numbersIntegrity(void);

/*!
	@brief Badanie spójności obiektów STATE z wyświetlonym obrazem
*/
void statesIntegrity(void);

/*!
	@brief Obsługa liczb pseudolosowych
	@details Funkcja zwraca pseudolosową liczbę z zakresu <0; upperLimit).
	Funkcja przy pierwszym jej urzyciu inicjalizuje zarodek generatora. W przypadku gdy nie został jeszcze pobrany
	aktualny czas (przez modbus) zarodek jest inicjalizowany stałą wartością.
*/
uint8_t getRandomInt(uint8_t upperLimit);

extern uint32_t seedRandom;

#endif /* LCDBUTTON_H_ */
