﻿#include <stdint.h>
#include <avr/io.h>
#include "lcdButtonTimers.h"
#include "processor.h"
#include "main.h"
#include "lcd.h"

/*!
	Flaga wykorzystywana do odmierzania stałych odstępów czasowych.
*/
volatile uint8_t timeMsFlag;

/*!
	@brief Tablica programowo zaimplementowanych timerów inkrementowanych na podstawie flagi timeMsFlag.
	@details Timery imkrementowane są co czas określony przez stałą @ref TIMEOUT_UNIT.
*/ 
uint8_t timersShort[TMR_SHORT_ALL];

/*!
	@brief Tablica programowych timerów odliczających każde 100ms.
*/
uint16_t timers100msec[TMR_100MSEC_ALL];

/*!
	@brief Tablica programowo zaimplementowanych timerów inkrementowanych co 1 sekundę.
*/ 
uint8_t timers1Sec[TMR_1SEC_ALL];

/*!
	@brief Tablica programowo zaimplementowanych timerów inkrementowanych co 1 minutę.
*/ 
uint8_t timers1Min[TMR_1MIN_ALL];

void initTimers( void )
{
	//wyzerowanie programowych timerow:
	for(int i=0; i<TMR_SHORT_ALL; i++)
		timersShort[i]=0;
	for(int i=0; i<TMR_1SEC_ALL; i++)
		timers1Sec[i]=0;
	for(int i=0; i<TMR_1MIN_ALL; i++)
		timers1Min[i]=0;
}

void setTimerShort(_timersShort timerIndex, uint8_t value)
{
	timersShort[timerIndex] = value;
}

uint8_t getTimerShort(_timersShort timerIndex)
{
	return timersShort[timerIndex];
}

void setTimer100msec(_timers100msec timerIndex, uint16_t value)
{
	timers100msec[timerIndex] = value;
}

uint16_t getTimer100msec(_timers100msec timerIndex)
{
	return timers100msec[timerIndex];
}


void setTimer1Sec(_timers1Sec timerIndex, uint8_t value)
{
	timers1Sec[timerIndex] = value;
}

uint8_t getTimer1Sec(_timers1Sec timerIndex)
{
	return timers1Sec[timerIndex];
}

void setTimer1Min(_timers1Min timerIndex, uint8_t value)
{
	timers1Min[timerIndex] = value;
}

uint8_t getTimer1Min(_timers1Min timerIndex)
{
	return timers1Min[timerIndex];
}

void timersService( void )
{
	//timer 5ms:
	if(timeMsFlag)
	{
		timeMsFlag = 0;
		for(int i=0; i<TMR_SHORT_ALL; i++)
		{
			if(timersShort[i])
			{
				timersShort[i]--;
			}
		}
	}
	
	//timer 20ms:
	if(!getTimerShort(TMR_SHORT_TIME_20_MSEC))
	{
		setTimerShort(TMR_SHORT_TIME_20_MSEC, 20/TIMEOUT_UNIT);
		seedRandom++;			// zwiekszanie zarodka
		_buttonData * buttonData = lcdButtonData.scene.user.firstButton;
		for(int i=0; i<lcdButtonData.scene.user.buttonsCount; i++)
		{
			if(buttonData->type == BUT_CODE_ROCKER && buttonData->value)
			{
				int8_t step;
				if(buttonData->timedown2)
				{
					step = (int8_t)buttonData->data[1];
					buttonData->timedown2--;
				}
				else
					step = (int8_t)buttonData->data[2];
				if(buttonData->timedown)
				{
					buttonData->timedown--;
					if(!buttonData->timedown)
					{
						uint16_t data = readModbus(buttonData->data[0]) + step;
						if((int16_t)data < *((int16_t *)(buttonData->data + 3)))
							data = *((int16_t *)(buttonData->data + 3));
						if((int16_t)data > *((int16_t *)(buttonData->data + 5)))
							data = *((int16_t *)(buttonData->data + 5));
						writeModbus(buttonData->data[0], data);
						buttonData->timedown = buttonData->data[7];
					}
				}				
			}
			buttonData++;
		}
	}
	
	
	
	//timer 100ms:
	if(!getTimerShort(TMR_SHORT_TIME_100_MSEC))
	{
		setTimerShort(TMR_SHORT_TIME_100_MSEC, TIME_100_MSEC_TIMEOUT);
		for(int i=0; i<TMR_100MSEC_ALL; i++)
		{
			if(timers100msec[i])
			{
				timers100msec[i]--;
			}
		}
		_buttonData * button = lcdButtonData.scene.user.firstButton;
		uint8_t reg, bit;
		uint16_t data;
		for(int i=0; i<lcdButtonData.scene.user.buttonsCount; i++)
		{
			if(button->type != BUT_CODE_ROCKER)
			{
				if(button->timedown)
				{
					button->timedown--;
					if(!button->timedown)
					{
						reg = button->data[0];
						bit = button->data[1];
						switch (button->type)
						{
							case BUT_CODE_INDEP_OFF:
							case BUT_CODE_BLIND:
								data = readModbus(reg) & (~((uint16_t)1<<bit));
								writeModbus(reg, data);
								break;
						}
					}
				}
			}
			button++;
		}
	}
	
	
	
	//timer sekundowy:
	if(!getTimerShort(TMR_SHORT_TIME_1_SEC))
	{	
		setTimerShort(TMR_SHORT_TIME_1_SEC, TIME_1_SEC_TIMEOUT);
		for(int i=0; i<TMR_1SEC_ALL; i++)
		{
			if(timers1Sec[i])
			{
				timers1Sec[i]--;
			}
		}
	}
	
	//timer minutowy:
	if(!getTimer1Sec(TMR_1SEC_TIME_1_MIN))
	{
		setTimer1Sec(TMR_1SEC_TIME_1_MIN, TIME_1_MIN_TIMEOUT);
		for(int i=0; i<TMR_1MIN_ALL; i++)
		{
			if(timers1Min[i])
			{
				timers1Min[i]--;
			}
		}
	}
}
