﻿/*!
 *  @file lcdButtonTimers.h
 *	@brief Obsługa timera sprzętowego i timerów programowych
 *	@details Timer sprzętowy odpowiada za odmierzanie stałych kwantów czasowych na podstawie których
 *	odliczane są timery programowe.
 */ 

#ifndef LCDBUTTONTIMERS_H_
#define LCDBUTTONTIMERS_H_

#include <stdint.h>

/*! 
	@def TIMER_PRESCALER
	@brief Preskaler timera sprzętowego
*/
#define TIMER_PRESCALER	1024

/*! 
	@def TIMEOUT_UNIT
    @brief Wartość jednostki czasu timerów programowych [ms]
	@details Stała określa jaki jest odstęp czasowy (wyrażony w milisekundach) pomiędzy kolejnymi ikrementacjami
	timerów programowych. Jeśli timer programowy ustawiony jest na 10, to łączny jego czas wynosi 10 * @ref TIMEOUT_UNIT [ms].
	@n Rzeczywisty odstęp czasowy obliczyć można ze wzoru:
	@n R = @ref TIMER_PRESCALER * (@ref TIMER_VALUE + 1) / @ref F_CPU * 1000 [ms]
	@n Różnicę między czasem zadanym a rzeczywistym można obliczyć ze wzoru:
	@n (@ref TIMEOUT_UNIT - R) / @ref TIMEOUT_UNIT * 100 [%]
	@n @b UWAGA @ref TIMEOUT_UNIT przyjmuje wartość od 4 do 14 włącznie.
*/
#define TIMEOUT_UNIT		5

/*! 
	@def TIMEOUT_LCD_NO_RESP
    @brief Ilość dekrementacji timera określająca czas w jakim oczekiwuje się uzyskać odpowiedź od wyświetlacza LCD
*/
#define TIMEOUT_LCD_NO_RESP	200/TIMEOUT_UNIT

/*!
	@brief Dluzszy czas oczekiwania na odpowiedz (wgrywanie danych do wyswietlacza) 
*/
#define TIMEOUT_LCD_NO_RESP_LONG	255

/*! 
	@def TIMER_VALUE
	@brief Wartość odliczana przez timer sprzętowy
*/
#define TIMER_VALUE		F_CPU*TIMEOUT_UNIT/TIMER_PRESCALER/1000 - 1

/*! 
	@def UART_WAGO_TIMEOUT
    @brief Maksymalny czas między kolejnymi bajtami przy odbieraniu komunikatu
	na porcie komunkującym się z WAGO
	
	Łączny czas: @ref TIMEOUT_UNIT * @ref UART_WAGO_TIMEOUT [ms]
*/
#define UART_WAGO_TIMEOUT	10

/*! 
	@def UART_LCD_TIMEOUT
    @brief Maksymalny czas między kolejnymi bajtami przy odbieraniu komunikatu
	na porcie komunkującym się z wyświetlaczem LCD
	@details Łączny czas: @ref TIMEOUT_UNIT * @ref UART_LCD_TIMEOUT [ms]
*/
#define UART_LCD_TIMEOUT	2

/**
 * @def LCD_CLICK_TIMEOUT
 * @brief Czas zabezpieczający przed nie odczytaniem komunikatu zwalniającego wcinięcie wywietlacza LCD
 */
#define LCD_CLICK_TIMEOUT	600/TIMEOUT_UNIT

/*! 
	@def TIME_100_MSEC_TIMEOUT
    @brief 100 milisekund
	@details Łączny czas: @ref TIMEOUT_UNIT * @ref TIME_100_MSEC_TIMEOUT [ms]
*/
#define TIME_100_MSEC_TIMEOUT	100/TIMEOUT_UNIT

/*! 
	@def TIME_1_SEC_TIMEOUT
    @brief 1 sekunda	
	@details Łączny czas: @ref TIMEOUT_UNIT * @ref TIME_1_SEC_TIMEOUT [ms]
*/
#define TIME_1_SEC_TIMEOUT	1000/TIMEOUT_UNIT

/*! 
	@def TIME_1_MIN_TIMEOUT
    @brief 1 minuta
*/
#define TIME_1_MIN_TIMEOUT	60

/*! 
	@def TMR_1SEC_WAGO_NO_CONNECT_TIMEOUT
    @brief Wykrywanie braku polaczenia z kontrolerem WAGO. Czas jest odliczany od odebrania komunikatu od WAGO
*/
#define TMR_1SEC_WAGO_NO_CONNECT_TIMEOUT	30

/*! 
	@def TMR_1SEC_HI_PRIO_DELAY_TIMEOUT
    @brief Czas przez jaki pomijane sa dodatkowe funkcjonalnosci po otrzymaniu ramki zapisujacej dane konfiguracyjne lub grafiki
*/
#define TMR_1SEC_HI_PRIO_DELAY_TIMEOUT	5


/*!
	@enum _timersShort
	@brief Lista programowo zaimplementowanych timerów inkrementowanych co @ref TIMEOUT_UNIT [ms]
*/
typedef enum {
	TMR_SHORT_UART0 = 0,		/*!< Timer programowy wykrywający koniec komunikacji na porcie UART0 */
	TMR_SHORT_UART1,			/*!< Timer programowy wykrywający koniec komunikacji na porcie UART1 */
	TMR_SHORT_TIME_20_MSEC,		/*!< Timer programowy wykrywający 20 milisekund */
	TMR_SHORT_TIME_100_MSEC,	/*!< Timer programowy wykrywający 100 milisekund */
	TMR_SHORT_TIME_1_SEC,		/*!< Timer programowy wykrywający 1 sekundę */
	TMR_SHORT_LCD_NO_RESP,		/*!< Timer programowy wykrywający brak odpowiedzi od wyswietlacza LCD */
	TMR_SHORT_LCD_CLICK,		/*!< Timer programowy zabezpieczający przed niewykryciem ramki kończącej wcisniety wywietlacz LCD */
	TMR_SHORT_LED,				/*!< Timer programowy do obslugi przerwan czasowych przy obsludze diody LED */
	TMR_SHORT_BUZZER_ALARM,		/*!< Odliczanie czasu pomiędzy kolejnymi sygnałami pracującego alarmu*/
	TMR_SHORT_PID,				/*!< Timer programowy odliczajacy kolejne pomiary*/
	TMR_SHORT_ALL,				/*!< Ilość wszystkich timerów programowych */
}_timersShort;

/*!
	@enum _timers100mSec
	@brief Lista programowo zaimplementowanych timerów inkrementowanych co 100 milisekund
*/
typedef enum {
	TMR_100MSEC_DARKER,					/*!< Odliczanie czasu do sciemnienia wyswietlacza, patrz adres modbus MODBUS_SYSTEM_DARKER_TIME*/
	TMR_100MSEC_SAVE_TO_EEPROM,			/*!< Odliczanie czasu do kolejnego zapisu do EEPROM*/
	TMR_100MSEC_CONVERT_TEMP_DELAY,		/*!< Odliczanie czasu pomiędzy kolejnymi pomiarami temperatury*/
	TMR_100MSEC_CONVERT_TEMP,			/*!< Odliczanie czasu pomiaru temperatury*/
	/// Ilość wszystkich timerów jednosekundowych
	TMR_100MSEC_ALL,
}_timers100msec;


/*!
	@enum _timers1Sec
	@brief Lista programowo zaimplementowanych timerów inkrementowanych co 1 sekundę
*/
typedef enum {
	TMR_1SEC_INIT_LCD,					/*!<  Odczekanie na uruchomienie wyświetlacza LCD*/
	TMR_1SEC_TIME_1_MIN,				/*!< Timer programowy wykrywający 1 minute */
	TMR_1SEC_HI_PRIO_DELAY,				/*!< Wykonywane jest zadanie o wysokim priorytecie i nalezy odliczyc czas przerwy innych czynnosci */
	TMR_1SEC_WAGO_NO_CONNECT,			/*!< do wykrywania ogolnego braku komunikacji z kontrolerem WAGO */
	/// Ilość wszystkich timerów jednosekundowych
	TMR_1SEC_ALL,
}_timers1Sec;

/*!
	@enum _timers1Min
	@brief Lista programowo zaimplementowanych timerów inkrementowanych co 1 minutę
*/
typedef enum {
	TMR_1MIN_TMP,
	/// Ilość wszystkich timerów jednominutowych
	TMR_1MIN_ALL,
}_timers1Min;

/*!
	@fn void initTimers(void)
    @brief Inicjalizacja timerów programowych oraz timera sprzętowego odmierzającego
	stałe odstępy czasowe ( @ref TIMEOUT_UNIT ).
*/
void initTimers(void);

/*!
	@fn void setTimerShort(_timersShort timerIndex, uint8_t value)
    @brief Ustawienie wartości do odmierzenia na timerze programowym
	@param timerIndex Timer, jaki należy ustawić
	@param value Wartość, jaką należy rozpocząć odliczać
*/
void setTimerShort(_timersShort timerIndex, uint8_t value);

/*!
	@fn uint8_t getTimerShort(_timersShort timerIndex)
    @brief Pobranie czasu, jaki jeszcze pozostał do końca odliczania na wskazanym timerze
	@param timerIndex Timer, który zostanie sprawdzony
*/
uint8_t getTimerShort(_timersShort timerIndex);


/*!
	@fn void setTimer1Sec(_timers1Sec timerIndex, uint8_t value)
    @brief Ustawienie wartości do odmierzenia na timerze jednosekundowym
	@param timerIndex Timer, jaki należy ustawić
	@param value Wartość, jaką należy rozpocząć odliczać
*/
void setTimer100msec(_timers100msec timerIndex, uint16_t value);

/*!
	@fn uint8_t getTimer1Sec(_timers1Sec timerIndex)
    @brief Pobranie czasu, jaki jeszcze pozostał do końca odliczania na wskazanym timerze jednosekundowym
	@param timerIndex Timer, który zostanie sprawdzony
*/
uint16_t getTimer100msec(_timers100msec timerIndex);




/*!
	@fn void setTimer1Sec(_timers1Sec timerIndex, uint8_t value)
    @brief Ustawienie wartości do odmierzenia na timerze jednosekundowym
	@param timerIndex Timer, jaki należy ustawić
	@param value Wartość, jaką należy rozpocząć odliczać
*/
void setTimer1Sec(_timers1Sec timerIndex, uint8_t value);

/*!
	@fn uint8_t getTimer1Sec(_timers1Sec timerIndex)
    @brief Pobranie czasu, jaki jeszcze pozostał do końca odliczania na wskazanym timerze jednosekundowym
	@param timerIndex Timer, który zostanie sprawdzony
*/
uint8_t getTimer1Sec(_timers1Sec timerIndex);

/*!
    @brief Ustawienie wartości do odmierzenia na timerze jednominutowym
	@param timerIndex Timer, jaki należy ustawić
	@param value Wartość, jaką należy rozpocząć odliczać
*/
void setTimer1Min(_timers1Min timerIndex, uint8_t value);

/*!
    @brief Pobranie czasu, jaki jeszcze pozostał do końca odliczania na wskazanym timerze jednominutowym
	@param timerIndex Timer, który zostanie sprawdzony
*/
uint8_t getTimer1Min(_timers1Min timerIndex);

/*!
	@fn void timersService(void)
    @brief Funkcja realizująca programowe timery
	
	@b UWAGA Funkcja ta musi być wywoływana możliwie jak najczęściej (najlepiej w przebiegu pętli głównej progrmu).
*/
void timersService(void);

extern volatile uint8_t timeMsFlag;

#endif /* LCDBUTTONTIMERS_H_ */
