﻿/*!
	@mainpage
	@author
		Marek Swatowski @n marek.swatowski@gmail.com
	@version
		1.0
	@date
		01.01.2012
*/

#include "main.h"

#include "compilationFlags.h"

#include "lcdButton.h"
#include "lcdButtonTimers.h"
#include "lcd.h"
#include "interrupts.h"
#include "uart.h"
#include "crc.h"
#include "protocolLcd.h"
#include "lcdButton.h"
#include "processor.h"
#include "led.h"
#include "ds.h"
#include "eemem.h"
#include "adc.h"
#include "dipswitch.h"

//  FUSES ={
// 	 {
//  		0xFF,					// FUSEBYTE1:				 
// 		0xFF,					// FUSEBYTE2:		
//  		0x91,					// FUSEBYTE0:
// 		0xFF,					// FUSEBYTE2:
// 		0x91,					// FUSEBYTE0:
// 		0xFF,					// FUSEBYTE2:
// 	 }
//  };


const _deviceInfo PROGMEM DeviceInfo =
{
	"AINFO",
	__INFO_DATA_STRUCT_VER__,
	MAKE_VERSION(__APP_VER_MAIN__, __APP_VER_REV__),
	(EE_ADDR_MODBUS_SPACE)+(MODBUS_SYSTEM_UART_SPEED)*(EE_ADDR_MODBUS_SPACE_REG_SIZE),
	(EE_ADDR_MODBUS_SPACE)+(MODBUS_SYSTEM_MODBUS_SOFT_ADDR)*(EE_ADDR_MODBUS_SPACE_REG_SIZE)
};

_lcdButtonData lcdButtonData;

/*!
	@brief Metoda sprawdza integralnosc pamieci FLASH w zakresie sekcji bootloadera
	@details Sprawdzenie integralności sekcji bootloadera ogranicza sie do wyszukania wlasciwego znacznika.
		
	@return 0-brak spojnosci pamieci FLASH (sekcji	bootloadera); 1-pamiec FLASH (sekcja aplikacji) jest spojna (wykryto znacznik)
*/
uint8_t checkBootSection(void)
{
	//sprawdzenie czy zawartosc pamieci flash (boot section) zawiera program i rownoczesne wyciagniecie wersji programu (jesli jest):
	/*!
		Zmienna tworzaca maszyne stanu, gdzie odpowiednie stany to:
			0 - nic nie znaleziono 
			1 - znaleziono pierwszy znak znacznika
			...-...
			5 - znaleziono piaty znak znacznika
			6 - odczytano wersje struktury danych
			7 - odczytano mlodszy bajt wersji programu
			8 - odczytano starszy bajt wersji programu
			255 - koniec (znaleziono pastylke informacyjna)
	*/
	uint8_t flashWalkerState = 0;
	uint16_t softVersionTmp = 0xFFFF;
	uint8_t flashData;
	uint16_t tmp = 0;
	for (uint32_t i = BOOT_SECTION_START; i < BOOT_SECTION_END;i++)
	{
		flashData = pgm_read_byte_far(i);
		if(flashWalkerState == 0 && flashData == 'B')
			flashWalkerState ++;
		else if(flashWalkerState == 1 && flashData == 'I')
			flashWalkerState ++;
		else if(flashWalkerState == 2 && flashData == 'N')
			flashWalkerState ++;
		else if(flashWalkerState == 3 && flashData == 'F')
			flashWalkerState ++;
		else if(flashWalkerState == 4 && flashData == 'O')
			flashWalkerState ++;
		else if(flashWalkerState == 5)
		{
			//odczyt wersji struktury pastylki informacyjnej
			if(flashData == 1)
				flashWalkerState ++;
			else
				break;	//nieobslugiwana wersja struktury pastylki - przerwij dalsza analize
		}			
		else if(flashWalkerState == 6)
		{
			//odczyt wersji programu (LSB)
			tmp = flashData;
			flashWalkerState ++;
		}				
		else if(flashWalkerState == 7)
		{
			//odczyt wersji programu (MSB)
			tmp |= (((uint16_t)flashData) << 8);
			softVersionTmp = tmp;
			flashWalkerState = 255;
			break;		//odczytano wymagane dane
		}			
		else
		{
			//powrot do stanu poczatkowego
			flashWalkerState = 0;
		}
	}
	lcdButtonData.bootSoftVersion = softVersionTmp;	
	if(flashWalkerState == 255)
		return 1;
	else
		return 0;
}

/*!
	@brief Odczytanie ewentualnego polecenia od bootloadera
*/
void scanBootCmd(void)
{
	_command bootCmd = *((uint16_t*)(RAMEND-1));
	*((uint16_t*)(RAMEND-1)) = CMD_EMPTY;			//wykasowanie wartosci rejestrow sluzacych wymianie polecen pomiedzy bootloaderem a zwyklym programem
	lcdButtonData.bootDetect = checkBootSection();
	
	if(lcdButtonData.bootDetect)
	{
		//wykryto program w sekcji bootloadera
		if ((bootCmd & CMD_MASK) == CMD_APP_NORMAL_RUN)
		{
			lcdButtonData.dipswitchValue = (uint8_t)(bootCmd & DATA_MASK);
		}
	}
	
	lcdButtonData.appLabel = pgm_read_byte(&DeviceInfo.label[0]);
}


void jobQueueAddCheckConnectionAndStart(void)
{
	_lcdStateContext * lcdStateContext;
	
	//wlaczenie podswietlenia:
	setTimer100msec(TMR_100MSEC_DARKER, readModbus(MODBUS_SYSTEM_DARKER_TIME));
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdStateContext != NULL)
	{
		lcdStateContext->state = LCD_STATE_BRIGHTNESS_JOB;
		lcdStateContext->substate = LCD_SUBSTATE_BRIGHTNESS_JOB_START;
		uint16_t lightLevel = readModbus(MODBUS_SYSTEM_BACKLIGHT_ON);
		if(lightLevel > 100)
			lightLevel = 100;
		lcdStateContext->data[0] = (uint8_t)lightLevel;
		enqueueLcdWork();
	}
	
#ifdef COMPILATION_ENABLE_WAGO_COMMUNICATION_CONTROL
	//dodanie do kolejki zadan imitacje inicjalizacji:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdStateContext != NULL)
	{
		lcdStateContext->state = LCD_STATE_WAGO_NO_CONNECT;
		lcdStateContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_START;
		enqueueLcdWork();
	}
#endif	//COMPILAtiON_ENABLE_WAGO_COMMUNICATION_CONTROL
	
	//dodanie do kolejki zadan wyswietlenie belki systemowej i sceny uzytkownika:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdStateContext != NULL)
	{
		if(lcdButtonData.sceneSwitchFlag)
			lcdButtonData.scene.activeScene = lcdButtonData.sceneSwitchNumber;
		else
			lcdButtonData.scene.activeScene = 0;
		
		lcdStateContext->state = LCD_STATE_SERVICE_SCENE_ACTIVE;
		lcdStateContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_START;
		enqueueLcdWork();
	}	
}

int16_t main(void)
{
	uint8_t prevHiPrioState = 0;
	_lcdStateContext * lcdStateContext;
	
	adcEnable();
	dipSwitchInit();
	initHardwareEarly();
#ifndef _DEBUG
	scanBootCmd();		//sprawdzenie komunikatów od bootloadera
#endif // _DEBUG
	initTimers();
	
#ifndef _DEBUG
	initData();
#endif // _DEBUG
	initUart();
	ledShowNormalWork();
	initHardwareLate();
			
	//dodanie do kolejki zadan dla portu LCD inicjalizacje wyswietlacza:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdStateContext != NULL)
	{
		lcdStateContext->state = LCD_STATE_INIT;
		lcdStateContext->substate = LCD_SUBSTATE_INIT_START;
		enqueueLcdWork();
	}
	
	//dodanie do kolejki pobranie danych konfiguracyjnych dotyczacym aktywacji urządzenia
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdStateContext != NULL)
	{
		lcdStateContext->state = LCD_STATE_INIT_DEVICE_WAKE_UP;
		lcdStateContext->substate = LCD_SUBSTATE_INIT_WAKE_UP_START;
		enqueueLcdWork();
	}
	
	//dodanie do kolejki pobranie danych konfiguracyjnych dotyczacym PID
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdStateContext != NULL)
	{
		lcdStateContext->state = LCD_STATE_INIT_PID;
		lcdStateContext->substate = LCD_SUBSTATE_INIT_PID_START;
		enqueueLcdWork();
	}

	//dodanie do kolejki zadan walidacje danych modbus:
	lcdStateContext = getLcdWorkQueueFree();
	if(lcdStateContext != NULL)
	{
		lcdStateContext->state = LCD_STATE_VALID_MODBUS_DATA;
		lcdStateContext->substate = LCD_SUBSTATE_VALID_MODBUS_START;
		enqueueLcdWork();
	}
		
	jobQueueAddCheckConnectionAndStart();

	dsPrepareToWork();
	
	while(1)
    {
		hardwareService();
		uartService();
		timersService();
		ledService();

		if(!getTimer1Sec(TMR_1SEC_HI_PRIO_DELAY))
		{
			//ponizsze czynnosci wykonujemy wowczas gdy nie jest wlasnie wykonywane jakies zadanie o wysokim priorytecie:
			lcdService();
			/*if(lcdButtonData.config.newUserConfigFlag)
			{
				lcdStateContext = getLcdWorkQueueFree();
				if(lcdStateContext != NULL)
				{
					setTimer1Sec(TMR_1SEC_HI_PRIO_DELAY, TMR_1SEC_HI_PRIO_DELAY_TIMEOUT);
					lcdStateContext->state = LCD_STATE_PROJECT_CLEAR_NOTUSE;
					lcdStateContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_START;
					enqueueLcdWork();
				}
			}
			else */
			if(prevHiPrioState)
			{
				while(1);	//restart watchdogiem
			}
			
			lcdTouchService();			
			inputService();
			eepromService(LCD_EEPROM_NORMAL_CMD);
			dsServiceWork();
			lcdBacklightService(LCD_BACKLIGHT_NORMAL_CMD);
			buttonsIntegrity();
			barsIntegrity();
			numbersIntegrity();
			statesIntegrity();			
			alarmBuzzerService();
			pidService();
		}
		else
		{
			prevHiPrioState = 1;
			lcdServiceHighPriority();
		}
		
#ifdef COMPILATION_ENABLE_WAGO_COMMUNICATION_CONTROL
		//wykrywanie braku komunikacji z WAGO:
		if(!getTimer1Sec(TMR_1SEC_WAGO_NO_CONNECT) && lcdButtonData.modbusActive)
		{
			lcdButtonData.modbusActive = 0;
			jobQueueAddCheckConnectionAndStart();
		}
#endif	//COMPILATION_ENABLE_WAGO_COMMUNICATION_CONTROL
		
		
		//odczekanie z realizacja niektorych zadan do mometu zakonczenia wysylania odpowiedzi:
		if(!uartWAGO->isTransmit())
		{
			//wykrywanie przejscia do bootloadera:
			if(getWorkState() == MODE_ACT)
			{
				if(lcdButtonData.bootDetect)
				{					
					*((uint16_t*)(RAMEND-1)) = CMD_STAY_IN_BOOT | lcdButtonData.dipswitchValue;
					jumpToBootloader();
				}
				else
				{
					setWorkState(MODE_ACT_END);
				}
			}
			//wykrywanie zmiany predkosci komunikacji:
			if(lcdButtonData.newModbusBaudRate != lcdButtonData.actualModbusBaudRate)
			{
				if(lcdButtonData.dipswitchValue == DIPSWITCH_SERVICE_MODE)
				{
					lcdButtonData.newModbusBaudRate = lcdButtonData.actualModbusBaudRate;
				}
				else
				{
					uartWAGO->setBaudRate(lcdButtonData.newModbusBaudRate);
					lcdButtonData.actualModbusBaudRate = lcdButtonData.newModbusBaudRate;
				}
				uartWAGO->startReceiving();
			}
			//wykrywanie zmiany adresu urzadzenia:
			if(lcdButtonData.modbusAddress != lcdButtonData.newModbusAddress)
			{
				if(lcdButtonData.dipswitchValue == DIPSWITCH_SERVICE_MODE)
				{
					lcdButtonData.newModbusAddress = lcdButtonData.modbusAddress;
				}
				else
				{
					lcdButtonData.modbusAddress = lcdButtonData.newModbusAddress;	
				}				
			}
		}
		
    }
}


void setWorkState(_workState newWorkState)
{
	lcdButtonData.deviceWorkState = newWorkState;
}

_workState getWorkState(void)
{
	return lcdButtonData.deviceWorkState;
}