﻿/*!
 *  @file processor.c
 */ 
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/interrupt.h>

#include "processor.h"
#include "lcdButtonTimers.h"
#include "main.h"
#include "adc.h"

#define TIMER1_PRESCALING_COUNT		5
uint8_t timer1PrescaligConfigArray[TIMER1_PRESCALING_COUNT] = {TIMER1_PRESCALER_1, TIMER1_PRESCALER_8, TIMER1_PRESCALER_64, TIMER1_PRESCALER_256, TIMER1_PRESCALER_1024};
uint16_t timer1PrescaligArray[TIMER1_PRESCALING_COUNT] = {1, 8, 64, 256, 1024};


void wait_us(unsigned int time_us)
{
	int i;
	uint32_t ocraValue;
	uint16_t timerPrescaler;
	
	if(time_us < 10)
		return;
	
	uint8_t flag;
	
	ocraValue = (uint32_t)(time_us - 10) * (F_CPU/ 1000000);
	timerPrescaler = TIMER1_PRESCALER_1;
	i = 1;
	while(ocraValue > 0xFFFF && i < TIMER1_PRESCALING_COUNT)
	{
		ocraValue = ocraValue / (timer1PrescaligArray[i] / timer1PrescaligArray[i-1]);
		timerPrescaler = timer1PrescaligConfigArray[i];
	}	
	OCR1A = ocraValue;
	TCNT1 = 0;
	TCCR1B |= timerPrescaler;
	flag = TIFR1 & (1<<OCF1A);
	while(flag == 0)
		flag = TIFR1 & (1<<OCF1A);//waiting
	TCCR1B &= ~((1<<CS12)|(1<<CS11)|(1<<CS10));	//timer stop
	TIFR1 = 0xFF;
}


void preInit(void)
{
	cli();
	MCUSR = 0x00;	//wykasowanie flag zrodla resetu
	wdt_disable();
	
	//Przeniesienie wektora przerwan do sekcji aplikacji:
	unsigned char tmp;
	unsigned char changeEnableCmd;
	unsigned char moveIntToAppCmd;
	tmp = MCUCR;						/* GET MCUCR*/
	changeEnableCmd = tmp | (1<<IVCE);
	moveIntToAppCmd = tmp & ~(1<<IVSEL);
	MCUCR = changeEnableCmd;			/* Enable change of Interrupt Vectors  */
	MCUCR = moveIntToAppCmd;			/* Move interrupts to Boot Flash section */
}

void initHardwareEarly(void)
{
	//porty:	
	//-----------------------------------------
	//uart0:
	UART0_TXD_DIR_REG &= ~(1<<UART0_TXD_LINE);		//input
	UART0_TXD_PORT_REG	|= ~(1<<UART0_TXD_LINE);	//pulled high
	UART0_RXD_DIR_REG &= ~(1<<UART0_RXD_LINE);		//input
	UART0_RXD_PORT_REG	|= ~(1<<UART0_RXD_LINE);	//pulled high
	UART0_RE_DIR_REG |= _BV(UART0_RE_LINE);
	UART0_DE_DIR_REG |= _BV(UART0_DE_LINE);
	//uart1:
	UART1_TXD_DIR_REG &= ~(1<<UART1_TXD_LINE);		//input
	UART1_TXD_PORT_REG	|= ~(1<<UART1_TXD_LINE);	//pulled high
	UART1_RXD_DIR_REG &= ~(1<<UART1_RXD_LINE);		//input
	UART1_RXD_PORT_REG	|= ~(1<<UART1_RXD_LINE);	//pulled high
	//lcd bussy pin:
	LCD_BUSY_DIR_REG &= ~(1<<LCD_BUSY_LINE);		//input
	LCD_BUSY_PORT_REG |= ~(1<<LCD_BUSY_LINE);		//pulled low
	
	//DS:
	DS_DIR_REG &= ~(_BV(DS_LINE));
 	DS_OUT_REG &= ~(_BV(DS_LINE));
	//LED:
	LED_DIR_REG |= _BV(LED_LINE);
 	LED_OUT_REG |= _BV(LED_LINE);
	//DIO:
	DIO_DIR_REG |= _BV(DO_0_LINE);
	DIO_DIR_REG |= _BV(DO_1_LINE);
	DIO_DIR_REG |= _BV(DO_2_LINE);
	DIO_DIR_REG |= _BV(DO_3_LINE);
	DO_0_LOW();
	DO_1_LOW();
	DO_2_LOW();
	DO_3_LOW();
	DIO_DIR_REG &= ~_BV(DI_0_LINE);
	DIO_DIR_REG &= ~_BV(DI_1_LINE);
	DIO_DIR_REG &= ~_BV(DI_2_LINE);
	DIO_DIR_REG &= ~_BV(DI_3_LINE);	 
	
	//DIO - konfiguracyjne:
	DIOC_OUT2_DIR_REG |= _BV(DIOC_OUT2_LINE);
	DIOC_OUT3_DIR_REG |= _BV(DIOC_OUT3_LINE);	
	DIOC_IN2_DIR_REG &= ~_BV(DIOC_IN2_LINE);
	DIOC_IN3_DIR_REG &= ~_BV(DIOC_IN3_LINE);
	
	//obecnosc:
	OBECNOSC_DIR_REG &= ~_BV(OBECNOSC_LINE);
	
		
	//timery:
	//-----------------------------------------
	//Timer0 - PWM (ustawienie PWM nastepuje pozniej):
	TIMSK0 = 0;
	TCCR0B = (1<<CS01);		//prescaler: /8
 	TCCR0A = (1<<WGM00)|(1<<WGM01);		//fast PWM 	
	//-----------------------------------------
	//Timer1 - odmierzanie odcinkow czasu w us (czujnik DS):
	TCCR1A = 0;
	TCCR1B = (1<<WGM12);	//timer stop
	TCCR1C = 0;
	TIMSK1 = 0;
	TCNT1 = 0;
	TIFR1 = 0xFF;			//wykasowanie flag
	//-----------------------------------------
	//Timer2 - odmierzanie standardowego interwalu (TIMEOUT_UNIT[ms]) i opoznienie rozpoczecia wysylania odpowiedzi po interfejsie uart0 (WAGO):
	OCR2A = TIMER_VALUE;
	TIMSK2 = (1<<OCIE2A);
	TCCR2A = (1<<WGM21);	
	TCCR2B = (1<<CS22)|(1<<CS21)|(1<<CS20);
}

void initHardwareLate(void)
{
	sei();
	wdt_enable(WDTO_2S);
}

void hardwareDisableUnnecessary()
{
	TCCR0B = 0;
	TCCR1B = 0;
}

void hardwareTurnOff(void)
{
	cli();
	wdt_disable();
	TCCR2B = 0;
}

inline void hardwareService(void){
	wdt_reset();
}

void inputService(void)
{
	static uint8_t lastPwmStatus = 0;
	
	uint16_t bitData1Reg, bitData2Reg;
	uint16_t extraReg, config, digOutReg;
	uint8_t error, input;
	
	//wejscia/wyjscia konfigurowalne:
	config = lcdButtonData.modbusSpace[MODBUS_SYSTEM_BIT_DATA];
	bitData1Reg = lcdButtonData.modbusSpace[MODBUS_READ_ONLY_BIT_DATA_1];
	bitData2Reg = lcdButtonData.modbusSpace[MODBUS_READ_ONLY_BIT_DATA_2];
	extraReg = lcdButtonData.modbusSpace[MODBUS_READ_ONLY_EXTRA_BIT_DATA];
	
	for(int i = 0; i < 2; i++)
	{
		if(config & (1<<(BIT_MODBUS_SYSTEM_BIT_PWM_DIO_1+i)))
		{
			//aktywny kanal PWM na wyjsciu:
			bitData1Reg &= ~(1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_ERROR_DIO1+i));		//brak bledu
			if(!(lastPwmStatus & (1<<i)))
			{
				//wlaczenie kanalu PWM:
				lastPwmStatus |= (1<<i);
				if(i == 0)
				{	
					OCR0A = (uint8_t)lcdButtonData.modbusSpace[MODBUS_SYSTEM_PWM_0_VALUE];
					TCCR0A |= (1<<COM0A1);
				}					
				else if(i == 1)
				{
					OCR0B = (uint8_t)lcdButtonData.modbusSpace[MODBUS_SYSTEM_PWM_1_VALUE];
					TCCR0A |= (1<<COM0B1);
				}					
			}
		}
		else
		{
			if(lastPwmStatus & (1<<i))
			{
				//wylaczenie kanalu PWM, deinicjalizacja timera:
				lastPwmStatus &= ~(1<<i);
				if(i == 0)
					TCCR0A &= ~((1<<COM0A1)|(1<<COM0A0));
				else if(i == 1)
					TCCR0A &= ~((1<<COM0B1)|(1<<COM0B0));
			}				
			
			if(i == 0)
				input = DIOC_IN2_INPUT();
			else
				input = DIOC_IN3_INPUT();
			if(config & (1<<(BIT_MODBUS_SYSTEM_BIT_CONFIG_DIO_1+i)))
			{
				//DIOn jako wyjscie, sprawdzic czy stan jest prawidlowy:
				error = 0;
				if(config & (1<<(BIT_MODBUS_SYSTEM_BIT_DATA_DIO_1+i)))
				{
					//na wyjsciu ma byc stan niski (logika odwrocona)
					if(i == 0)
						DIOC_OUT2_HI();
					else
						DIOC_OUT3_HI();
					if(!input)
					{
						//na wyjsciu faktycznie jest stan wysoki(logika odwrocona)
						error = 1;
					}				
				}
				else
				{
					//na wyjsciu ma byc stan wysoki (logika odwrocona)
					if(i == 0)
						DIOC_OUT2_LOW();
					else
						DIOC_OUT3_LOW();
					if(input)
						//na wyjsciu faktycznie jest stan niski(logika odwrocona)
						error = 1;
				}
				if(error)
				{
					bitData1Reg |= (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_ERROR_DIO1+i));
				}
				else
				{
					bitData1Reg &= ~(1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_ERROR_DIO1+i));
				}
			}
			else
			{
				//DIOn jako wejscie, sprawdzic czy wykryto zbocze opadajace lub rosnace:
				if(bitData1Reg & (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_STATE_DIO1+i)))
				{
					//poprzednio odczytany stan to niskie napiecie (logika odwrocona)
					if(!input)
					{
						//aktualny stan to wysokie napiecie (logika odwrocona) - wykryto zbocze narastajace
						bitData1Reg &= ~(1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_STATE_DIO1+i));
						bitData1Reg |= (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_RISING_DIO1+i));					
					}
				}
				else
				{
					//poprzednio odczytany stan to wysokie napiecie (logika odwrocona)
					if(input)
					{
						//aktualny stan to niskie napiecie (logika odwrocona) - wykryto zbocze opadajace
						bitData1Reg |= (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_STATE_DIO1+i));
						bitData1Reg |= (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_FALLING_DIO1+i));
					}
				}
			}			
		}			
	}
	lcdButtonData.modbusSpace[MODBUS_READ_ONLY_BIT_DATA_1] = bitData1Reg;
	
	//wejscia zwykle:
	if(DI_0_INPUT())
	{
		bitData2Reg |= (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_0);
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_0);
	}		
	else
	{
		bitData2Reg &= ~(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_0);
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_0);
	}		
	
	if(DI_1_INPUT())
	{
		bitData2Reg |= (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_1);		
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_1);
	}		
	else
	{
		bitData2Reg &= ~(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_1);
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_1);
	}		
	
	if(DI_2_INPUT())
	{
		bitData2Reg |= (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_2);
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_2);
	}		
	else
	{
		bitData2Reg &= ~(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_2);
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_2);
	}		
		
	if(DI_3_INPUT())
	{
		bitData2Reg |= (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_3);
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_3);
	}		
	else
	{
		bitData2Reg &= ~(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_3);
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_3);
	}		
		
	//obecnosc:
	if(OBECNOSC_INPUT())
	{
		bitData2Reg |= (1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_OBECNOSC);
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_OBECNOSC);
	}		
	else
	{
		bitData2Reg &= ~(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_OBECNOSC);
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_OBECNOSC);
	}		
	lcdButtonData.modbusSpace[MODBUS_READ_ONLY_BIT_DATA_2] = bitData2Reg;
	
	//rejestr extra z powtorzonymi stanami na poszczegolnych liniach:
	if(config & (1<<(BIT_MODBUS_SYSTEM_BIT_CONFIG_DIO_1)))
	{
		//DIO1 skonfigurowane jako wyjscie:
		if(config & (1<<(BIT_MODBUS_SYSTEM_BIT_DATA_DIO_1)))
			extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO1);
		else
			extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO1);
	}
	else
	{
		//DIO1 skonfigurowane jako wejscie:
		if(bitData1Reg & (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_STATE_DIO1)))
			extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO1);
		else
			extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO1);
	}
	
	if(config & (1<<(BIT_MODBUS_SYSTEM_BIT_CONFIG_DIO_2)))
	{
		//DIO2 skonfigurowane jako wyjscie:
		if(config & (1<<(BIT_MODBUS_SYSTEM_BIT_DATA_DIO_2)))
			extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO2);
		else
			extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO2);
	}
	else
	{
		//DIO2 skonfigurowane jako wejscie:
		if(bitData1Reg & (1<<(BIT_MODBUS_READ_ONLY_BIT_DATA_1_STATE_DIO2)))
			extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO2);
		else
			extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO2);
	}
	
	digOutReg = lcdButtonData.modbusSpace[MODBUS_SYSTEM_BIT_DATA];	
	if(digOutReg & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_0))
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_0);
	else
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_0);
		
	if(digOutReg & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_1))
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_1);
	else
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_1);
		
	if(digOutReg & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_2))
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_2);
	else
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_2);
		
	if(digOutReg & (1<<BIT_MODBUS_SYSTEM_BIT_DATA_DO_3))
		extraReg |= (1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_3);
	else
		extraReg &= ~(1<<MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_3);

	lcdButtonData.modbusSpace[MODBUS_READ_ONLY_EXTRA_BIT_DATA] = extraReg;
	
	//naslonecznienie:
	adcSetChannel(ADC_CHANNEL_2);
	uint16_t rawData = adcConvert();
	lcdButtonData.modbusSpace[MODBUS_READ_ONLY_LIGHT_VOLT] = rawData;
	
	uint16_t luksy = 0;
	if(rawData < 810)
		luksy = 14 * rawData / 10 - 17;
	else
		luksy = 41 * rawData - 32080;
	lcdButtonData.modbusSpace[MODBUS_READ_ONLY_LIGHT] = luksy;
}

void jumpToBootloader(void)
{
	hardwareTurnOff();
	uartTurnOff();
	((void (*)(void))(BOOT_SECTION_START/2))();
}
