﻿/*!
 *  @file processor.h
 *	@brief Definicje stałych i makr związanych z ustawieniami sprzętowymi
 */ 

#ifndef PROCESSOR_H_
#define PROCESSOR_H_

#include "uart.h"

#define BOOT_SECTION_START	0x1E000
#define BOOT_SECTION_END	FLASHEND		//FLASH_END: deklaracja w pliku iom1284p.h

//uart baud rates:
#if F_CPU == 18432000
	#define BAUDRATE_1200_UBRR		0x03BF
	#define BAUDRATE_2400_UBRR		0x01DF
	#define BAUDRATE_4800_UBRR		0x00EF
	#define BAUDRATE_9600_UBRR		0x0077
	#define BAUDRATE_19200_UBRR		0x003B
	#define BAUDRATE_38400_UBRR		0x001D
	#define BAUDRATE_57600_UBRR		0x0013
	#define BAUDRATE_115200_UBRR	0x0009
#elif F_CPU == 11059200
	#define BAUDRATE_1200_UBRR		0x023F
	#define BAUDRATE_2400_UBRR		0x011F
	#define BAUDRATE_4800_UBRR		0x008F
	#define BAUDRATE_9600_UBRR		0x0047
	#define BAUDRATE_19200_UBRR		0x0023
	#define BAUDRATE_38400_UBRR		0x0011
	#define BAUDRATE_57600_UBRR		0x000B
	#define BAUDRATE_115200_UBRR	0x0005
#endif // F_CPU == 18432000 | 11059200
#define BAUDRATE_NONE_UBRR		0xFFFF

//uart0 (WAGO):
#define UART0_TXD_DIR_REG	DDRD
#define UART0_TXD_PORT_REG	PORTD
#define UART0_TXD_LINE		PD1
#define UART0_RXD_DIR_REG	DDRD
#define UART0_RXD_PORT_REG	PORTD
#define UART0_RXD_LINE		PD0

#ifdef FINALLY
	#define UART0_RE_DIR_REG	DDRD
	#define UART0_RE_PORT_REG	PORTD
	#define UART0_RE_LINE		PD5
	#define UART0_DE_DIR_REG	DDRD
	#define UART0_DE_PORT_REG	PORTD
	#define UART0_DE_LINE		PD4
#else
	#define UART0_RE_DIR_REG	DDRC
	#define UART0_RE_PORT_REG	PORTC
	#define UART0_RE_LINE		PC1
	#define UART0_DE_DIR_REG	DDRC
	#define UART0_DE_PORT_REG	PORTC
	#define UART0_DE_LINE		PC1
#endif // FINALLY

//uart1 (LCD):
#define UART1_TXD_DIR_REG	DDRD
#define UART1_TXD_PORT_REG	PORTD
#define UART1_TXD_LINE		PD3
#define UART1_RXD_DIR_REG	DDRD
#define UART1_RXD_PORT_REG	PORTD
#define UART1_RXD_LINE		PD2
//LCD:
#ifdef FINALLY
	#define	LCD_BUSY_DIR_REG	DDRA
	#define	LCD_BUSY_PORT_REG	PORTA
	#define	LCD_BUSY_PIN_REG	PINA
	#define	LCD_BUSY_LINE		DDA3
#else
	#define	LCD_BUSY_DIR_REG	DDRD
	#define	LCD_BUSY_PORT_REG	PORTD
	#define	LCD_BUSY_PIN_REG	PIND
	#define	LCD_BUSY_LINE		DDD4
#endif // FINALLY

#define LCD_NOT_BUSY_IN	(LCD_BUSY_PIN_REG)&(1<<(LCD_BUSY_LINE))		//0-bufor lcd zapelniony; !0-lcd gotowe na przyjecie kolejnego polecenia

//dipswitch:
#define DIP_L_DIR_REG	DDRA
#define DIP_L_LINE		DDA0
#define DIP_H_DIR_REG	DDRA
#define DIP_H_LINE		DDA1


//DS:
#ifdef FINALLY
	#define	DS_OUT_REG			PORTB	//rejestr wyjsciowy portu dla czujnika
	#define	DS_IN_REG			PINB	//rejestr wejsciowy portu dla czujnika
	#define	DS_DIR_REG			DDRB	//rejestr kierunku portu dla czujnika
	#define	DS_LINE				1		//nr linii dla sygnalu DQ
#else
	#define	DS_OUT_REG			PORTC	//rejestr wyjsciowy portu dla czujnika
	#define	DS_IN_REG			PINC	//rejestr wejsciowy portu dla czujnika
	#define	DS_DIR_REG			DDRC	//rejestr kierunku portu dla czujnika
	#define	DS_LINE				7		//nr linii dla sygnalu DQ
#endif // FINALLY

//LED:
#ifdef FINALLY
	#define	LED_OUT_REG			PORTB	//rejestr wyjsciowy portu dla diody led
	#define	LED_DIR_REG			DDRB	//rejestr kierunku portu dla diody led
	#define	LED_LINE			6		//nr linii diody led
#else
	#define	LED_OUT_REG			PORTB	//rejestr wyjsciowy portu dla diody led
	#define	LED_DIR_REG			DDRB	//rejestr kierunku portu dla diody led
	#define	LED_LINE			0		//nr linii diody led
#endif // FINALLY



//DI/O:
#ifdef FINALLY
	#define	DIO_OUT_REG			PORTC	//rejestr wyjsciowy portu dla DIO
	#define	DIO_IN_REG			PINC	//rejestr wejsciowy portu dla DIO
	#define	DIO_DIR_REG			DDRC	//rejestr kierunku portu dla DIO
	#define	DI_0_LINE			0		//nr linii dla wejscia  0
	#define	DI_1_LINE			1		//nr linii dla wejscia  1
	#define	DI_2_LINE			2		//nr linii dla wejscia  2
	#define	DI_3_LINE			3		//nr linii dla wejscia  3
	#define	DO_0_LINE			4		//nr linii dla wyjscia	0
	#define	DO_1_LINE			5		//nr linii dla wyjscia	1
	#define	DO_2_LINE			6		//nr linii dla wyjscia	2
	#define	DO_3_LINE			7		//nr linii dla wyjscia	3
#else
	#define	DIO_OUT_REG			PORTA	//rejestr wyjsciowy portu dla DIO
	#define	DIO_IN_REG			PINA	//rejestr wejsciowy portu dla DIO
	#define	DIO_DIR_REG			DDRA	//rejestr kierunku portu dla DIO
	#define	DI_0_LINE			0		//nr linii dla wyjscia  0
	#define	DI_1_LINE			1		//nr linii dla wyjscia  1
	#define	DI_2_LINE			2		//nr linii dla wyjscia  2
	#define	DI_3_LINE			3		//nr linii dla wyjscia  3
	#define	DO_0_LINE			4		//nr linii dla wejscia	0
	#define	DO_1_LINE			5		//nr linii dla wejscia	1
	#define	DO_2_LINE			6		//nr linii dla wejscia	2
	#define	DO_3_LINE			7		//nr linii dla wejscia	3
#endif // FINALLY

#define DO_0_HI()		DIO_OUT_REG |= _BV(DO_0_LINE)
#define DO_0_LOW()		DIO_OUT_REG &= ~(_BV(DO_0_LINE))
#define DO_1_HI()		DIO_OUT_REG |= _BV(DO_1_LINE)
#define DO_1_LOW()		DIO_OUT_REG &= ~(_BV(DO_1_LINE))
#define DO_2_HI()		DIO_OUT_REG |= _BV(DO_2_LINE)
#define DO_2_LOW()		DIO_OUT_REG &= ~(_BV(DO_2_LINE))
#define DO_3_HI()		DIO_OUT_REG |= _BV(DO_3_LINE)
#define DO_3_LOW()		DIO_OUT_REG &= ~(_BV(DO_3_LINE))

#define DI_0_INPUT()	((DIO_IN_REG & (1<<DI_0_LINE)) != 0)
#define DI_1_INPUT()	((DIO_IN_REG & (1<<DI_1_LINE)) != 0)
#define DI_2_INPUT()	((DIO_IN_REG & (1<<DI_2_LINE)) != 0)
#define DI_3_INPUT()	((DIO_IN_REG & (1<<DI_3_LINE)) != 0)

//DI/O configured:
#define	DIOC_OUT2_DIR_REG			DDRB
#define	DIOC_IN2_DIR_REG			DDRB
#define	DIOC_OUT2_OUT_REG			PORTB
#define	DIOC_IN2_IN_REG				PINB
#define	DIOC_OUT2_LINE				3
#define	DIOC_IN2_LINE				2
#define DIOC_OUT2_HI()				DIOC_OUT2_OUT_REG |= _BV(DIOC_OUT2_LINE)
#define DIOC_OUT2_LOW()				DIOC_OUT2_OUT_REG &= ~(_BV(DIOC_OUT2_LINE))
#define DIOC_IN2_INPUT()			((DIOC_IN2_IN_REG & (1<<DIOC_IN2_LINE)) != 0)

#define	DIOC_OUT3_DIR_REG			DDRB
#define	DIOC_IN3_DIR_REG			DDRB
#define	DIOC_OUT3_OUT_REG			PORTB
#define	DIOC_IN3_IN_REG				PINB
#define	DIOC_OUT3_LINE				4
#define	DIOC_IN3_LINE				5
#define DIOC_OUT3_HI()				DIOC_OUT3_OUT_REG |= _BV(DIOC_OUT3_LINE)
#define DIOC_OUT3_LOW()				DIOC_OUT3_OUT_REG &= ~(_BV(DIOC_OUT3_LINE))
#define DIOC_IN3_INPUT()			((DIOC_IN3_IN_REG & (1<<DIOC_IN3_LINE)) != 0)

//obecnosc:
#define	OBECNOSC_IN_REG				PINB	//rejestr wejsciowy portu dla obecnosci
#define	OBECNOSC_DIR_REG			DDRB	//rejestr kierunku portu dla obecnosci
#define	OBECNOSC_LINE				7		//nr linii dla sygnalu obecnosci
#define OBECNOSC_INPUT()			((OBECNOSC_DIR_REG & (1<<OBECNOSC_LINE)) != 0)


#define TIMER1_PRESCALER_1			(1<<CS10)
#define TIMER1_PRESCALER_8			(1<<CS11)
#define TIMER1_PRESCALER_64			((1<<CS11) | (1<<CS10))
#define TIMER1_PRESCALER_256		(1<<CS12)
#define TIMER1_PRESCALER_1024		((1<<CS12) | (1<<CS10))

/*!
	@brief Odczekanie 1 us
	@details UWAGA! zalezne od procesora!
*/
#if F_CPU == 18432000
	#define wait_1us()	asm volatile("nop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\t")
#elif F_CPU == 11059200
	#define wait_1us()	asm volatile("nop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\tnop\n\t")
#endif // F_CPU == 18432000 | 11059200

/*!
	@brief Odczekanie zadanego w us czasu
*/
void wait_us(unsigned int time_us);


/*!
	@brief Wstepny etap inicjalizacji systemu, jeszcze przed uruchomieniem funkcji main.
	@details W etapie tym dokonuje sie wylaczenia watchdobra, co ma na celu ochronę przed zapetleniem programu w przypadku
	gdy bedzie przerwanie od watchdoga.
*/
void preInit(void) __attribute__((naked)) __attribute__((section(".init3")));

/*!
	@brief Wczesne inicjalizowanie ustawien sprzetowych
	@details Dotyczy to ustawien podstawowych.
*/
void initHardwareEarly(void);

/*!
	@brief Pozne inicjalizowanie ustawien sprzetowych
	@details Dotyczy to ustawien wymaganych do wykonania na koncu procesu inicjalizacji.
*/
void initHardwareLate(void);

/*!
	@brief Wylaczenie funkcjonalnosci nieistotnych dla trybu pracy o podwyzszonym priorytecie
*/
void hardwareDisableUnnecessary();

/*!
	@brief Wyłączenie sprzetu
*/
void hardwareTurnOff(void);

/*!
	@brief Obsluga peryferii mikrokontrolera wykonywana w kazdym cyklu petli glownej
	@details Resetowanie ukladu Watchdog Timera w kazdym cyklu petli glownej.
*/
void hardwareService(void);

/*!
	@brief Kontrola wejść cyfrowych 
*/
void inputService(void);

/*!
	@brief Skok do sekcji bootloadera (do adresu BOOT_SECTION_START)
*/
void jumpToBootloader(void);

#endif /* PROCESSOR_H_ */
