﻿#include <stdint.h>
#include <string.h>
#include "protocolLcd.h"
#include "uart.h"
#include "crc.h"
#include "lcdButtonTimers.h"
#include "lcd.h"
#include "main.h"

_uartData * uartLCD;

extern _lcdStateContext lcdWorkQueue[LCD_WORK_QUEUE_MAX];
extern _lcdButtonData lcdButtonData;

/**************************************************************************
Podstawowa funkcjonalnosc
**************************************************************************/
//////////////////////////////////////////////////////////////////////////
uint8_t isFrameDetectedLcd()
{
	if(uartLCD->counterRx>=5)
		if(uartLCD->bufferRx[0] == LCD_FRAME_START)
			if(uartLCD->bufferRx[uartLCD->counterRx-4] == LCD_FRAME_END_1)
				if(uartLCD->bufferRx[uartLCD->counterRx-3] == LCD_FRAME_END_2)
					if(uartLCD->bufferRx[uartLCD->counterRx-2] == LCD_FRAME_END_3)
						if(uartLCD->bufferRx[uartLCD->counterRx-1] == LCD_FRAME_END_4)
							return 1;
	return 0;
}
//////////////////////////////////////////////////////////////////////////

/*!
	@brief
		Transformacja typu wyliczeniowego predkosci transmisji na wartosc odpowiadajaca
		tej predkosci w protokole LCD
	@param baudRate
		Predkosc, dla jakiej chcemy uzyskac wartosc wykorzystywana przez LCD
	@return 
		Wartosc wykorzystywana przez LCD do prezentacji danej prędkości komunikacji
*/
uint8_t protocolLcdBaudRate2Data(_baudRate baudRate)
{
	uint8_t baudCode;
	if(baudRate == BAUDRATE_1200)
		baudCode = 0;
	else if(baudRate == BAUDRATE_2400)
		baudCode = 1;
	else if(baudRate == BAUDRATE_4800)
		baudCode = 2;
	else if(baudRate == BAUDRATE_9600)
		baudCode = 3;
	else if(baudRate == BAUDRATE_19200)
		baudCode = 4;
	else if(baudRate == BAUDRATE_38400)
		baudCode = 5;
	else if(baudRate == BAUDRATE_57600)
		baudCode = 6;
	else
		baudCode = 7;	//if(baudRate == BAUDRATE_115200)
	return baudCode;
}

/*!
	@brief
		Transformacja wartosci wykorzystywanej przez LCD do prezentacji predkosci transmisji
		na typ wyliczeniowy
	@param dataBaudRate
		Wartosc wykorzystywana przez LCD do prezentacji danej prędkości komunikacji		
	@return 
		Typ wyliczeniowy dla danej predkosci komunikacji
*/
_baudRate protocolLcdData2BaudRate(uint8_t baudCode)
{
	_baudRate baudRate = BAUDRATE_NONE;
	if(baudCode == 0)
		baudRate = BAUDRATE_1200;
	else if(baudCode == 1)
		baudCode = BAUDRATE_2400;
	else if(baudCode == 2)
		baudRate = BAUDRATE_4800;
	else if(baudCode == 3)
		baudRate = BAUDRATE_9600;
	else if(baudCode == 4)
		baudRate = BAUDRATE_19200;
	else if(baudCode == 5)
		baudRate = BAUDRATE_38400;
	else if(baudCode == 6)
		baudRate = BAUDRATE_57600;
	else if(baudCode == 7)
		baudRate = BAUDRATE_115200;
	return baudRate;
}

/**************************************************************************
Analiza odbieranych komunikatow
**************************************************************************/
//////////////////////////////////////////////////////////////////////////
uint16_t parseFrameLcd(void)
{
	_lcdStateContext * lcdContext;
	
	uint16_t bytesToSend = 0;
		
	if(uartLCD->bufferRx[1] == 0x00)
	{
		//odebrano handshake:
		if(uartLCD->bufferRx[2] == 'O' && uartLCD->bufferRx[3] == 'K')
		{	
			lcdData.version[0] = uartLCD->bufferRx[6];
			lcdData.version[1] = uartLCD->bufferRx[8];
			lcdData.id = uartLCD->bufferRx[9];
			lcdData.baudRate = protocolLcdData2BaudRate(uartLCD->bufferRx[10]);
			lcdData.options = uartLCD->bufferRx[11];
		}
	}
	else if(uartLCD->bufferRx[1] == 0x72)
	{
		// koniec klikniecia w wyswietlacz
		if(lcdButtonData.button.clickedState != CLICK_EMPTY)
		{
			lcdContext = getLcdWorkQueueFree();
			if(lcdContext != NULL)
			{
				lcdContext->data[0] = CLICK_NONE;
				lcdContext->state = LCD_STATE_TOUCH;
				lcdContext->substate = LCD_SUBSTATE_TOUCH_START;
				enqueueLcdWork();
			}
		}
	}
	else if(uartLCD->bufferRx[1] == 0x73)
	{
		// kliknieto w wyswietlacz
		setTimerShort(TMR_SHORT_LCD_CLICK, LCD_CLICK_TIMEOUT);
		if(lcdButtonData.button.clickedState != CLICK_LONG)
		{
			if((lcdButtonData.button.clickedState == CLICK_NONE || lcdButtonData.button.clickedState == CLICK_EMPTY) && readModbus(MODBUS_READ_ONLY_BIT_DATA_2)&(1<<BIT_MODBUS_READ_ONLY_BIT_DATA_2_ALARM)) 
			{
				alarmBuzzerDeactive();
			}
			else
			{				
				lcdContext = getLcdWorkQueueFree();
				if(lcdContext != NULL)
				{
					if(lcdButtonData.button.clickedState == CLICK_SHORT)
					{
						lcdContext->data[0] = CLICK_LONG;
					}
					else if(lcdButtonData.button.clickedState == CLICK_NONE || lcdButtonData.button.clickedState == CLICK_EMPTY)
					{
						lcdContext->data[0] = CLICK_SHORT;
						lcdContext->data[1] = uartLCD->bufferRx[3];
						lcdContext->data[2] = uartLCD->bufferRx[2];
						lcdContext->data[3] = uartLCD->bufferRx[5];
						lcdContext->data[4] = uartLCD->bufferRx[4];
					}
					lcdContext->state = LCD_STATE_TOUCH;
					lcdContext->substate = LCD_SUBSTATE_TOUCH_START;
					enqueueLcdWork();
				}
			}				
		}
	}
	else if(uartLCD->bufferRx[1] == 0x7B)
	{
		// wyswietlenie obrazu i podanie jego sumy CRC
		uartWAGO->stopReceiving();
		uartWAGO->bufferTx[2] = 2;
		uartWAGO->bufferTx[3] = uartLCD->bufferRx[2];
		uartWAGO->bufferTx[4] = uartLCD->bufferRx[3];
		uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, 5);
 		uartWAGO->bufferTx[5] = (uint8_t)crc2;
 		uartWAGO->bufferTx[6] = (uint8_t)(crc2>>8);
		uartWAGO->startTransmit(7);		
		lcdContext = getLcdWorkQueuePeek();
		if(lcdContext != NULL && lcdContext->state == LCD_STATE_IMAGE_CRC)
		{
			lcdContext->substate = LCD_SUBSTATE_IMAGE_CRC_EXIT;
		}
	}	
	else if(uartLCD->bufferRx[1] == 0xE0)
	{
		//odebrano odpowiedz po ustawieniach konfiguracyjnych:
	}
	else if(uartLCD->bufferRx[1] == 0x90)
	{
		// zapisanie danych do pamieci LCD
		lcdContext = getLcdWorkQueuePeek();
		if(lcdContext != NULL)
		{
			if(lcdContext->state == LCD_STATE_PROJECT_SAVE)
			{
				lcdContext->substate = LCD_SUBSTATE_PROJECT_SAVE_EXIT;
			}
			else if(lcdContext->state == LCD_STATE_PROJECT_CLEAR_NOTUSE)
			{
				lcdContext->substate = LCD_SUBSTATE_PROJECT_CLEAR_NOTUSE_NEXT;
			}			
		}
	}
	else if(uartLCD->bufferRx[1] == 0x91)
	{
		// odczyt danych z pamieci LCD
		uint32_t address = ((uint32_t)uartLCD->bufferRx[2])<<24 | ((uint32_t)uartLCD->bufferRx[3])<<16 | ((uint32_t)uartLCD->bufferRx[4])<<8 | (uint32_t)(uartLCD->bufferRx[5]);
		uint16_t length = ((uint16_t)uartLCD->bufferRx[6])<<8 | uartLCD->bufferRx[7];
		uint8_t * uartLCDdata = uartLCD->bufferRx + 8;
		uint32_t shift = address & 0x0001FFFF;
		uint8_t lcdFlashPage = address / 0x020000;
		lcdContext = getLcdWorkQueuePeek();
		
		_lcdButtonScene * sceneWork;	
		if(lcdFlashPage >= SYSTEM_SCENE_PAGE)
			sceneWork = &lcdButtonData.scene.system;
		else
			sceneWork = &lcdButtonData.scene.user;

		if(lcdContext != NULL)
		{
			if(
				lcdContext->state == LCD_STATE_WAGO_NO_CONNECT ||
				lcdContext->state == LCD_STATE_SCENE_ACTIVE ||
				lcdContext->state == LCD_STATE_PRINT_BUTTONS ||
				lcdContext->state == LCD_STATE_PRINT_BARS ||
				lcdContext->state == LCD_STATE_PRINT_STATES ||
				lcdContext->state == LCD_STATE_PRINT_NUMBERS ||
				lcdContext->state == LCD_STATE_SERVICE_SCENE_ACTIVE ||
				lcdContext->state == LCD_STATE_SERVICE_PRINT_BUTTONS ||
				lcdContext->state == LCD_STATE_SERVICE_PRINT_BARS ||
				lcdContext->state == LCD_STATE_SERVICE_PRINT_STATES ||
				lcdContext->state == LCD_STATE_SERVICE_PRINT_NUMBERS ||
				lcdContext->state == LCD_STATE_INIT_DEVICE_WAKE_UP ||
				lcdContext->state == LCD_STATE_INIT_PID ||
				lcdContext->state == LCD_STATE_PROJECT_CLEAR_NOTUSE ||
				lcdContext->state == LCD_STATE_VALID_MODBUS_DATA)
			{
#ifdef _DEBUG
				if(shift < 0x17EF)
				{
					uint16_t crc = 0;
					for(uint8_t i = 0; i<length; i++)
					{
						crc += uartLCDdata[i];
					}
					crc = (crc & 0xFF);
					
					if(crc != 0)
					{
						//blad sumy CRC


					}
				}
#endif // _DEBUG
				
				
				if(lcdFlashPage < USER_BAR_PAGE || lcdFlashPage == SYSTEM_SCENE_PAGE)
				{
					if(shift == 0 && length == 79)
					{
						if(lcdContext->substate == LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_INIT_DATA ||
							lcdContext->substate == LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_INIT_DATA)
						{
							sceneWork->labelsCount = uartLCDdata[0];
							sceneWork->textsCount = uartLCDdata[1];
							sceneWork->barsCount = uartLCDdata[2];
							sceneWork->statesCount = uartLCDdata[3];
							sceneWork->numbersCount = uartLCDdata[4];
							sceneWork->buttonsCount = uartLCDdata[5];
							memmove(sceneWork->textsLengths, &uartLCDdata[15], BIG_OBJECTS_MAX_COUNT);
							
							//zachowanie w pamięci podrecznej zadania danych dotyczacych grafiki tla
							memmove(&lcdContext->data[LCD_WORK_CONTEXT_BACKGROUND_INDEX], &uartLCDdata[6], 9);
							
							if(lcdFlashPage == SYSTEM_SCENE_PAGE)
							{
								uint16_t y1 = (uint16_t)lcdContext->data[LCD_WORK_CONTEXT_BACKGROUND_INDEX+3] | (uint16_t)lcdContext->data[LCD_WORK_CONTEXT_BACKGROUND_INDEX+4]<<8;
								uint16_t y2 = (uint16_t)lcdContext->data[LCD_WORK_CONTEXT_BACKGROUND_INDEX+7] | (uint16_t)lcdContext->data[LCD_WORK_CONTEXT_BACKGROUND_INDEX+8]<<8;
								lcdButtonData.scene.logoHeight = y2-y1+1;
							}								
							
							lcdContext->backupCounter = 0;
							if(lcdContext->state == LCD_STATE_WAGO_NO_CONNECT)
							{
								lcdContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_READ_BACKGROUND_LOGO;
							}								
							else
								lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_PREPARE_CONTROL_DATA;
							
							
						}
						else if(lcdContext->substate == LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_INIT_DATA)
						{
							//sprawdza czy dana scena jest prawidlowa: czy sa zawarte prawidlowe dane
							if(uartLCDdata[0] < USER_PAGE_GRAPHIC_MAX_COUNT + 1)
							{
								if(uartLCDdata[1] < USER_PAGE_TEXT_MAX_COUNT + 1)
								{
									if(uartLCDdata[2] < USER_PAGE_BAR_MAX_COUNT + 1)
									{
										if(uartLCDdata[3] < USER_PAGE_STATE_MAX_COUNT + 1)
										{											
											if(uartLCDdata[4] < USER_PAGE_NUMBER_MAX_COUNT + 1)
											{
												if(uartLCDdata[5] < USER_PAGE_BUTTON_MAX_COUNT + 1)
												{
													lcdContext->data[0]= uartLCDdata[2];	//ilosc obiektow BAR na scenie
													lcdContext->data[1]= uartLCDdata[3];	//ilosc obiektow STATE na scenie
													lcdContext->data[2]= uartLCDdata[4];	//ilosc obiektow NUMBER na scenie
													lcdContext->data[3]= uartLCDdata[5];	//ilosc obiektow BUTTON na scenie
													lcdContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_INIT_DATA_OK;
												}
											}												
										}	
									
									}
								}
							}	
							else
							{
								lcdContext->substate = LCD_SUBSTATE_VALID_MODBUS_END;
							}		
						}						
					}				
					else if(shift >= 0x4F && shift < 0xA27)
 					{
						if(lcdContext->state == LCD_STATE_VALID_MODBUS_DATA)
						{
							memmove(lcdContext->data, uartLCDdata, 19);	//kopiowana jest najwieksza ilosc danych mozliwych do analizy
							if(lcdContext->substate == LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BAR_CONTROL_DATA)
							{
								lcdContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BAR_CONTROL_DATA_OK;
							}
							else if(lcdContext->substate == LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_NUMBER_CONTROL_DATA)
							{
								lcdContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_NUMBER_CONTROL_DATA_OK;
							}
							else if(lcdContext->substate == LCD_SUBSTATE_VALID_MODBUS_WATCHDOG_BUTTON_CONTROL_DATA)
							{
								lcdContext->substate = LCD_SUBSTATE_VALID_MODBUS_READ_BUTTON_CONTROL_DATA_OK;
							}							
						}
						else
						{
							if(lcdContext->data[0] == LCD_OBJECT_BAR)
							{
								sceneWork->firstBar->x = (uint16_t)uartLCDdata[0] | ((uint16_t)(uartLCDdata[1])<<8);
								sceneWork->firstBar->y = (uint16_t)uartLCDdata[2] | ((uint16_t)(uartLCDdata[3])<<8);
								sceneWork->firstBar->status = uartLCDdata[4];
								sceneWork->firstBar->min = (uint16_t)uartLCDdata[5] | ((uint16_t)(uartLCDdata[6])<<8);
								sceneWork->firstBar->max = (uint16_t)uartLCDdata[7] | ((uint16_t)(uartLCDdata[8])<<8);
								sceneWork->firstBar->graphId = (uint16_t)uartLCDdata[9] | ((uint16_t)(uartLCDdata[10])<<8);
								sceneWork->firstBar->mbWord = uartLCDdata[11];
								sceneWork->firstBar->value = readModbus(sceneWork->firstBar->mbWord);
								sceneWork->firstBar++;
								sceneWork->firstState = (_stateData *)sceneWork->firstBar;
								sceneWork->firstNumber = (_numberData *)sceneWork->firstBar;
								sceneWork->firstButton = (_buttonData *)sceneWork->firstBar;
								lcdContext->backupCounter = 0;
								lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA;
							}
							else if(lcdContext->data[0] == LCD_OBJECT_STATE)
							{
								sceneWork->firstState->x = (uint16_t)uartLCDdata[0] | ((uint16_t)(uartLCDdata[1])<<8);
								sceneWork->firstState->y = (uint16_t)uartLCDdata[2] | ((uint16_t)(uartLCDdata[3])<<8);
								sceneWork->firstState->status = uartLCDdata[4];
								sceneWork->firstState->graphId = (uint16_t)uartLCDdata[5] | ((uint16_t)(uartLCDdata[6])<<8);
								sceneWork->firstState->mbWord = uartLCDdata[7];
								sceneWork->firstState->mbBit = uartLCDdata[8];
								sceneWork->firstState->value = ((readModbus(sceneWork->firstState->mbWord) & (1<<(sceneWork->firstState->mbBit))) != 0);
								sceneWork->firstState++;
								sceneWork->firstNumber = (_numberData *)sceneWork->firstState;
								sceneWork->firstButton = (_buttonData *)sceneWork->firstState;
								lcdContext->backupCounter = 0;
								lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA;
							}
							else if(lcdContext->data[0] == LCD_OBJECT_NUMBER)
							{
								sceneWork->firstNumber->x = (uint16_t)uartLCDdata[0] | ((uint16_t)(uartLCDdata[1])<<8);
								sceneWork->firstNumber->y = (uint16_t)uartLCDdata[2] | ((uint16_t)(uartLCDdata[3])<<8);
								sceneWork->firstNumber->status = uartLCDdata[4];
								sceneWork->firstNumber->min = (uint16_t)uartLCDdata[5] | ((uint16_t)(uartLCDdata[6])<<8);
								sceneWork->firstNumber->max = (uint16_t)uartLCDdata[7] | ((uint16_t)(uartLCDdata[8])<<8);
								sceneWork->firstNumber->graphId = (uint16_t)uartLCDdata[9] | ((uint16_t)(uartLCDdata[10])<<8);
								sceneWork->firstNumber->mbWord = uartLCDdata[11];
								sceneWork->firstNumber->value = readModbus(sceneWork->firstNumber->mbWord);
								sceneWork->firstNumber++;
								sceneWork->firstButton = (_buttonData *)sceneWork->firstNumber;
								lcdContext->backupCounter = 0;
								lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA;
							}
							else if(lcdContext->data[0] == LCD_OBJECT_BUTTON)
							{
								sceneWork->firstButton->xLGR = (uint16_t)uartLCDdata[0] | ((uint16_t)(uartLCDdata[1])<<8);
								sceneWork->firstButton->yLGR = (uint16_t)uartLCDdata[2] | ((uint16_t)(uartLCDdata[3])<<8);
								sceneWork->firstButton->xPDR = (uint16_t)uartLCDdata[4] | ((uint16_t)(uartLCDdata[5])<<8);
								sceneWork->firstButton->yPDR = (uint16_t)uartLCDdata[6] | ((uint16_t)(uartLCDdata[7])<<8);
								sceneWork->firstButton->type = uartLCDdata[8];
								memmove(&sceneWork->firstButton->data, &uartLCDdata[9], 9);
								if(sceneWork->firstButton->type == BUT_CODE_SCENE ||
								sceneWork->firstButton->type == BUT_CODE_ROCKER ||
								sceneWork->firstButton->type == BUT_CODE_EMPTY ||
								sceneWork->firstButton->type == BUT_CODE_PASS_CLR ||
								sceneWork->firstButton->type == BUT_CODE_PASS_SET ||
								sceneWork->firstButton->type == BUT_CODE_PASS_NUM)
								{
									sceneWork->firstButton->value = 0;
								}
								sceneWork->firstButton->timedown = 0;
								sceneWork->firstButton->timedown2 = 0;
								sceneWork->firstButton++;
								lcdContext->backupCounter = 0;
								lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_READ_CONTROL_DATA;
							}
						}							
					}
					else if(shift >= 0x0A27 && shift < 0x17EF)
					{
						if(lcdContext->data[0] == LCD_OBJECT_IMAGE)
 						{
							 
							if(lcdContext->state == LCD_STATE_WAGO_NO_CONNECT)
							{
								lcdContext->backupSubstate = lcdContext->data[1];
								lcdContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_PRINT;
							}
							else
							{
								lcdContext->backupSubstate = LCD_SUBSTATE_SCENE_ACTIVE_READ_IMAGE;
								lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_SEND_GRAPHIC;	
							}
							memmove(lcdContext->data, &uartLCDdata[0], 13);
							lcdContext->backupCounter = 0;							
 						}
						else if(lcdContext->data[0] == LCD_OBJECT_BUTTON)
 						{
							_buttonData * buttData = sceneWork->firstButton+lcdContext->data[1];
							memmove(lcdContext->data, &uartLCDdata[0], 9);
							lcdContext->data[9] = (uint8_t) (buttData->xLGR);
							lcdContext->data[10] = (uint8_t) (buttData->xLGR>>8);
							lcdContext->data[11] = (uint8_t) (buttData->yLGR);
							lcdContext->data[12] = (uint8_t) (buttData->yLGR>>8);
							lcdContext->backupCounter = 0;
							lcdContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_EXEC_GRAPHIC;
 						}
					}
					else if(shift >= 0x17EF)
					{
						if(lcdContext->substate == LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_TEXT ||
							lcdContext->substate == LCD_SUBSTATE_PRINT_OBJECTS_WATCHDOG ||
							lcdContext->substate == LCD_SUBSTATE_WAGO_NO_CONNECT_WATCHDOG_TEXT_GRAPHIC)
						{
							memmove(lcdContext->data, &uartLCDdata[0], 8);							
							memmove(lcdButtonData.tekst, &uartLCDdata[8], uartLCD->counterRx);
							lcdContext->backupCounter = 0;
							if(lcdContext->substate == LCD_SUBSTATE_SCENE_ACTIVE_WATCHDOG_TEXT)
							{
								lcdContext->substate = LCD_SUBSTATE_SCENE_ACTIVE_SEND_TEXT;
							}								
							else if(lcdContext->substate == LCD_SUBSTATE_PRINT_OBJECTS_WATCHDOG)
							{
								lcdContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_EXEC_TEXT;
							}
							else //if(lcdContext->substate == LCD_SUBSTATE_WAGO_NO_CONNECT_PRINT)
							{
								//lcdContext->backupSubstate = LCD_SUBSTATE_WAGO_NO_CONNECT_LOGO_WAIT;
								lcdContext->substate = LCD_SUBSTATE_WAGO_NO_CONNECT_SEND_TEXT;
							}
						}
						
					}
				}
				else if(lcdFlashPage < USER_STATE_PAGE || lcdFlashPage == SYSTEM_BAR_PAGE)
				{
					if(lcdContext->data[0] == LCD_OBJECT_BAR)
					{
						_barData * barData = sceneWork->firstBar+lcdContext->data[1];
						memmove(lcdContext->data, &uartLCDdata[0], 9);
						lcdContext->data[9] = (uint8_t) (barData->x);
						lcdContext->data[10] = (uint8_t) (barData->x>>8);
						lcdContext->data[11] = (uint8_t) (barData->y);
						lcdContext->data[12] = (uint8_t) (barData->y>>8);
						lcdContext->backupCounter = 0;
						lcdContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_EXEC_GRAPHIC;;
					}
				}
				else if(lcdFlashPage == USER_STATE_PAGE || lcdFlashPage == SYSTEM_STATE_PAGE)
				{
					if(lcdContext->data[0] == LCD_OBJECT_STATE)
					{
						_stateData * stateData = sceneWork->firstState+lcdContext->data[1];
						memmove(lcdContext->data, &uartLCDdata[0], 9);
						lcdContext->data[9] = (uint8_t) (stateData->x);
						lcdContext->data[10] = (uint8_t) (stateData->x>>8);
						lcdContext->data[11] = (uint8_t) (stateData->y);
						lcdContext->data[12] = (uint8_t) (stateData->y>>8);
						lcdContext->backupCounter = 0;
						lcdContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_EXEC_GRAPHIC;
					}
				}
				else if(lcdFlashPage == USER_NUMBER_PAGE || lcdFlashPage == SYSTEM_NUMBER_PAGE)
				{
					if(lcdContext->data[0] == LCD_OBJECT_NUMBER)
					{
						_numberData * numberData = sceneWork->firstNumber+lcdContext->data[1];
						memmove(lcdContext->data, &uartLCDdata[0], 9);						
						uint16_t x = *((uint16_t *)(&lcdContext->data[LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX]));
						uint16_t prevDelta_x = *((uint16_t *)(&lcdContext->data[LCD_WORK_CONTEXT_NUMBER_DELTA_X_PREVIOUS_INDEX]));
						uint16_t graph_x0 = ((uint16_t)lcdContext->data[2])<<8 | (uint16_t)lcdContext->data[1];
						uint16_t graph_x1 = ((uint16_t)lcdContext->data[6])<<8 | (uint16_t)lcdContext->data[5];
						uint16_t delta_x = graph_x1 - graph_x0 + 1;
						if(numberData->status & 1<<2)
						{
							*((uint16_t *)(&lcdContext->data[LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX])) = x + prevDelta_x;
						}							
						else
						{
							*((uint16_t *)(&lcdContext->data[LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX])) = x - delta_x;
						}
						*((uint16_t *)(lcdContext->data + LCD_WORK_CONTEXT_NUMBER_DELTA_X_PREVIOUS_INDEX)) = delta_x;
						lcdContext->data[9] = lcdContext->data[LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX];
						lcdContext->data[10] = lcdContext->data[LCD_WORK_CONTEXT_NUMBER_X_ACTUAL_INDEX+1];
						lcdContext->data[11] = (uint8_t) (numberData->y);
						lcdContext->data[12] = (uint8_t) ((numberData->y)>>8);
						lcdContext->backupCounter = 0;
						lcdContext->data[LCD_WORK_CONTEXT_NUMBER_DIGIT_ACTUAL_INDEX]++;
						lcdContext->substate = LCD_SUBSTATE_PRINT_OBJECTS_EXEC_GRAPHIC;
					}
				}
				else if(lcdFlashPage == COMMON_FUNCTIONS_PAGE)
				{
					if(shift == 0)
					{
						//dane zwiazane z wybudzeniem urzadzenia:
						if(uartLCDdata[0] == 'O' && uartLCDdata[1] == 'K')
						{						
							if(uartLCDdata[2] < SCENES_COUNT)
							{
								lcdButtonData.sceneSwitchFlag = 1;
								lcdButtonData.sceneSwitchNumber = uartLCDdata[2];
							}
							else
							{
								lcdButtonData.sceneSwitchFlag = 0;
							}
							if(uartLCDdata[3] < LIGHT_SCENE_COUNT)
							{
								lcdButtonData.lightSceneSwitchFlag = 1;
								lcdButtonData.lightSceneSwitchMask = (1 << uartLCDdata[3]);
							}
							else
							{
								lcdButtonData.lightSceneSwitchFlag = 0;
							}						
							if(uartLCDdata[4] <= WAKE_UP_REGISTERS_COUNT)
							{
								lcdButtonData.newValuesCount = uartLCDdata[4];
								uint8_t * dataPnt = &uartLCDdata[5];
								for(int i=0; i<lcdButtonData.newValuesCount; i++)
								{
									lcdButtonData.wakeUpMbAddresses[i] = *dataPnt;
									dataPnt+=1;
									lcdButtonData.wakeUpMbValues[i] = *((uint16_t *)dataPnt);
									dataPnt+=2;
								}
							}
							else
							{
								lcdButtonData.newValuesCount = 0;
							}
						}
						lcdContext->substate = LCD_SUBSTATE_INIT_WAKE_UP_END;
					}
					else if(shift == 54)
					{
						//dane zwiazane z regulatorami PID:
						if(uartLCDdata[0] == 'O' && uartLCDdata[1] == 'K')
						{						
							uint8_t * dataPnt = &uartLCDdata[2];
							lcdButtonData.pidCount = *dataPnt;
							dataPnt++;
							for(int i=0; i<lcdButtonData.pidCount; lcdButtonData.pidCount++)
							{
								lcdButtonData.pidData[i].config = *dataPnt;
								dataPnt++;
								lcdButtonData.pidData[i].paramKp = *((uint16_t *)dataPnt);
								dataPnt+=2;
								lcdButtonData.pidData[i].paramTi = *((uint16_t *)dataPnt);
								dataPnt+=2;
								lcdButtonData.pidData[i].paramTd = *((uint16_t *)dataPnt);
								dataPnt+=2;
								lcdButtonData.pidData[i].setPoint = *((uint16_t *)dataPnt);
								dataPnt+=2;
								lcdButtonData.pidData[i].actualAddress = *dataPnt;
								dataPnt++;
								lcdButtonData.pidData[i].outputAddress = *dataPnt;
								dataPnt++;
								lcdButtonData.pidData[i].min = *dataPnt;
								dataPnt++;
								lcdButtonData.pidData[i].max = *dataPnt;
								dataPnt++;
								lcdButtonData.pidData[i].activatorAddress = *dataPnt;
								dataPnt++;
								lcdButtonData.pidData[i].activatorBitNumber = *dataPnt;
								dataPnt++;
							}
						}
						
					}
				}				
 			}
		}
	}
	else
	{
// 		uartWAGO->stopReceiving();
// 		memmove(uartWAGO->buffer, uartLCD->buffer, uartLCD->counterRx);
// 		uartWAGO->startTransmit(uartLCD->counterRx);
	}
	
	return bytesToSend;
}
//////////////////////////////////////////////////////////////////////////
int lcdFinishFrame(int dataLength)
{
	uartLCD->bufferTx[0] = LCD_FRAME_START;
	uartLCD->bufferTx[dataLength++] = LCD_FRAME_END_1;
	uartLCD->bufferTx[dataLength++] = LCD_FRAME_END_2;
	uartLCD->bufferTx[dataLength++] = LCD_FRAME_END_3;
	uartLCD->bufferTx[dataLength++] = LCD_FRAME_END_4;
	return dataLength;
}
//////////////////////////////////////////////////////////////////////////
void lcdSendHandshake(void)
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
 	uartLCD->bufferTx[frameSize++] = 0;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendConfig(_baudRate baudRate, uint8_t options)
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0xE0;
	uartLCD->bufferTx[frameSize++] = 0x55;
	uartLCD->bufferTx[frameSize++] = 0xAA;
	uartLCD->bufferTx[frameSize++] = 0x5A;
	uartLCD->bufferTx[frameSize++] = 0xA5;
	uartLCD->bufferTx[frameSize++] = lcdData.id;
	uartLCD->bufferTx[frameSize++] = protocolLcdBaudRate2Data(baudRate);
	uartLCD->bufferTx[frameSize++] = options;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendColorPalette(uint16_t foregroundColor, uint16_t backgroundColor)
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0x40;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(foregroundColor>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)foregroundColor;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(backgroundColor>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)backgroundColor;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendClearScreen()
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0x52;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
//void lcdSendDirectVideo(uint8_t * compresDataPnt)
uint16_t lcdSendDirectVideo(uint8_t * compresDataPnt)
{	
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	int seriesCount;
	uint8_t pixelsCount;
	uint8_t repeatFlag;
	uartLCD->bufferTx[frameSize++] = 0x72;
	uartLCD->bufferTx[frameSize++] = *(compresDataPnt+2);
	uartLCD->bufferTx[frameSize++] = *(compresDataPnt+1);
	uartLCD->bufferTx[frameSize++] = *(compresDataPnt+0);
	compresDataPnt+=3;
	seriesCount = *compresDataPnt++;
	for(int i=0; i<seriesCount; i++)
	{
		pixelsCount = *compresDataPnt++;
		if(pixelsCount & 0x80)
			repeatFlag = 0;
		else
			repeatFlag = 1;
		pixelsCount = pixelsCount & (~0x80);
		if(repeatFlag)
		{
			//powtorzenia:
			for(int j=0; j<pixelsCount; j++)
			{
				uartLCD->bufferTx[frameSize++] = *(compresDataPnt+1);
				uartLCD->bufferTx[frameSize++] = *(compresDataPnt+0);
			}
			compresDataPnt+=2;
		}
		else
		{
			//pojedyncze:
			for(int j=0; j<pixelsCount; j++)
			{
				uartLCD->bufferTx[frameSize++] = *(compresDataPnt+1);
				uartLCD->bufferTx[frameSize++] = *(compresDataPnt+0);
				compresDataPnt+=2;
			}
			
		}		
	}
 	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
	
	//
	return frameSize;
}
//////////////////////////////////////////////////////////////////////////
void lcdSendImageSave(uint8_t screenId)
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0xE2;
	uartLCD->bufferTx[frameSize++] = screenId;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendImagePrint(uint8_t screenId)
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0x70;
	uartLCD->bufferTx[frameSize++] = screenId;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendImagePrintWithCrc(uint8_t screenId)
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0x7B;
	uartLCD->bufferTx[frameSize++] = screenId;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendProjectData(uint8_t * framePnt, uint8_t dataSize)
{	
	uartLCD->stopReceiving();
	
	if(dataSize > 10)
	{
		if(framePnt[0] == 0xAA &&
			framePnt[1] == 0x90 &&
			framePnt[2] == 0x55 &&
			framePnt[3] == 0xAA &&
			framePnt[4] == 0x5A &&
			framePnt[5] == 0xA5 &&
			framePnt[6] == 0x00 &&
			(framePnt[7] & 0x01) == 0 &&
			framePnt[8] == 0x00 &&
			framePnt[9] == 0x00
			)
		{
			int8_t sceneIndex = framePnt[7] >> 1;
			if(sceneIndex < SCENES_COUNT)
			{
				if(!lcdButtonData.config.newUserConfigFlag)
				{
					lcdButtonData.config.newUserConfigFlag = 1;
					lcdButtonData.config.lastUserSceneSave = sceneIndex;
				}	
				if(sceneIndex > lcdButtonData.config.lastUserSceneSave)
				{
					lcdButtonData.config.lastUserSceneSave = sceneIndex;
				}
			}
		}
	}
	memmove(uartLCD->bufferTx, framePnt, dataSize);
	uartLCD->startTransmit(dataSize);
}
/////////////////////////////////////////////////////////////////////////
void lcdSendDataRead(uint32_t address, uint16_t dataLength)
{
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0x91;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(address>>24);
	uartLCD->bufferTx[frameSize++] = (uint8_t)(address>>16);
	uartLCD->bufferTx[frameSize++] = (uint8_t)(address>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)address;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(dataLength>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)dataLength;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
/////////////////////////////////////////////////////////////////////////
void lcdSendSceneDataRead(uint32_t relAddress, uint16_t dataLength, uint8_t systemSceneFlag)
{
	uint32_t address;
	if(systemSceneFlag)
		address = (uint32_t)SYSTEM_SCENE_PAGE * 0x020000 | relAddress;
	else
		address = (uint32_t)lcdButtonData.scene.activeScene * 0x020000 | relAddress;
	lcdSendDataRead(address, dataLength);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendDisplayIcon(uint8_t screenId, uint16_t xLGRcut, uint16_t yLGRcut, uint16_t xPDRcut, uint16_t yPDRcut, uint16_t xLGRpaste, uint16_t yLGRpaste)
{
	if((xPDRcut-xLGRcut == 0) || (yPDRcut-yLGRcut)==0)
		return;
		
	uint8_t logoShift = 0;
	if(screenId < SERVICE_SCREEN_ID)
		logoShift = lcdButtonData.scene.logoHeight;
	
	yLGRpaste += logoShift;
	int frameSize = LCD_DATA;
	uartLCD->stopReceiving();
	uartLCD->bufferTx[frameSize++] = 0x71;
	uartLCD->bufferTx[frameSize++] = screenId;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(xLGRcut>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)xLGRcut;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(yLGRcut>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)yLGRcut;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(xPDRcut>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)xPDRcut;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(yPDRcut>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)yPDRcut;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(xLGRpaste>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)xLGRpaste;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(yLGRpaste>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)yLGRpaste;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendDisplayText(uint16_t x, uint16_t y, uint8_t library, uint8_t fontSize, uint16_t color, uint8_t * data, uint8_t maxTextLength, uint8_t systemSceneFlag)
{
	data = lcdButtonData.tekst;
	uartLCD->stopReceiving();
	int textSize = 0;
	maxTextLength *= 2;		//wyrazone w ilosci bajtow
	while((data[textSize] != 0 || data[textSize+1]!=0) && textSize<maxTextLength)
		textSize+=2;
	memmove(uartLCD->bufferTx + 13, data, textSize);
	
	uint8_t logoShift = lcdButtonData.scene.logoHeight;
	if(systemSceneFlag)
		logoShift = 0;
		
	int frameSize = LCD_DATA;
	uartLCD->bufferTx[frameSize++] = 0x98;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(x>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)x;
	y += logoShift;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(y>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)y;
	uartLCD->bufferTx[frameSize++] = library;
	uartLCD->bufferTx[frameSize++] = 0x85;
	uartLCD->bufferTx[frameSize++] = fontSize;
	uartLCD->bufferTx[frameSize++] = (uint8_t)(color>>8);
	uartLCD->bufferTx[frameSize++] = (uint8_t)color;
	uartLCD->bufferTx[frameSize++] = 0;
	uartLCD->bufferTx[frameSize++] = 0;
	frameSize+=textSize;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendBarGraphicRead(uint16_t barId, uint8_t barValue, uint8_t systemSceneFlag)
{
	uint8_t page = barId / 100;
	uint8_t barIdOnPage = barId % 100;
	uint32_t address;
	if(systemSceneFlag)
		address = (uint32_t)SYSTEM_BAR_PAGE * 0x020000 + 0x0020000 * (uint32_t)page + 1280 * (uint32_t)barIdOnPage + 10 * (uint32_t) barValue;
	else
		address = (uint32_t)USER_BAR_PAGE * 0x020000 + 0x0020000 * (uint32_t)page + 1280 * (uint32_t)barIdOnPage + 10 * (uint32_t) barValue;
	lcdSendDataRead(address, 10);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendStateGraphicRead(uint16_t stateId, uint8_t stateValue, uint8_t systemSceneFlag)
{	
	uint32_t address;
	if(systemSceneFlag)
		address = (uint32_t)SYSTEM_STATE_PAGE * 0x020000 + 20 * (uint32_t)stateId;
	else
		address = (uint32_t)USER_STATE_PAGE * 0x020000 + 20 * (uint32_t)stateId;
	if(stateValue!=0)
		address +=10;
	lcdSendDataRead(address, 10);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendNumberGraphicRead(uint16_t numberId, uint8_t numberValue, uint8_t systemSceneFlag)
{
	uint32_t address;
	if(systemSceneFlag)
		address = (uint32_t)SYSTEM_NUMBER_PAGE * 0x020000 + 130 * (uint32_t)numberId + 10 * (uint32_t)numberValue;
	else
		address = (uint32_t)USER_NUMBER_PAGE * 0x020000 + 130 * (uint32_t)numberId + 10 * (uint32_t)numberValue;
	lcdSendDataRead(address, 10);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendSceneButtonGraphicRead(uint16_t buttonId, uint8_t buttonValue, uint8_t systemSceneFlag)
{
	uint32_t valueShift;
	if(buttonValue == 0)
		valueShift = 0;
	else
		valueShift = 10;
	uint32_t relAddress = 0x00000A27+(uint32_t)lcdButtonData.scene.user.labelsCount*14 + buttonId * 20 + valueShift;
	lcdSendSceneDataRead(relAddress, 10, systemSceneFlag);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendSceneButtonTextRead(uint16_t buttonId, uint8_t systemSceneFlag)
{
	_lcdButtonScene * sceneWork;
	if(systemSceneFlag)
		sceneWork = &lcdButtonData.scene.system;
	else
		sceneWork = &lcdButtonData.scene.user;
	uint32_t relAddress = 0x000017EF + (uint32_t)lcdButtonData.scene.activeLang * 6993 + (uint32_t)(sceneWork->textsCount + buttonId) * 111;
	lcdSendSceneDataRead(relAddress, 8 + sceneWork->textsLengths[sceneWork->textsCount + buttonId]*2, systemSceneFlag);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendSceneTextRead(uint16_t textId, uint8_t systemSceneFlag)
{
	_lcdButtonScene * sceneWork;
	if(systemSceneFlag)
		sceneWork = &lcdButtonData.scene.system;
	else
		sceneWork = &lcdButtonData.scene.user;
	uint32_t relAddress = 0x000017EF + (uint32_t)lcdButtonData.scene.activeLang * 6993 +(uint32_t)(textId) * 111;
	lcdSendSceneDataRead(relAddress, 8 + sceneWork->textsLengths[textId]*2, systemSceneFlag);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendTouchBuzzorOn(uint8_t duration)
{
	uartLCD->stopReceiving();
	int frameSize = LCD_DATA;
	uartLCD->bufferTx[frameSize++] = 0x79;
	uartLCD->bufferTx[frameSize++] = duration;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
void lcdSendBrightness(uint16_t percentVal)
{
	uartLCD->stopReceiving();
	int frameSize = LCD_DATA;
	uartLCD->bufferTx[frameSize++] = 0x5F;
	if(percentVal > 100)
		percentVal = 100;
	float val = percentVal;
	val = val*0.63;	//maksymlna wartość bazwzględna jasności: 0x3F (63) stąd przelicznik 1% = 0.63
	uartLCD->bufferTx[frameSize++] = (uint8_t)val;
	frameSize = lcdFinishFrame(frameSize);
	uartLCD->startTransmit(frameSize);
}
//////////////////////////////////////////////////////////////////////////
