﻿/*!
 *  @file protocolLcd.h
 *	@brief Obsługa protokołu komunikacji z wyświetlaczem LCD
 */ 


#ifndef PROTOCOLLCD_H_
#define PROTOCOLLCD_H_

#include <stdint.h>
#include "uart.h"


#define USER_SCENE_PAGE								0
#define USER_BAR_PAGE								(SCENES_COUNT)		/*! < Strona flash w lcd pod ktora zapisana jest scena serwisowa*/
#define USER_STATE_PAGE								74
#define USER_NUMBER_PAGE							75
#define COMMON_FUNCTIONS_PAGE						76		/*! < Strona z funkcjonalnosciami wspolnymi dla wszystkich scen, czyli: konfiguracja zachowania po wybudzeniu, algorytmy PID, automatyzacja wyjsc */
#define SYSTEM_SCENE_PAGE							235		/*! < Strona flash w lcd pod ktora zapisana jest scena serwisowa*/
#define SYSTEM_BAR_PAGE								236
#define SYSTEM_STATE_PAGE							237
#define SYSTEM_NUMBER_PAGE							238

#define USER_PAGE_GRAPHIC_MAX_COUNT					SMALL_OBJECTS_MAX_COUNT
#define USER_PAGE_TEXT_MAX_COUNT					(BIG_OBJECTS_MAX_COUNT)
#define USER_PAGE_BAR_MAX_COUNT						(BIG_OBJECTS_MAX_COUNT)
#define USER_PAGE_STATE_MAX_COUNT					MEDIUM_OBJECTS_MAX_COUNT
#define USER_PAGE_NUMBER_MAX_COUNT					(BIG_OBJECTS_MAX_COUNT)
#define USER_PAGE_BUTTON_MAX_COUNT					(BIG_OBJECTS_MAX_COUNT)

/*! 
	@def SERVICE_SCREEN_ID
    @brief Id screenu od ktorego zaczynaja sie grafiki systemowe
*/
#define SERVICE_SCREEN_ID							201


/*! 
	@def LCD_DATA
    @brief Numer bajtu, od ktorego zaczynają sie dane ramki
*/
#define LCD_DATA									1

/*! 
	@def LCD_FRAME_START
    @brief Bajt rozpoczynający ramkę (komunikacja z LCD).
*/
#define LCD_FRAME_START								0xAA

/*! 
	@def LCD_FRAME_END_1
    @brief Pierwszy bajt kończący ramkę (komunikacja z LCD).
*/
#define LCD_FRAME_END_1								0xCC

/*! 
	@def LCD_FRAME_END_2
    @brief Drugi bajt kończący ramkę (komunikacja z LCD).
*/
#define LCD_FRAME_END_2								0x33

/*! 
	@def LCD_FRAME_END_3
    @brief Trzeci bajt kończący ramkę (komunikacja z LCD).
*/
#define LCD_FRAME_END_3								0xC3

/*! 
	@def LCD_FRAME_END_4
    @brief Czwarty bajt kończący ramkę (komunikacja z LCD).
*/
#define LCD_FRAME_END_4								0x3C

/*!
	@brief Analiza ramki zgodnie z protokołem wyświetlacza LCD
	
	Metoda analizuje otrzymaną ramkę i w razie potrzeby przygotowuje odpowiedź.
	@return Ilość bajtów jaka jest przygotowana do odesłania. W przypadku gdy nie potrzebna jest odpowiedź,
		zwracana jest wartość 0.
*/
uint16_t parseFrameLcd(void);

/*!
	@brief Wykrywanie poprawnej ramki zgodnej z protokołem wyświetlacza LCD
	
	Metoda analizuje czy we wskazanej strukturze @ref _uartData znajduje się poprawnie odczytana ramka. Pierwszy bajt ramki musi znajdować się
	w pierwszym bajcie bufora danych, w związku z tym niezbędne jest wykrywanie braku
	komunikacji i zerowanie wskażników danych.
	@return 0 - nie wykryto ramki; @n 1 - wykryto ramkę.
*/
uint8_t isFrameDetectedLcd();

/*!
	@brief Wyslanie polecenia Handshake
*/
void lcdSendHandshake(void);

/*!
	@brief Wysłanie danych konfiguracyjnych
	@param baudRate Wartość prędkości komunikacji do ustawienia
	@param options Wartość opcji do ustawienia
*/
void lcdSendConfig(_baudRate baudRate, uint8_t options);

/*!
	@brief Ustawienie palety kolorow
	@param foregroundColor Kolor napislów
	@param backgroundColor Kolor tła
*/
void lcdSendColorPalette(uint16_t foregroundColor, uint16_t backgroundColor);

/*!
	@brief Wyczyszczenie ekranu kolorem  tła
*/
void lcdSendClearScreen(void);

/*!
	@brief Wyświetlenie pikseli na ekranie
	@param compresDataPnt Wskaźnik do skompresowanych danych
*/
//void lcdSendDirectVideo(uint8_t * compresDataPnt);
uint16_t lcdSendDirectVideo(uint8_t * compresDataPnt);

/*!
	@brief Zapisanie aktualnego na wyświetlaczu ekranu do pamięci flasz pod wskazane miejsce
	@param screenId Identyfikator miejsca pod które należy zapisać grafikę z danego ekranu
*/
void lcdSendImageSave(uint8_t screenId);

/*!
	@brief Wyświetlenie danego ekranu i pobranie jego crc
	
	Suma CRC danego ekranu ma postać x^16 + x^15 + x^2 + 1 (Modbus).
	Dla każdego piksela pobierany jest najpierw bajt mniej znaczący, później bajt bardziej znaczący.
	
	@param screenId Identyfikator ekranu
*/
void lcdSendImagePrintWithCrc(uint8_t screenId);

/*!
	@brief Pobranie koloru danego piksela
	
	@param screenId Identyfikator ekranu
	@param x Współrzędna X
	@param y Współrzędna Y
	@param background Flaga określająca czy należy pobrać kolor tła (!0) czy napisu (0)
*/
void lcdSendGetPixelColor(uint16_t x, uint16_t y, uint8_t background);

/*!
	@brief Wysłanie danych do zapisu w pamięci flash panelu LCD
	@param framePnt Wskaźnik na dane będące kompletną ramką jaką należy wysłać do wyświetlacza
	@param dataSize Ilość danych które mają być wysłane do wyświetlacza
*/
void lcdSendProjectData(uint8_t * framePnt, uint8_t dataSize);

/*!
	@brief Odczytanie dowolnych danych z pamięci FLASH wyświetlaczu
	@brief address Adres danych które należy odczytać z pamięci flash wyświetlacza LCD
	@brief dataLength ilość danych które mają być odczytane
*/
void lcdSendDataRead(uint32_t address, uint16_t dataLength);

/*!
	@brief Odczytanie dowolnych danych z aktualnie wybranej sceny
	@param relAddress Adres względny danych
	@param dataLength Ilość danych do przeczytania
*/
void lcdSendSceneDataRead(uint32_t relAddress, uint16_t dataLength, uint8_t systemSceneFlag);

/*!
	@brief Wycięcie i wyświetlenie ikony na wyświetlaczu
	@param screenId
	@param xLGRcut
	@param yLGRcut
	@param xPDRcut
	@param yPDRcut
	@param xLGRpaste
	@param yLGRpaste
*/
void lcdSendDisplayIcon(uint8_t screenId, uint16_t xLGRcut, uint16_t yLGRcut, uint16_t xPDRcut, uint16_t yPDRcut, uint16_t xLGRpaste, uint16_t yLGRpaste);

/*!
	@brief Wyświetlenie tekstu
	@param x
	@param y
	@param library
	@param fontSize
	@param color
	@param data
	@param maxTextLength	maksymalna ilość znaków Unicode dla danego języka w różnych językach
*/
void lcdSendDisplayText(uint16_t x, uint16_t y, uint8_t library, uint8_t fontSize, uint16_t color, uint8_t * data, uint8_t maxTextLength, uint8_t systemSceneFlag);

/*!
	@brief Funkcja wysyła pytanie o dane grafiki dla odpowiedniej wartości wskazanego obiektu BAR
	@param barId Numer porządkowy obiektu BAR <0;99>
	@param barValue Numer kolejnej grafiki <0;127>
*/
void lcdSendBarGraphicRead(uint16_t barId, uint8_t barValue, uint8_t systemSceneFlag);

/*!
	@brief Funkcja wysyła pytanie o dane grafiki dla odpowiedniej wartości wskazanego obiektu STATE
	@param stateId Numer porządkowy obiektu STATE <0;6552>
	@param stateValue Numer kolejnej grafiki <0;1>
*/
void lcdSendStateGraphicRead(uint16_t stateId, uint8_t stateValue, uint8_t systemSceneFlag);

/*!
	@brief Funkcja wysyła pytanie o dane grafiki dla odpowiedniej wartości wskazanego obiektu NUMBER
	@param numberId Numer porządkowy obiektu bar <0;6552>
	@param numberValue Numer kolejnej grafiki <0;12>
*/
void lcdSendNumberGraphicRead(uint16_t numberId, uint8_t numberValue, uint8_t systemSceneFlag);

/*!
	@brief Funkcja wysyła pytanie o dane grafiki dla obiektu BUTTON z aktualnej sceny
	@param butttonId Numer porządkowy obiektu BUTTON <0;63>
	@param buttonValue Numer kolejnej grafiki: 0 (wyłączony), !0(włączony).
*/
void lcdSendSceneButtonGraphicRead(uint16_t buttonId, uint8_t buttonValue, uint8_t systemSceneFlag);

/*!
	@brief Funkcja wysyła pytanie o dane tekstu dla obiektu BUTTON z aktualnej sceny
	@param butttonId Numer porządkowy obiektu BUTTON <0;63>
*/
void lcdSendSceneButtonTextRead(uint16_t buttonId, uint8_t systemSceneFlag);

/*!
	@brief Funkcja wysyła pytanie o dane tekstu dla obiektu LABEL z aktualnej sceny
	@param textId Numer porządkowy obiektu LABEL <0;63>
*/
void lcdSendSceneTextRead(uint16_t textId, uint8_t systemSceneFlag);

/**
 * @brief Aktywacja buzzera
 */
void lcdSendTouchBuzzorOn(uint8_t duration);

/*!
	@brief Funkcja wysyła polecenie ustawiające jasność świecenia wyświetlacza
	@param percentVal jasność świecenia <0;100>[%]
*/
void lcdSendBrightness(uint16_t percentVal);

extern _uartData * uartLCD;
extern _uartData * uartWAGO;

#endif /* PROTOCOLLCD_H_ */
