﻿#include <avr/io.h>
#include <stdint.h>
#include <string.h>
#include "protocolWago.h"
#include "compilationFlags.h"
#include "crc.h"
#include "uart.h"
#include "lcd.h"
#include "main.h"
#include "processor.h"

_uartData * uartWAGO;

uint16_t parseFrameWago(void)
{
	uint16_t bytesToSend = 0;
	
	uint8_t receiverAddress = uartWAGO->bufferRx[0];
	uint8_t functionCode = uartWAGO->bufferRx[1];
	uint16_t beginAddress = uartWAGO->bufferRx[2]<<8 | uartWAGO->bufferRx[3];
	uint16_t regCount = uartWAGO->bufferRx[4]<<8 | uartWAGO->bufferRx[5];
	
	uint16_t lastAddress = beginAddress + regCount - 1;
	uint8_t resultCode = MODBUS_EXCEPTION_NONE;
	
	uartWAGO->bufferTx[0] = lcdButtonData.modbusAddress;
	uartWAGO->bufferTx[1] = functionCode;
	_lcdStateContext * lcdContext;
	
#ifdef COMPILATION_ENABLE_WAGO_COMMUNICATION_CONTROL
	if(MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress) || MODBUS_ADDRESS_DEST_TEST(receiverAddress))
	{
		lcdButtonData.modbusActive = 1;
		setTimer1Sec(TMR_1SEC_WAGO_NO_CONNECT, TMR_1SEC_WAGO_NO_CONNECT_TIMEOUT);	
	}
#endif	//COMPILATION_ENABLE_WAGO_COMMUNICATION_CONTROL
	
	if(functionCode == MODBUS_FUN_CODE_READ_HOLD_REGS)
	{
		if(!MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress) && MODBUS_ADDRESS_DEST_TEST(receiverAddress))
		{
			if(regCount == 0 || regCount > MODBUS_MAX_REG_COUNT_READ_HOLD_REGS)
			{
				resultCode = MODBUS_EXCEPTION_QUANTITY;
			}
			else if(lastAddress > MODBUS_LAST_ADDRESS)
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
			else if(beginAddress >= MODBUS_FIRST_ADDRESS && beginAddress <= MODBUS_LAST_ADDRESS)
			{
				bytesToSend = regCount*2;
				uartWAGO->bufferTx[2] = (uint8_t)bytesToSend;			//ilosc bajtow wynikowych
				uint8_t * bufferPnt = uartWAGO->bufferTx + 3;
				uint8_t * modbusDataPnt = (uint8_t *)(&lcdButtonData.modbusSpace[beginAddress]);
				for(uint16_t i=0; i<bytesToSend; i+=2)
				{
					*bufferPnt = *(modbusDataPnt+1);
					*(bufferPnt+1) = *modbusDataPnt;
					bufferPnt+=2;
					modbusDataPnt+=2;
				}
				bytesToSend = (uint16_t)bufferPnt - (uint16_t)uartWAGO->bufferTx;
				uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, bytesToSend);
 				uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
 				uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
			}
			else
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
		}		
	}
	else if(functionCode == MODBUS_FUN_CODE_READ_INPUT_REGS)
	{
		if(!MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress) && MODBUS_ADDRESS_DEST_TEST(receiverAddress))
		{
			if(beginAddress >= INPUT_REGISTER_BOOT_VER && beginAddress <= INPUT_REGISTER_SOFT_VER)
			{
				if(beginAddress + regCount - 1 > INPUT_REGISTER_SOFT_VER )
				{
					resultCode = MODBUS_EXCEPTION_QUANTITY;
				}
				else
				{
					bytesToSend = regCount*2;
					uartWAGO->bufferTx[2] = (uint8_t)bytesToSend;			//ilosc bajtow wynikowych
					uint8_t * bufferPnt = uartWAGO->bufferTx + 3;
					uint16_t inputRegData;
					for(uint8_t i=0; i<regCount; i++)
					{
						inputRegData = readInputRegister(beginAddress+i);
						*bufferPnt = (inputRegData >> 8);
						*(bufferPnt+1) = (inputRegData & 0xFF);
						bufferPnt+=2;	
					}
					bytesToSend = (uint16_t)bufferPnt - (uint16_t)uartWAGO->bufferTx;
					uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, bytesToSend);
					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
				}					
			}
			else if((beginAddress & MODBUS_SPEC_READ_FUN_REG_ADDR_MASK) == MODBUS_SPEC_READ_FUN_REG_ADDR)
			{
				if((beginAddress & MODBUS_SPEC_READ_FUN_CODE_MASK) == MODBUS_SPEC_READ_FUN_CODE_CHECK_BITMAP)
				{
					if(regCount != 1)	//mozna odczytac tylko crc 1 screenu na raz
					{
						resultCode = MODBUS_EXCEPTION_QUANTITY;
					}
					else
					{
						if(!getTimer1Sec(TMR_1SEC_HI_PRIO_DELAY))
						{
							hardwareDisableUnnecessary();
							clearLcdWorkQueue();
						}
						setTimer1Sec(TMR_1SEC_HI_PRIO_DELAY, TMR_1SEC_HI_PRIO_DELAY_TIMEOUT);
						lcdContext = getLcdWorkQueueFree();
						if(lcdContext != NULL)
						{
							lcdContext->state = LCD_STATE_IMAGE_CRC;
							lcdContext->substate = LCD_SUBSTATE_IMAGE_CRC_START;
							lcdContext->data[0] = uartWAGO->bufferRx[3];
							enqueueLcdWork();
						}
					}	
				}								
			}				
		}
	}
	else if(functionCode == MODBUS_FUN_CODE_WRITE_SINGLE_REG)
	{
		if(MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress) || MODBUS_ADDRESS_DEST_TEST(receiverAddress))
		{
			uint16_t value = regCount;
			if(value != MODE_ACT_END && value != MODE_ACT)
			{
				resultCode = MODBUS_EXCEPTION_QUANTITY;
			}
			else if(beginAddress != MODBUS_VOLATILE_ACTUALIZATION)
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
			else
			{
				if(value == MODE_ACT)
				{
					setWorkState(MODE_ACT);
				}

				uartWAGO->bufferTx[2] = (uint8_t)(2 * regCount);	//ilosc bajtow wynikowych
			
				if(!MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress))
				{
					bytesToSend = 6;
					memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, bytesToSend);
					uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, bytesToSend);
					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
				}
			}
		}		
	}
	
	else if(functionCode == MODBUS_FUN_CODE_WRITE_MULTI_REGS)
	{
		uint8_t byteCount = uartWAGO->bufferRx[6];
		uint8_t * data = uartWAGO->bufferRx + 7;
		
		if(MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress) || MODBUS_ADDRESS_DEST_TEST(receiverAddress))
		{
			if(regCount == 0 || regCount > MODBUS_MAX_REG_COUNT_WRITE_MULTI_REGS || regCount * 2 != byteCount)
			{
				resultCode = MODBUS_EXCEPTION_QUANTITY;
			}
			else if(beginAddress == MODBUS_SPEC_WRITE_FUN_REG_ADDR)
			{
				if(!getTimer1Sec(TMR_1SEC_HI_PRIO_DELAY))
				{
					hardwareDisableUnnecessary();
					clearLcdWorkQueue();
				}
				setTimer1Sec(TMR_1SEC_HI_PRIO_DELAY, TMR_1SEC_HI_PRIO_DELAY_TIMEOUT);
				
				if(data[1] == MODBUS_SPEC_WRITE_FUN_CODE_PRINT_PXLS)
				{
					lcdContext = getLcdWorkQueueFree();
					if(lcdContext != NULL)
					{
						lcdContext->state = LCD_STATE_PRINT;
						lcdContext->substate = LCD_SUBSTATE_PRINT_START;
						uint16_t pntVal = (uint16_t)(&data[2]);
						memmove(&lcdContext->data[0], &pntVal, 2);
						enqueueLcdWork();						
					}
				}					
				else if(data[1] == MODBUS_SPEC_WRITE_FUN_CODE_CLEER_SCR)
				{
					lcdContext = getLcdWorkQueueFree();
					if(lcdContext != NULL)
					{
						lcdContext->state = LCD_STATE_CLEAR;
						lcdContext->substate = LCD_SUBSTATE_CLEAR_START;
						lcdContext->data[0] = data[2];
						lcdContext->data[1] = data[3];
						enqueueLcdWork();
					}
				}
				else if(data[1] == MODBUS_SPEC_WRITE_FUN_CODE_SAVE_BITMAP)
				{
					lcdContext = getLcdWorkQueueFree();
					if(lcdContext != NULL)
					{
						lcdContext->state = LCD_STATE_IMAGE_SAVE;
						lcdContext->substate = LCD_SUBSTATE_IMAGE_SAVE_START;
						lcdContext->data[0] = data[2];
						enqueueLcdWork();
					}
				}
				else if(data[1] == MODBUS_SPEC_WRITE_FUN_CODE_SAVE_CONFIG)
				{
					while(uartLCD->isBusy()){}
					lcdSendProjectData(&data[3], data[2]);

					lcdContext = getLcdWorkQueueFree();
					if(lcdContext != NULL)
					{
						lcdContext->state = LCD_STATE_PROJECT_SAVE;
						lcdContext->substate = LCD_SUBSTATE_PROJECT_SAVE_WORKING;
						setTimerShort(TMR_SHORT_LCD_NO_RESP, TIMEOUT_LCD_NO_RESP_LONG);
						enqueueLcdWork();
					}
				}
			}
			else if(lastAddress > MODBUS_LAST_ADDRESS)
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
 			}
			else if(beginAddress >= MODBUS_FIRST_ADDRESS && beginAddress <= MODBUS_LAST_ADDRESS)
			{	
				uint16_t value;
				
				//zmiana wartosci
				for(uint16_t actualModbusAddress = beginAddress; actualModbusAddress <= lastAddress; actualModbusAddress++)
				{
					value = (((uint16_t)(*data)) << 8);
					data++;
					value |= *data;
					data++;
				
					writeModbus(actualModbusAddress, value);
				}
			
				//odpowiedz:
				if(!MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress))
				{
					bytesToSend = 6;
					memmove(uartWAGO->bufferTx, uartWAGO->bufferRx, bytesToSend);
					uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, bytesToSend);
 					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
 					uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
				}			 
			}
			else
			{
				resultCode = MODBUS_EXCEPTION_ADDRESS;
			}
		}		
	}		
	
	if(resultCode != MODBUS_EXCEPTION_NONE && !MODBUS_ADDRESS_BROADCAST_TEST(receiverAddress) && MODBUS_ADDRESS_DEST_TEST(receiverAddress))
	{
		bytesToSend = 0;
		uartWAGO->bufferTx[bytesToSend++] = lcdButtonData.modbusAddress;
		uartWAGO->bufferTx[bytesToSend++] = uartWAGO->bufferRx[1] | 0x80;
		uartWAGO->bufferTx[bytesToSend++] = resultCode;
		uint16_t crc2 = crc16Modbus(uartWAGO->bufferTx, bytesToSend);
 		uartWAGO->bufferTx[bytesToSend++] = (uint8_t)crc2;
 		uartWAGO->bufferTx[bytesToSend++] = (uint8_t)(crc2>>8);
	}
	return bytesToSend;
}

uint8_t isFrameDetectedWago()
{
	return crc16Modbus(uartWAGO->bufferRx, uartWAGO->counterRx) == 0;
}