﻿/*!
 *  @file protocolWago.h
 *	@brief Obsługa protokołu komunikacji z WAGO (Modbus)
 */ 
#ifndef PROTOCOLWAGO_H_
#define PROTOCOLWAGO_H_

#include <stdint.h>
#include "dipswitch.h"


#define MODBUS_EXCEPTION_NONE		0
#define MODBUS_EXCEPTION_FUNCTION	1
#define MODBUS_EXCEPTION_ADDRESS	2
#define MODBUS_EXCEPTION_QUANTITY	3
#define MODBUS_EXCEPTION_WRITE		4

//Adres urzadzenie (w protokole modbus) w przypadku gdy na dipswitchu ustawiono wartosc 0
#define MODBUS_ADDRESS_SERVICE		1

//Maksymalny adres urzadzenia jaki moze zostac nadany (dotyczy protokołu modbus)
#define MODBUS_ADDRESS_MAX_VALUE	255

//Test na to czy dana ramka jest zaadresowana na adres rozgloszeniowy
#define MODBUS_ADDRESS_BROADCAST_TEST(address)	(address == 0)

//Test na to czy dana ramka jest zaadresowana do tego urzadzenia
#define MODBUS_ADDRESS_DEST_TEST(address)		(address == lcdButtonData.modbusAddress)
									
//Test na to czy mozna modyfikowac programowy adres urzadzenia
#define MODBUS_ADDRESS_MODIF_TEST	(lcdButtonData.dipswitchValue == DIPSWITCH_SERVICE_MODE || lcdButtonData.dipswitchValue == DIPSWITCH_SOFT_MODE)


//Kod funkcji Read holding registers
#define MODBUS_FUN_CODE_READ_HOLD_REGS						0x03
//Kod funkcji Read input registers
#define MODBUS_FUN_CODE_READ_INPUT_REGS						0x04
//Kod funkcji Write single register
#define MODBUS_FUN_CODE_WRITE_SINGLE_REG					0x06
//Kod funkcji Write multiple registers
#define MODBUS_FUN_CODE_WRITE_MULTI_REGS					0x10

//Maksymalna ilosc rejestrow dla funkcji Read holding registers
#define MODBUS_MAX_REG_COUNT_READ_HOLD_REGS					125
//Maksymalna ilosc rejestrow dla funkcji Write multiply registers
#define MODBUS_MAX_REG_COUNT_WRITE_MULTI_REGS				123

//Adres rejestru pozwalajacy uruchomic specjalna funkcjonalnosc: zapisywanie grafik, konfiguracji
#define MODBUS_SPEC_WRITE_FUN_REG_ADDR						0xFF00
//specjalna funkcjonalnosc: wyswietlenie pikseli na wyswietlaczu
#define MODBUS_SPEC_WRITE_FUN_CODE_PRINT_PXLS				0x0A
//specjalna funkcjonalnosc: wyczyszczenie ekranu wskazanym kolorem
#define MODBUS_SPEC_WRITE_FUN_CODE_CLEER_SCR				0x0B
//specjalna funkcjonalnosc: zapisanie aktualnego ekranu pod wskazany slot
#define MODBUS_SPEC_WRITE_FUN_CODE_SAVE_BITMAP				0x0C
//specjalna funkcjonalnosc: zapis komunikatu konfiguracji
#define MODBUS_SPEC_WRITE_FUN_CODE_SAVE_CONFIG				0x0E

//Adres rejestru pozwalajacy uruchomic specjalna funkcjonalnosc: odczyt sumy CRC wyswietlanego obrazu
#define MODBUS_SPEC_READ_FUN_REG_ADDR						0xF000
//Maska do wyciagniecia adresu rejestru uruchamiajacego specjalna funkcjonalnosc
#define MODBUS_SPEC_READ_FUN_REG_ADDR_MASK					0xF000
//specjalna funkcjonalnosc: odczytanie sumy kontrolnej (CRC) aktualnie wyswietlanego obrazu
#define MODBUS_SPEC_READ_FUN_CODE_CHECK_BITMAP				0x0D00
//Maska do wyciagniecia kodu funkcji
#define MODBUS_SPEC_READ_FUN_CODE_MASK						0x0F00



//MODBUS_READ_ONLY_BIT_DATA_1:
//flagi sygnalizujące stan konfigurowalnych linii DIO:
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_STATE_DIO1		0
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_STATE_DIO2		1
//flagi sygnalizujące wykrycie zbocza rosnacego od ostatniego odczytu na linii DIO:
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_RISING_DIO1		2
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_RISING_DIO2		3
//flagi sygnalizujące wykrycie zbocza opadajacego od ostatniego odczytu na linii DIO:
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_FALLING_DIO1	4
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_FALLING_DIO2	5
//flagi sygnalizujące blad na linii DIO (niezgodnosc wyjscia z faktycznym wejsciem):
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_ERROR_DIO1		6
#define BIT_MODBUS_READ_ONLY_BIT_DATA_1_ERROR_DIO2		7

//MODBUS_READ_ONLY_BIT_DATA_2:
//flagi sygnalizujące że pod daną pozycją zapisany jest jakiś kod ROM czujnika:
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_ROM		0
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_1_ROM		1
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_2_ROM		2
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_3_ROM		3
//flagi sygnalizujące że kolejny czujnik dziala poprawnie (czujnik odpowiada):
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_0_OK		4
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_1_OK		5
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_2_OK		6
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DS_3_OK		7
//flagi sygnalizujące stan wejsc binarnych:
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_0			8
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_1			9
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_2			10
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_DI_3			11
//flagi sygnalizujące inne stany wewnetrzne:
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_ALARM			12			/*!< 0-alarm nieaktywny (w danej chwili); 1-alarm aktywny (w danej chwili). Alarm polega na mruganiu podświetleniem i dźwiękiem, może być aktywowany przez modbus lub na skutek właściwej godziny oraz deaktywowany przez modbus lub kliknięcie ekranu */
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_OBECNOSC		13			/*!< stan linii dla czujnika obecnosci */
#define BIT_MODBUS_READ_ONLY_BIT_DATA_2_URZADZENIE		14			/*!< stan urzadzenia (nieaktywne/aktywne) */

//MODBUS_VOLATILE_CONTROL:
//flagi wybierające czujniki do realizacji zadania:
#define BIT_MODBUS_VOLATILE_DS_0_CONTROL		0
#define BIT_MODBUS_VOLATILE_DS_1_CONTROL		1
#define BIT_MODBUS_VOLATILE_DS_2_CONTROL		2
#define BIT_MODBUS_VOLATILE_DS_3_CONTROL		3
//rodzaj wybranego zadania dla wybranych czujników:
#define BIT_MODBUS_VOLATILE_DS_CONTROL_DELETE	4
#define BIT_MODBUS_VOLATILE_DS_CONTROL_SWAP		5
//kontrola alarmu
#define BIT_MODBUS_VOLATILE_ALARM_ACTIVE		8			/*!< 1-natychmiastowe aktywowanie alarmu*/
#define BIT_MODBUS_VOLATILE_ALARM_DEACTIVE		9			/*!< 1-deaktywowanie alarmu jeśli jest włączony w danej chwili*/

//MODBUS_SYSTEM_BIT_DATA:
//kontrola alarmu
#define BIT_MODBUS_SYSTEM_BIT_DATA_ALARM_SET	0		/*!< 0-alarm nie ustawiony; 1-alarm ustawiony */
//flagi sygnalizujące PWM na wyjsciach konfigurowalnych:
#define BIT_MODBUS_SYSTEM_BIT_PWM_DIO_1			6
#define BIT_MODBUS_SYSTEM_BIT_PWM_DIO_2			7
//flagi sygnalizujące stan wyjsc binarnych:
#define BIT_MODBUS_SYSTEM_BIT_DATA_DO_0			8
#define BIT_MODBUS_SYSTEM_BIT_DATA_DO_1			9
#define BIT_MODBUS_SYSTEM_BIT_DATA_DO_2			10
#define BIT_MODBUS_SYSTEM_BIT_DATA_DO_3			11
//flagi sygnalizujące stan wyjsc linii DIO:
#define BIT_MODBUS_SYSTEM_BIT_DATA_DIO_1		12
#define BIT_MODBUS_SYSTEM_BIT_DATA_DIO_2		13
//flagi sygnalizujące konfiguracje linii DIO:
#define BIT_MODBUS_SYSTEM_BIT_CONFIG_DIO_1		14
#define BIT_MODBUS_SYSTEM_BIT_CONFIG_DIO_2		15

//MODBUS_READ_ONLY_EXTRA_BIT_DATA:
//stan linii DIO:
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO1		0
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_STATE_DIO2		1
//stan obecnosci:
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_OBECNOSC		2
//stan linii DI:
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_0			3
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_1			4
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_2			5
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DI_3			6
//stan linii DO:
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_0			7
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_1			8
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_2			9
#define MODBUS_READ_ONLY_EXTRA_BIT_DATA_DO_3			10


/*!
	@brief Przestrzen adresowa modbus dla funkcji odczytu input registers (4)
*/
enum
{
	INPUT_REGISTER_BOOT_VER = 0x0D,				/** < wersja bootloadera */
	INPUT_REGISTER_SOFT_VER,					/** < wersja oprogramowania */
};	

/*!
	@brief Przestrzen adresowa modbus dla funkcji odczytu zapisu wielu danych (3,16)
*/
enum{
	MODBUS_FIRST_ADDRESS = 0,										/*! < Adres pierwszego rejestru, jaki jest dostepny po protokole modbus */
	//tylko do odczytu:
	MODBUS_READ_ONLY_RESERVED = MODBUS_FIRST_ADDRESS,				/**< rejestr zarezerwowany */
	MODBUS_READ_ONLY_BIT_DATA_1,									/** < zbiór danych jednobitowych tylko do odczytu*/
	MODBUS_READ_ONLY_BIT_DATA_2,									/** < zbiór danych jednobitowych tylko do odczytu*/
	MODBUS_READ_ONLY_CLOCK_ALARM_H,									/** < jeśli budzik jest aktywny: godzina alarmu; jeśli budzik nie jest aktywny: 0xFFFF */
	MODBUS_READ_ONLY_CLOCK_ALARM_M,									/** < jeśli budzik jest aktywny: minuta alarmu; jeśli budzik nie jest aktywny: 0xFFFF */	
	MODBUS_READ_ONLY_TEMP_0,										/** < temperatura z pierwszego czujnika */
	MODBUS_READ_ONLY_TEMP_1,										/** < temperatura z drugiego czujnika */
	MODBUS_READ_ONLY_TEMP_2,										/** < temperatura z trzeciego czujnika */
	MODBUS_READ_ONLY_TEMP_3,										/** < temperatura z czwartego czujnika */
	MODBUS_READ_ONLY_LIGHT,											/** < poziom nasłonecznienia (luksy) */
	MODBUS_READ_ONLY_LIGHT_VOLT,							//10	/** < poziom nasłonecznienia (napiecie w skali 0-1023) */
	MODBUS_READ_ONLY_PASS_DIGITS,									/** < Ilość dotychczas wprowadzonych znaków w hasle, max. 4; wartość zerowana klawiszem PassClearButton oraz podczas ładowania nowej sceny*/
	MODBUS_READ_ONLY_PASSWORD,										/** < Wprowadzone haslo, wartosc zapisywana po wcisnieciu klawisza PassSetButton; zerowana podczas ładowania nowej sceny */	
	MODBUS_READ_ONLY_END,									//13
	//input/output (not eeprom save):
	MODBUS_VOLATILE_BEGIN = MODBUS_READ_ONLY_END,			//13
	MODBUS_VOLATILE_ACTUALIZATION = MODBUS_VOLATILE_BEGIN,	//13	/** < status aktualizatora*/
	MODBUS_VOLATILE_ACTIVE_SCENE,									/** < Aktualnie wybrana scena */
	MODBUS_VOLATILE_CONTROL,										/** < rejestr sterujący dla czujników DS (wykasaowanie numerow, odwrocenie pozycji) i alarmu*/
	MODBUS_VOLATILE_SAVE_LIGHT_SCENE,								/** < rejestr służący do zapisywania aktualnych wartości rejestrów LisghtSceneRegister pod wybraną scenę świetlną */
	MODBUS_VOLATILE_DATE_YAER,										/** < data, rok */
	MODBUS_VOLATILE_DATE_MONTH,										/** < data, miesiąc <0;11>*/
	MODBUS_VOLATILE_DATE_DAY,										/** < data, dzien */
	MODBUS_VOLATILE_CLOCK_0_H,								//20	/** < zegar pierwszy, godziny */
	MODBUS_VOLATILE_CLOCK_0_M,										/** < zegar pierwszy, minuty */
	MODBUS_VOLATILE_CLOCK_1_H,										/** < zegar drugi, godziny */
	MODBUS_VOLATILE_CLOCK_1_M,										/** < zegar drugi, minuty */
	MODBUS_VOLATILE_CLOCK_2_H,										/** < zegar trzeci, godziny */
	MODBUS_VOLATILE_CLOCK_2_M,										/** < zegar trzeci, minuty */
	MODBUS_VOLATILE_CLOCK_3_H,										/** < zegar czwarty, godziny */
	MODBUS_VOLATILE_CLOCK_3_M,										/** < zegar czwarty, minuty */
	MODBUS_VOLATILE_CLOCK_4_H,										/** < zegar piąty, godziny */
	MODBUS_VOLATILE_CLOCK_4_M,										/** < zegar piąty, minuty */
	MODBUS_VOLATILE_END,									//30
	//input/output (eeprom save):
	MODBUS_SYSTEM_BEGIN = MODBUS_VOLATILE_END,				//30
	MODBUS_SYSTEM_CLOCK_ALARM_H_SET = MODBUS_SYSTEM_BEGIN,	//30	/** < ustawianie budzika: godziny */
	MODBUS_SYSTEM_CLOCK_ALARM_M_SET,								/** < ustawianie budzika: minuty */	
	MODBUS_SYSTEM_BACKLIGHT_ON,										/** < intensywność podświetlenia wyświetlacza <0%;100%> podczas trybu aktywnego; 0xFFFF -> zależne od nasłonecznienia */
	MODBUS_SYSTEM_BACKLIGHT_OFF, 									/** < intensywność podświetlenia wyświetlacza <0%;100%> podczas stanu wygaszonego */
	MODBUS_SYSTEM_DARKER_TIME,										/** < czas po jakim następuje wygaszenie wyświetlacza; 0x0000 -> opcja wyłączona; *100ms */
	MODBUS_SYSTEM_BUZZER_TIME,										/** < dlugość trwania dzwięku po kliknięciu w jakichs przycisk */
	MODBUS_SYSTEM_DEFAULT_SCENE,									/** < scena domyślna, uruchamiana po odzyskaniu zasilania*/	
	MODBUS_SYSTEM_LANG_USER,										/** < Wybrany język */
	MODBUS_SYSTEM_BIT_DATA,											/** < zbiór danych jednobitowych do zapisu*/
	MODBUS_SYSTEM_UART_SPEED,										/** < predkosc komunikacji*/
	MODBUS_SYSTEM_MODBUS_SOFT_ADDR,							//40	/** < programowy adres urządzenia*/
	MODBUS_SYSTEM_SHOW_LIGHT_SCENE,									/** < rejestr wyboru sceny (LSB-scena 0; MSB-scena F*/
	MODBUS_SYSTEM_LIGHT_SCENE_REG_FIRST,
	MODBUS_SYSTEM_LIGHT_SCENE_REG_0 = MODBUS_SYSTEM_LIGHT_SCENE_REG_FIRST,	/*! < rejestr nr 0 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_1,								/*! < rejestr nr 1 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_2,								/*! < rejestr nr 2 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_3,								/*! < rejestr nr 3 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_4,								/*! < rejestr nr 4 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_5,								/*! < rejestr nr 5 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_6,								/*! < rejestr nr 6 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_7,								/*! < rejestr nr 7 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_8,						//50	/*! < rejestr nr 8 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_9,								/*! < rejestr nr 9 sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_A,								/*! < rejestr nr A sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_B,								/*! < rejestr nr B sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_C,								/*! < rejestr nr C sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_D,								/*! < rejestr nr D sceny świetlnej */
	MODBUS_SYSTEM_LIGHT_SCENE_REG_LAST = MODBUS_SYSTEM_LIGHT_SCENE_REG_D,
	MODBUS_SYSTEM_PWM_0_VALUE,										/*! < wartosc PWM dla kanalu 1 */
	MODBUS_SYSTEM_PWM_1_VALUE,										/*! < wartosc PWM dla kanalu 2 */
	MODBUS_READ_ONLY_EXTRA_BIT_DATA,								/*! < specjalny zbiór danych jednobitowych tylko do odczytu */
	MODBUS_SYSTEM_LAST = MODBUS_READ_ONLY_EXTRA_BIT_DATA,			/*! < Koniec przestrzeni systemowej  */
	MODBUS_USER_FIRST,										//59	/*! < poczatek przestrzeni uzytkownika */
	//...
	MODBUS_LAST_ADDRESS = 255,										/*!< Adres ostatniego rejestru, jaki jest dostepny po protokole modbus */
	MODBUS_SIZE = 256											/*!< Koniec przestrzeni modbus (ostatni element ma adres 255) */
};


#define MODBUS_DEF_VAL_CLOCK_ALARM_INACTIVE		0xFFFF			/*!< wartosc ta dla rejestrow @ref MODBUS_READ_ONLY_CLOCK_ALARM_H i @ref MODBUS_READ_ONLY_CLOCK_ALARM_M oznacza ze budzik nie jest ustawiony */
#define MODBUS_DEF_VAL_ACTUALIZATION_STATE		0xFF00			/*!< wartosc oznacza ze program znajduje sie w normalnym trybie */


/*!
	@brief Analiza ramki zgodnie z protokołem WAGO (Modbus)
	
	Metoda analizuje otrzymaną ramkę i w razie potrzeby przygotowuje odpowiedź.
	@return Ilość bajtów jaka jest przygotowana do odesłania. W przypadku gdy nie potrzebna jest odpowiedź,
		zwracana jest wartość 0.
*/
uint16_t parseFrameWago(void);

/*!
	@brief Wykrywanie poprawnej ramki zgodnej z protokołem WAGO (Modbus)
	
	Metoda analizuje czy we wskazanej strukturze @ref _uartData znajduje się poprawnie odczytana ramka. Pierwszy bajt ramki musi znajdować się
	w pierwszym bajcie bufora danych, w związku z tym niezbędne jest wykrywanie braku
	komunikacji i zerowanie wskażników danych.
	@return 0 - nie wykryto ramki; @n 1 - wykryto ramkę.
*/
uint8_t isFrameDetectedWago();


#endif /* PROTOCOLWAGO_H_ */