﻿#include <avr/io.h>
#include <stdint.h>

#include "uart.h"
#include "lcdButtonTimers.h"
#include "processor.h"
#include "protocolWago.h"
#include "protocolLcd.h"
#include "crc.h"
#include "lcd.h"

extern _lcdButtonData lcdButtonData;

/**************************************************************************
zmienne publiczne:
**************************************************************************/
_uartData uart0;	/*!< Dane odnoszace sie do portu UART0 */
_uartData uart1;	/*!< Dane odnoszace sie do portu UART1 */

void uartService(void)
{
	uint16_t transmitLength;
	//wago:
	if(uartWAGO->readyToParse)
	{
		uartWAGO->stopReceiving();
		transmitLength = uartWAGO->parseFrame();
		if(transmitLength)
		{
			uartWAGO->startTransmit(transmitLength);
		}
		else
		{
			uartWAGO->startReceiving();
		}
		uartWAGO->readyToParse = 0;
	}
	
	//uartLCD:	
	if(uartLCD->isReceiveFinished())
	{
		uartLCD->stopReceiving();
		transmitLength = 0;
		if(uartLCD->isFrameDetected())
			transmitLength = uartLCD->parseFrame();
			
		if(transmitLength)
			uartLCD->startTransmit(transmitLength);
		else
			uartLCD->startReceiving();
	}
}


void uartClear(_uartData * data)
{
	data->counterRx = 0;
	data->counterTx = 0;
	data->currentId = 0;
}

void startReceivingUart0(void);
void startReceivingUart1(void);
void stopReceivingUart0(void);
void stopReceivingUart1(void);
void startTransmitUart0(int16_t frameSize);
void startTransmitUart1(int16_t frameSize);
uint8_t isReceiveFinishedUart0(void);
uint8_t isReceiveFinishedUart1(void);
uint8_t isBusyUart0(void);
uint8_t isBusyUart1(void);
uint8_t isTransmitUart0(void);
uint8_t isTransmitUart1(void);
void setBaudRateUartx(_baudRate baudRate, _uartData * data);
void setBaudRateUart0(_baudRate baudRate);
void setBaudRateUart1(_baudRate baudRate);

void initUart(void)
{
	//uart0 (WAGO):
	//przypisanie odwolan do wlasciwych funkcji:
	uart0.startReceiving = startReceivingUart0;
	uart0.stopReceiving = stopReceivingUart0;
	uart0.startTransmit = startTransmitUart0;
	uart0.setBaudRate = setBaudRateUart0;
	uart0.isReceiveFinished = isReceiveFinishedUart0;
	uart0.isBusy = isBusyUart0;
	uart0.isTransmit = isTransmitUart0;
	uart0.isFrameDetected = isFrameDetectedWago;
	uart0.parseFrame = parseFrameWago;
	//ustawienie formatu ramek: (8bit, 1stop)
	UCSR0C = (3<<UCSZ00);
	
	//ustawienie predkosci na porcie oraz wlaczenie odbiornika i nadajnika:
	if(lcdButtonData.dipswitchValue == DIPSWITCH_SERVICE_MODE)
	{
		lcdButtonData.newModbusBaudRate = BAUDRATE_DEFAULT_UART0;
		lcdButtonData.newModbusAddress = MODBUS_ADDRESS_SERVICE;
	}
	else
	{		
		if(lcdButtonData.dipswitchValue == DIPSWITCH_SOFT_MODE)
			lcdButtonData.newModbusAddress = readModbus(MODBUS_SYSTEM_MODBUS_SOFT_ADDR);
		else
			lcdButtonData.newModbusAddress = lcdButtonData.dipswitchValue;
	}
	lcdButtonData.actualModbusBaudRate = lcdButtonData.newModbusBaudRate;
	lcdButtonData.modbusAddress = lcdButtonData.newModbusAddress;
	uart0.setBaudRate(lcdButtonData.actualModbusBaudRate);
	//rozpoczecie pracy:
	uart0.startReceiving();
	uartWAGO = &uart0;
	
	//uart1 (LCD):
	//przypisanie odwolan do wlasciwych funkcji:
	uart1.startReceiving = startReceivingUart1;
	uart1.stopReceiving = stopReceivingUart1;
	uart1.startTransmit = startTransmitUart1;
	uart1.setBaudRate = setBaudRateUart1;
	uart1.isReceiveFinished = isReceiveFinishedUart1;
	uart1.isBusy = isBusyUart1;
	uart1.isTransmit = isTransmitUart1;
	uart1.isFrameDetected = isFrameDetectedLcd;
	uart1.parseFrame = parseFrameLcd;
	//ustawienie formatu ramek: (8bit, 1stop)
	UCSR1C = (3<<UCSZ10);
	//ustawienie predkosci na portach oraz wlaczenie odbiornika i nadajnika:
	uart1.setBaudRate(BAUDRATE_DEFAULT_UART1);		//ustawic na najnizsza predkosc bo przy braku odpowiedzi przestawiana jest predkosc na wyzsza
	//rozpoczecie pracy:
	uart1.startReceiving();
	uartLCD = &uart1;
	
}

void uartTurnOff(void)
{
	UCSR0B = 0;
	UCSR1B = 0;
}
/**************************************************************************
uart0:
**************************************************************************/
void startReceivingUart0(void)
{
	UART0_RE_PORT_REG &= ~_BV(UART0_RE_LINE);
	UART0_DE_PORT_REG &= ~_BV(UART0_DE_LINE);
	UCSR0B &= ~((1<<TXCIE0) | (1<<UDRIE0));
	uartClear(&uart0);
	UCSR0B |= (1<<RXCIE0);
}
void stopReceivingUart0(void)
{
	UART0_RE_PORT_REG |= _BV(UART0_RE_LINE);
	UART0_DE_PORT_REG |= _BV(UART0_DE_LINE);
	UCSR0B &= ~(1<<RXCIE0);
}
void startTransmitUart0(int16_t frameSize)
{
	uint16_t waitTimeUs;
	UART0_RE_PORT_REG |= _BV(UART0_RE_LINE);
	UART0_DE_PORT_REG |= _BV(UART0_DE_LINE);
	UCSR0B &= ~(1<<RXCIE0);
	uartClear(&uart0);
	uart0.counterTx = frameSize;	
	//zwloka okreslana jest jako czas [us] wysylania UART_RESPONSE_CHAR_DELAY znakow z aktualna predkoscia
	//Przy obliczaniu czasu wystepuje dzielenie przez 10, poniewaz wartosc predkosci pomniejszona jest juz o 10
	switch(lcdButtonData.actualModbusBaudRate)
	{
		case BAUDRATE_1200:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_1200);
			break;
		case BAUDRATE_2400:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_2400);
			break;
		case BAUDRATE_4800:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_4800);
			break;
		case BAUDRATE_9600:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_9600);
			break;
		case BAUDRATE_19200:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_19200);
			break;
		case BAUDRATE_38400:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_38400);
			break;
		case BAUDRATE_57600:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_57600);
			break;
		case BAUDRATE_115200:
		default:
			waitTimeUs = UART_RESPONSE_CHAR_DELAY * 1000000 / (BAUDRATE_115200);
			break;
	}
	wait_us(waitTimeUs);
	UCSR0B |= (1<<TXCIE0) | (1<<UDRIE0);
}
uint8_t isReceiveFinishedUart0(void)
{
	return !getTimerShort(TMR_SHORT_UART0) && uart0.counterRx>0;
}
uint8_t isBusyUart0(void)
{
	if(UCSR0B & (1<<TXCIE0))
		return 1;
	if(uart0.counterRx > 0)
		return 1;
	return 0;
}

uint8_t isTransmitUart0(void)
{
	if(UCSR0B & (1<<TXCIE0))
		return 1;
	return 0;
}

/**************************************************************************
uart1:
**************************************************************************/
void startReceivingUart1(void)
{
	UCSR1B &= ~((1<<TXCIE1) | (1<<UDRIE1));
	uartClear(&uart1);
	UCSR1B |= (1<<RXCIE1);
}
void stopReceivingUart1(void)
{
	UCSR1B &= ~(1<<RXCIE1);
}


void startTransmitUart1(int16_t frameSize)
{
	UCSR1B &= ~(1<<RXCIE1);
	uartClear(&uart1);
	uart1.counterTx = frameSize;
	UCSR1B |= (1<<TXCIE1) | (1<<UDRIE1);
}


uint8_t isReceiveFinishedUart1(void)
{
	return !getTimerShort(TMR_SHORT_UART1) && uart1.counterRx>0;
}
uint8_t isBusyUart1(void)
{
#ifdef _DEBUG
	//if(!(LCD_NOT_BUSY_IN))	//swat!
	//	return 0;
#endif // _DEBUG
	if(UCSR1B & (1<<TXCIE1))
		return 1;
	if(uart1.counterRx > 0)
		return 1;
	return 0;
}
uint8_t isTransmitUart1(void)
{
	if(UCSR1B & (1<<TXCIE1))
		return 1;
	return 0;
}

/*!
	@brief 
		Funkcja przekształca predkość transmisji na wartość konfiguracyjna odpowiadająca
		tej predkości.
	@param baudRate
		Prędkość komunikacji (podzielona przez 10) dla jakiej należy uzyskać wartość
		konfiguracyjną
	@return
		Wartość konfiguracyjna jaką należy ustawić dla portu UART żeby uzyskać żądaną
		prędkość komunikacji lub 0xFFFF w przypadku prędkości niedozwolonej
*/
static uint16_t baudRateToUbrr(_baudRate baudRate)
{
	uint16_t ubrr = BAUDRATE_NONE_UBRR;
	
	if(baudRate == BAUDRATE_1200)
		ubrr = BAUDRATE_1200_UBRR;
	else if(baudRate == BAUDRATE_2400)
		ubrr = BAUDRATE_2400_UBRR;
	else if(baudRate == BAUDRATE_4800)
		ubrr = BAUDRATE_4800_UBRR;
	else if(baudRate == BAUDRATE_9600)
		ubrr = BAUDRATE_9600_UBRR;
	else if(baudRate == BAUDRATE_19200)
		ubrr = BAUDRATE_19200_UBRR;
	else if(baudRate == BAUDRATE_38400)
		ubrr = BAUDRATE_38400_UBRR;
	else if(baudRate == BAUDRATE_57600)
		ubrr = BAUDRATE_57600_UBRR;
	else if(baudRate == BAUDRATE_115200)
		ubrr = BAUDRATE_115200_UBRR;
	
	return ubrr;
}

uint8_t ifBaudRateCorrect(uint16_t baudRate)
{	
	uint16_t ubrr = baudRateToUbrr(baudRate);
	if(ubrr == BAUDRATE_NONE_UBRR)
	{
		return 0;
	}
	return 1;
}
/*!
	@brief
		Funkcja ustawia prędkość komunikacji dla wskazanego portu.
	@details
		Jeśli nowa prędkość komunikacji jest inna niż poprzednia, następuje wyczyszczenie
		bufora. W przypadku niewłaściwej prędkości komunikacji funkcja ta nic nie robi,
	@param baudRate
		Nowa prędkość jaka ma być ustawiona
	@param uart
		Struktura danych związana z portem, dla którego ma być zmieniona prędkość
*/
void setBaudRateUartx(_baudRate baudRate, _uartData * uart)
{
	uint16_t newConfig = baudRateToUbrr(baudRate);	
	if(newConfig != BAUDRATE_NONE_UBRR)
	{		
		uart->baudRate = baudRate;
		if(uart == &uart0)
		{
			UCSR0B &= ~((1<<RXEN0) | (1<<TXEN0));
			uartClear(uart);
			UBRR0 = newConfig;
			UCSR0B = (1<<RXEN0) | (1<<TXEN0);
		}
		else if(uart == &uart1)
		{
			UCSR1B &= ~((1<<RXEN1) | (1<<TXEN1));
			uartClear(uart);
			UBRR1 = newConfig;
			UCSR1B = (1<<RXEN1) | (1<<TXEN1);
		}		
	}
}
void setBaudRateUart0(_baudRate baudRate)
{
	setBaudRateUartx(baudRate, &uart0);
}
	
void setBaudRateUart1(_baudRate baudRate)
{
	setBaudRateUartx(baudRate, &uart1);
}
uint8_t isFrameDetectedUart1(void)
{
	return isFrameDetectedLcd(&uart1);
}

/*!
	@brief
		Funkcja zwraca kolejna predkosc komunikacji
	@param 
		Aktualna prędkość komunikacji
	@return 
		Następna (większa) prędkość komunikacji
*/
_baudRate uartGetNextBaudRate(_baudRate baudRate)
{
	_baudRate nextBaudRate;
	if(baudRate == BAUDRATE_1200)
		nextBaudRate = BAUDRATE_2400;
	else if(baudRate == BAUDRATE_2400)
		nextBaudRate = BAUDRATE_4800;
	else if(baudRate == BAUDRATE_4800)
		nextBaudRate = BAUDRATE_9600;
	else if(baudRate == BAUDRATE_9600)
		nextBaudRate = BAUDRATE_19200;
	else if(baudRate == BAUDRATE_19200)
		nextBaudRate = BAUDRATE_38400;
	else if(baudRate == BAUDRATE_38400)
		nextBaudRate = BAUDRATE_57600;
	else if(baudRate == BAUDRATE_57600)
		nextBaudRate = BAUDRATE_115200;
	else
		nextBaudRate = BAUDRATE_NONE;	//if(baudRate == BAUDRATE_115200)
	return nextBaudRate;
}
