﻿/*!
 *  @file uart.h
 *	@brief Obsługa komunikacji szeregowej UART
 */ 

#ifndef UART_H_
#define UART_H_

#include <stdint.h>
#include <avr/io.h>
#include "lcdButtonTimers.h"
#include "crc.h"

extern uint8_t timersShort[TMR_SHORT_ALL];

/*! 
	@def UART_BUFFER_SIZE
    @brief Wielkość buforów na dane dla UART-ów.
*/
#define UART_BUFFER_SIZE						300

/*!
	@def UART_RESPONSE_CHAR_DELAY
	@breif Zwloka wyrazona w ilosciach znakow, przed rozpoczeciem nadawania na uarcie
*/
#define UART_RESPONSE_CHAR_DELAY				3.5
/*!
	@enum _baudRate
	@brief Prędkość z jaką odbywa się komunikacja
*/
typedef enum 
{
	BAUDRATE_1200			= 120,				/*!< Prędkość komunikacji 1200bps */
	BAUDRATE_2400			= 240,				/*!< Prędkość komunikacji 2400bps */
	BAUDRATE_4800			= 480,				/*!< Prędkość komunikacji 4800bps */
	BAUDRATE_9600			= 960,				/*!< Prędkość komunikacji 9600bps */
	BAUDRATE_19200			= 1920,				/*!< Prędkość komunikacji 19200bps */
	BAUDRATE_38400			= 3840,				/*!< Prędkość komunikacji 38400bps */
	BAUDRATE_57600			= 5760,				/*!< Prędkość komunikacji 57600bps */
	BAUDRATE_115200			= 11520,			/*!< Prędkość komunikacji 115200bps */
	BAUDRATE_DEFAULT_UART0	= BAUDRATE_9600,	/*!< Domyślna prędkość komunikacji dla portu UART0 */
	BAUDRATE_DEFAULT_UART1	= BAUDRATE_1200,	/*!< Domyślna prędkość komunikacji dla portu UART1 */
	BAUDRATE_NONE			= 0xFFFF,
}_baudRate;

/*!
	@struct _uartData
	@brief Dane dla UART-u.
	
	Struktura przechowująca dane dla komunikacji UART.
*/
typedef struct{
	
	_baudRate baudRate;							/*!< Prędkość komunikacji ustawiona na danym porcie */
	uint8_t bufferRx[UART_BUFFER_SIZE];			/*!< Bufor odbieranego komunikatu */
	uint8_t bufferTx[UART_BUFFER_SIZE];			/*!< Bufor wysylanego komunikatu */
	int16_t counterRx;							/*!< Ilość otrzymanych do tej pory bajtów */
	int16_t counterTx;							/*!< Ilość bajtów jakie należy wysłać */
	int16_t currentId;							/*!< indeks bajtu który ma być wysłany */
	uint16_t crc;								/*!< Suma crc odebranych danych */
	uint8_t readyToParse;						/*!< Flaga wskazujaca ze odebrano cala ramke ktora jest gotowa do obsluzenia */
	
	/*!
		@brief Włączenie odbióru danych
		
		Podczas gdy metoda zostanie włączona podaczas wysyłania komunikatu, nastąpi
		przerwanie wysyłania komunikatu i przełączenie w tryb odbierania.
	*/
	void (*startReceiving)(void);
	
	/*!
		@brief Wyłączenie odbióru danych
		
		Metoda wyłącza odbiornik w celu umożliwienia przygotowania danych do wysłania
		(zapisu do bufora).
	*/
	void (*stopReceiving)(void);
	
	/*!
		@brief Rozpoczęcie wysyłania danych
		
		Metoda wyłącza odbiornik (jeśli nie był wcześniej wyłączony), aktywuje nadajnik
		i rozpoczyna wysyłanie danych.
		@param frameSize Ilość bajtów wysyłanego komunikatu
	*/
	void (*startTransmit)(int16_t frameSize);
	
	/*!
		@brief Wykrywanie odebrania danych
		
		Metoda sprawdza czy dany port zakończył odbieranie danych.
		@return 0 - brak nowych danych; @n 1 - odebrano nowe dane.
	*/
	uint8_t (*isReceiveFinished)(void);
	
	/*!
		@brief Sprawdzenie czy port jest wolny
		
		Metoda sprawdza czy na danym porcie trwa właśnie jakakolwiek komunikacja lub czy w buforze  znajdują się dane wymagające obsłużenia.
		@return 0 - port jest wolny (nie trwa komunikacja ani nie ma nieobłużonych danych);
			@n 1 - port jest zajęty (trwa komunikacja lub są nieobłużone dane);
	*/
	uint8_t (*isBusy)(void);
	
		/*!
		@brief Ustawienie prędkości transmisji
		
		Metoda zmienia prędkość komunikacji danego portu i zeruje stan odebranych/wysłanych danych.
		@param baudRate Prędkość transmisji jaką należy ustawić
	*/
	void (*setBaudRate)(_baudRate baudRate);
	
	/*!
		@brief Okresla czy trwa obecnie wysylanie danych
	*/
	uint8_t (*isTransmit)(void);
	
	/*!
		@brief Wykrywanie poprawnej ramki
		
		Metoda analizuje czy w buforze znajduje się poprawnie odczytana ramka. Pierwszy bajt ramki musi znajdować się
		w pierwszym bajcie bufora danych, w związku z tym niezbędne jest wykrywanie braku
		komunikacji i zerowanie wskażników danych.
		@return 0 - nie wykryto ramki; @n 1 - wykryto ramkę.
	*/
	uint8_t (*isFrameDetected)(void);
	
	/*!
		@brief Analiza ramki
		
		Metoda analizuje otrzymaną ramkę i w razie potrzeby przygotowuje odpowiedź.
		@return Ilość bajtów jaka jest przygotowana do odesłania. W przypadku gdy nie potrzebna jest odpowiedź,
			zwracana jest wartość 0.
	*/
	uint16_t (*parseFrame)(void);
	
}_uartData;

/*!
	@brief Wykrywanie oraz obsługa komunikatów na obu portach
	
	@b UWAGA Funkcja ta musi być wywoływana możliwie jak najczęściej (najlepiej w przebiegu pętli głównej progrmu).
*/
void uartService(void);

/*!
    @brief Inizjalizacja dwóch portów UART
*/
void initUart(void);

/*!
	@brief Wyłączenie obu portów uart
*/
void uartTurnOff(void);

/*!
    @brief Wyczyszczenie struktury.
	@param data Wskaźnik do struktury danych
*/
void uartClear(_uartData * data);

/*!
	@brief
		Funkcja sprawdza czy dostępna jest dana prędkość dla portów UART
	@param baudRate
		Prędkość do sprawdzenia
	@return 
		0 - podana prędkość nie jest dostępna dla portów UART
		1 - podana prędkość jest dostępna
*/
uint8_t ifBaudRateCorrect(uint16_t baudRate);

_baudRate uartGetNextBaudRate(_baudRate baudRate);

/// Struktura danych dla komunikacji z WAGO
extern _uartData uart0;
/// Struktura danych dla komunikacji z wyświetlaczem LCD
extern _uartData uart1;
//uart0:
static inline void uart0RxVectImpl(void)
{
	uint8_t tmp = UDR0;
	
	//jesli nie zakonczono obrobki poprzedniego komunikatu, to nie analizuj tego bajtu:
	if(uart0.readyToParse)
	{
		return;
	}
	
	if(!timersShort[TMR_SHORT_UART0] || uart0.counterRx >= UART_BUFFER_SIZE)
	{
		uart0.counterRx = 0;
		uart0.counterTx = 0;
		uart0.currentId = 0;
		uart0.crc = 0xFFFF;
	}
	uart0.bufferRx[uart0.counterRx++] = tmp;
	crc16ModbusNext(tmp, &uart0.crc);	
	if(uart0.crc == 0)
	{
		uint8_t cmdWago = uart0.bufferRx[1];
		//przetestowanie pod katem ilości odebranych danych w porównaniu to wywolywanego polecenia (WAGO):
		if(cmdWago >= 1 && cmdWago <= 6)
		{
			if(uart0.counterRx == 8)
			{
				timersShort[TMR_SHORT_UART0] = 0;
				uart0.readyToParse = 1;
				return;
			}
		}
		else if(cmdWago == 0x0F || cmdWago == 0x10)
		{
			if(uart0.counterRx == 9 + uart0.bufferRx[6])
			{
				timersShort[TMR_SHORT_UART0] = 0;
				uart0.readyToParse = 1;
				return;
			}
		}
	}
	timersShort[TMR_SHORT_UART0] = UART_WAGO_TIMEOUT;
}
static inline void uart0UdreVectImpl(void)
{	
	if(uart0.currentId < UART_BUFFER_SIZE && uart0.counterTx > uart0.currentId)
		UDR0 = uart0.bufferTx[uart0.currentId++];
	else
		UCSR0B &= ~(1<<UDRIE0);
}
static inline void uart0TxVectImpl(void)
{
	uart0.startReceiving();
}

//uart1:
static inline void uart1RxVectImpl(void)
{
	uint8_t tmp = UDR1;
	if(uart1.counterRx < UART_BUFFER_SIZE)
	{
		uart1.bufferRx[uart1.counterRx++] = tmp;
		setTimerShort(TMR_SHORT_UART1, UART_LCD_TIMEOUT);	
	}
}
static inline void uart1UdreVectImpl(void)
{	
	if(uart1.currentId < UART_BUFFER_SIZE && uart1.counterTx > uart1.currentId)
		UDR1 = uart1.bufferTx[uart1.currentId++];
	else
		UCSR1B &= ~(1<<UDRIE1);
}
static inline void uart1TxVectImpl(void)
{
	uart1.startReceiving();
}

extern _uartData uart0;
extern _uartData uart1;

#endif /* UART_H_ */